import os
import itertools
from typing import List

from nile.api.v1 import files as nfl

from projects.common import wheels

DEFAULT_CLUSTER = 'hahn'

DEFAULT_POOL = 'taxi-delivery'

PYTHON3_YT_SPEC = {
    'layer_paths': [
        '//home/taxi_ml/bin/taximl-py3.7-yt-layer.tar.gz',
        '//porto_layers/ubuntu-xenial-base.tar.xz',
    ],
}


def get_default_nile_requirements() -> List[str]:
    return ['numpy', 'scipy', 'scikit-learn', 'six', 'python-dateutil']


def module_to_develop_package(module: object) -> nfl.DevelopPackage:
    setup_dir = os.path.join(os.path.dirname(module.__file__), os.pardir)
    if os.path.exists(os.path.join(setup_dir, 'setup.py')):
        return nfl.DevelopPackage(os.path.abspath(setup_dir))
    else:
        raise ValueError(f'Invalid module path: {module.__file__}')


def filter_new_files(nfl_files, obj):
    names = {
        nfl_file.filename
        for nfl_file in obj.environment.files
        if hasattr(nfl_file, 'filename')
    }
    return [
        nfl_file
        for nfl_file in nfl_files
        if not hasattr(nfl_file, 'filename') or nfl_file.filename not in names
    ]


def configure_environment(
        obj,
        requirements=None,
        extra_requirements=tuple(),
        add_pyml=True,
        add_projects=True,
        add_cxx_bindings=False,
        nfl_files=None,
        verbose=False,
        parallel_operations_limit=None,
        yt_pool=None,
):
    """
    :param obj: cluster or job
    :param requirements: list of package names which have to be built
    :param extra_requirements: package names in addition to requirements
    :param add_pyml: flag to add taxi_pyml used in virtualenv
    :param add_projects: flag to add projects used ins virtualenv
    :param add_cxx_bindings: flag to add cxx_bindings used in virtualenv
    :param nfl_files: any list of additional files to send in obj
    :param verbose: flag to enable wheels build log
    :param parallel_operations_limit: parallel_operations_limit for nile
    :param yt_pool: pool for YT operation
    :return: obj with configured environment
    """
    if yt_pool is None:
        yt_pool = os.environ.get('YT_POOL')

    yt_pool = DEFAULT_POOL
    print ('yt_pool', yt_pool)
    obj = obj.env(
        yt_spec_defaults={
            'mapper': PYTHON3_YT_SPEC,
            'reducer': PYTHON3_YT_SPEC,
            'reduce_combiner': PYTHON3_YT_SPEC,
            'scheduling_tag_filter': 'porto',
            'pool': yt_pool,
        },
    )

    if requirements is None:
        requirements = get_default_nile_requirements()

    if parallel_operations_limit is not None:
        obj = obj.env(parallel_operations_limit=parallel_operations_limit)

    files = list()

    # if add_pyml:
    #     import taxi_pyml
    #     files.append(module_to_develop_package(taxi_pyml))

    if add_projects:
        import projects
        files.append(module_to_develop_package(projects))

    # if add_cxx_bindings:
    #     import cprojects
    #     import ctaxi_pyml
    #     files.append(nfl.LocalFile(os.path.abspath(cprojects.__file__)))
    #     files.append(nfl.LocalFile(os.path.abspath(ctaxi_pyml.__file__)))

    # hack to fix duplication of packages
    obj = obj.env(package_paths=[wheels.CACHE_DIR])
    files.extend(
        nfl.Package(name)
        for name in wheels.build_deps_for_venv_packages(
            names=set(itertools.chain(requirements, extra_requirements)),
            verbose=verbose,
        )
    )

    if nfl_files:
        files.extend(nfl_files)

    return obj.env(files=files)