"""Functions for dates transform"""
import calendar
import datetime
from typing import AnyStr, Union

import ciso8601
import six
import dateutil.parser
import pytz

UNIX_EPOCH = datetime.datetime.utcfromtimestamp(0)
SECONDS_IN_MINUTE = 60
SECONDS_IN_HOUR = SECONDS_IN_MINUTE * 60
SECONDS_IN_DAY = SECONDS_IN_HOUR * 24
SECONDS_IN_WEEK = SECONDS_IN_DAY * 7


def timestamp(
        stamp: datetime.datetime,
        timezone: str = 'Europe/Moscow',
        milliseconds: bool = False,
) -> int:
    """
    :return: UNIX timestamp
    """
    return calendar.timegm(localize(stamp, timezone=timezone).timetuple()) * (
        1000 if milliseconds else 1
    )


def localize(
        stamp: datetime.datetime = None, timezone: str = 'Europe/Moscow',
) -> datetime.datetime:
    """
    localize timestamp
    """
    if stamp is None:
        stamp = datetime.datetime.utcnow()
    elif not isinstance(stamp, datetime.datetime):  # datetime.date
        stamp = datetime.datetime(stamp.year, stamp.month, stamp.day)

    if stamp.tzinfo is None:
        stamp = stamp.replace(tzinfo=pytz.utc)

    return stamp.astimezone(pytz.timezone(timezone))


def timestring(
        stamp: datetime.datetime = None,
        timezone: str = 'Europe/Moscow',
        time_format: str = '%Y-%m-%dT%H:%M:%S%z',
) -> str:
    """Format timestamp to string with given format
    :param stamp: datetime (if naive, then UTC!)
    """
    return localize(stamp=stamp, timezone=timezone).strftime(time_format)


def parse_timestring_2_aware_dt(
        time_string: str,
        output_timezone: str = None,
        default_str_timezone: str = 'UTC',
) -> datetime.datetime:
    """Parse timestring into aware datetime
    :param time_string: in ISO-8601 format
    :param output_timezone: timezone's name of the answer
    :param default_str_timezone: if time_string contains no timezone,
    this argument is used, default is UTC
    :return: aware datetime with the specified timezone
    """
    result = dateutil.parser.parse(time_string)
    if result.tzinfo is None:
        result = pytz.timezone(default_str_timezone).localize(result)
    if output_timezone is not None and output_timezone != str(result.tzinfo):
        result = result.astimezone(pytz.timezone(output_timezone))
    return result


def parse_timestring(
        time_string: str, timezone: str = 'Europe/Moscow',
) -> datetime.datetime:
    """Parse timestring into naive UTC
    :param time_string: in ISO-8601 format
    :param timezone: if time_string contains no timezone, this argument is used
    :return: naive time in UTC
    """
    return parse_timestring_2_aware_dt(time_string, 'UTC', timezone).replace(
        tzinfo=None,
    )


def datetime_2_timestamp(
        datetime_obj: datetime.datetime, milliseconds: bool = False,
) -> int:
    # this function is 4.5 times faster than dates.timestamp
    if datetime_obj.tzinfo is not None:
        datetime_obj = datetime_obj.astimezone(pytz.utc).replace(tzinfo=None)
    return int(
        (datetime_obj - UNIX_EPOCH).total_seconds()
        * (1000 if milliseconds else 1),
    )


def datetime_2_epoch_minutes(datetime_obj: datetime.datetime) -> int:
    return datetime_2_timestamp(datetime_obj) // SECONDS_IN_MINUTE


def parse_datetime(
        value: str,
        value_len: int = 19,
        format: str = '%Y-%m-%d %H:%M:%S',
        start_pos: int = 0,
) -> datetime.datetime:
    # this implementation is 15 times faster than dateutil.parser.parse
    value = six.ensure_str(value)
    return datetime.datetime.strptime(
        value[start_pos : (start_pos + value_len)], format,
    )


def parse_epoch_minutes(
        value: str,
        value_len: int = 19,
        format: str = '%Y-%m-%d %H:%M:%S',
        start_pos: int = 0,
) -> int:
    return datetime_2_epoch_minutes(
        parse_datetime(value, value_len, format, start_pos),
    )


def create_date_parser(dt_format: str, return_datetime: bool = True):
    def parser(val):
        dt = datetime.datetime.strptime(val, dt_format)
        if return_datetime:
            return dt
        else:
            return dt.strftime(dt_format)

    return parser


def parse_timedelta(value: str):
    """
    Parses duration in following format
    10s - 10 seconds
    2m  - 2 minutes
    3h  - 3 hours
    4d  - 4 days
    5w  - 5 weeks
    :param value: string to parse
    :return: datetime.timedelta
    """
    if len(value) < 2:
        raise ValueError('String is too short')
    int_part = int(value[:-1])
    delta_part = value[-1]
    if delta_part == 's':
        return datetime.timedelta(seconds=int_part)
    elif delta_part == 'm':
        return datetime.timedelta(minutes=int_part)
    elif delta_part == 'h':
        return datetime.timedelta(hours=int_part)
    elif delta_part == 'd':
        return datetime.timedelta(days=int_part)
    elif delta_part == 'w':
        return datetime.timedelta(weeks=int_part)
    else:
        raise ValueError('Unknown type for timedelta {}'.format(delta_part))


DEFAULT_DATE_PARSER = create_date_parser('%Y-%m-%d')


def str_time_diff(begin: str, end: str):
    """
    :param begin: date in format %Y-%m-%d %H:%M:%S
    :param end: date in format %Y-%m-%d %H:%M:%S
    :return: seconds between begin and end
    """
    if begin is None or end is None:
        return None
    return (parse_datetime(end) - parse_datetime(begin)).total_seconds()


def get_date_end(dttm: datetime.datetime) -> datetime.datetime:
    date = datetime.datetime(year=dttm.year, month=dttm.month, day=dttm.day)
    return date + datetime.timedelta(days=1, microseconds=-1)


def ensure_datetime(
        value: Union[AnyStr, datetime.datetime],
) -> datetime.datetime:
    if isinstance(value, datetime.datetime):
        return value
    return ciso8601.parse_datetime(six.ensure_str(value))