
import datetime

from nile.api.v1 import extractors as ne
from nile.api.v1 import filters as nf
from nile.api.v1 import Record
from qb2.api.v1 import filters as qf

from projects.common.decorators import cached_method
from projects.common.nile import filters as pf
from projects.common.nile.dates import range_selector
from projects.common.time_utils import datetime_2_timestamp, parse_timestring
from . import paths


class DataContext:
    def __init__(
            self,
            job,
            begin_dttm: datetime.datetime,
            end_dttm: datetime.datetime,
            sources=None,
    ):
        sources = sources or ['claims', 'special_requirements']

        self._cache = dict()
        self._job = job
        self.begin_dttm = begin_dttm
        self.end_dttm = end_dttm

        self._sources = sources
        self._source_logs_map = {
            'claims': self.get_claims,
            'special_requirements': self.get_special_requirements
        }

        self.time_filter = lambda ts_field: pf.dttm_between(
            begin_dttm, end_dttm, ts_field,
        )

    def get_job(self):
        return self._job


    @cached_method
    def get_sources_logs(self, sources=None):
        sources = sources or self._sources
        sources_logs = [
            self._source_logs_map.get(sources)() for sources in sources
        ]
        return self._job.concat(*sources_logs)

    @cached_method
    def get_claims(self):
        return (
            self._job.table(paths.CLAIMS).project(
                'id', 'corp_client_id', #'created_ts',
                'status', 'taxi_order_id',
                'zone_id', 'dispatch_flow', 'idempotency_token',
                'uuid_id', 'final_price', 'final_pricing_calc_id',
                'is_delayed', 'due', 'eta', 'currency',
                'last_status_change_ts', 'claim_kind',
                # TODO: change last_status_change_ts with tie of
                #  final status from claims_audit
                timestamp=ne.custom(lambda x: int(x), 'created_ts')
                # TODO: make local time
            ).filter(
                pf.ts_between(self.begin_dttm, self.end_dttm, 'timestamp')
            ).project(
                ne.all(),
                source=ne.const('claims')
            )
        )

    @cached_method
    def get_claims_change_log(self):
        return (
            self._job.table(paths.CLAIMS_CHANGE_LOG).project(
                "claim_id", # ex c334b1cd37e34e87b84fd7140594d08a
                "change_type",
                "new_status",
                "updated_ts",
                # "client_id"

                # "new_price"
                # "new_currency"
                # "client_op_id"
                # "revision"
                ch_l_timestamp=ne.custom(lambda x: int(x), 'updated_ts')
            ).filter(
                pf.ts_between(self.begin_dttm, self.end_dttm, 'ch_l_timestamp')
            ).project(
                ne.all(),
                source=ne.const('claims_change_log')
            )
        )

    @cached_method
    def get_claim_audit(self):
        return (
            self._job.table(paths.CLAIM_AUDIT).project(
                "id",
                "old_status",
                "new_status",
                "claim_id",
                "event_time",
                "old_current_point",
                "new_current_point",
                cl_a_timestamp=ne.custom(lambda x: int(x), 'event_time')
            ).filter(
                pf.ts_between(self.begin_dttm, self.end_dttm, 'ch_l_timestamp')
            ).project(
                ne.all(),
                source=ne.const('claim_audit')
            )
        )

    @cached_method
    def get_cargo_orders(self):
        return (
            self._job.table(paths.CARGO_ORDERS).project(
                'cargo_pricing_price', 'waybill_ref', 'provider_order_id',
                co_timestamp=ne.custom(lambda x: int(x), 'created')
            ).filter(
                pf.ts_between(self.begin_dttm, self.end_dttm, 'co_timestamp')
            )#.project(
            #     ne.all(),
            #     source=ne.const('claims')
            # )
        )

    @cached_method
    def get_points(self):
        return (
            # "id"

            self._job.table(paths.POINTS).project(
                "longitude",
                "latitude",
                "fullname",
                # "time_interval_from",
                # "time_interval_to",
                # "created_ts",
                "external_order_id",
                'comment',
                point_id="id",
                p_timestamp=ne.custom(lambda x: int(x), 'created_ts')
            ).filter(
                pf.ts_between(self.begin_dttm, self.end_dttm, 'p_timestamp')
            )
            #     .project(
            #     ne.all(),
            #     source=ne.const('claims')
            # )
        )

    @cached_method
    def get_claim_points(self):
        return (
            self._job.table(paths.CLAIM_POINTS).project(
                "claim_id",
                "point_id",
                "type",
                "visit_order",
                "visit_status",
                'uuid',
                claim_point_id="id",
                cl_p_timestamp=ne.custom(lambda x: int(x), 'updated_ts')
            ).filter(
                pf.ts_between(self.begin_dttm, self.end_dttm, 'cl_p_timestamp')
            )
            # .project(
            #     ne.all(),
            #     source=ne.const('claims')
            # )
        )


    # @cached_method
    # def get_claim_points(self):
    #     return (
    #
    #     )


    @cached_method
    def get_segments(self):
        return (
            # "segment_id"
            # "claim_id"

            # "status"
            # "revision"
            # "created_ts"
            # "resolution"
            self._job.table(paths.SEGMENTS).project(
                "claim_id",
                "segment_id",
                "chosen_waybill",
                s_timestamp=ne.custom(lambda x: int(x), 'created_ts')
            ).filter(
                pf.ts_between(self.begin_dttm, self.end_dttm, 's_timestamp')
            )
            #     .project(
            #     ne.all(),
            #     source=ne.const('claims')
            # )
        )

    @cached_method
    def get_waybills_segments(self):
        return (
            self._job.table(paths.WAYBILLS_SEGMENTS).project(
                "waybill_external_ref",
                "segment_id",
                "waybill_building_version",
                wbs_timestamp=ne.custom( lambda x: int(x), 'updated_ts')
            ).filter(
                pf.ts_between(self.begin_dttm, self.end_dttm, 'wbs_timestamp')
            )
        )

    @cached_method
    def get_waybills(self):
        return (
            # "external_ref"
            # "created_ts"
            # "revision"
            # "status"
            # "resolution"
            # "taxi_order_requirements"
            # "order_id"
            # "order_revision"
            # "taxi_order_id"
            # "special_requirements"
            # "order_expired_ts"
            # "order_last_performer_found_ts"
            self._job.table(paths.WAYBILLS).project(

                'waybill_building_deadline',

                'resolved_at',
                "external_ref",
                # "router_id",
                "created_ts",
                # "updated_ts",
                # "revision",s
                "resolution",
                # "claims_changes_version",
                # "handle_processing_claims_changes_version",
                # "priority",
                # "waybill_building_deadline",
                # "taxi_order_requirements",

                # "order_revision",

                "taxi_order_id",
                # "need_commit",
                # "orders_notified_on_claims_changes_version",
                # "orders_notify_claims_changes_version",
                # "paper_flow",
                # "cancel_state",
                # "special_requirements",
                # "resolved_at",
                # "autoreorder_flow",
                # "taximeter_state_version",
                # "initial_waybill_ref",
                # "previous_waybill_ref",
                # "is_actual_waybill",
                # "proposal_kind",
                # "chain_parent_cargo_order_id",
                # "order_expired_ts",
                # "order_last_performer_found_ts",
                cargo_order_id="order_id",
                wb_status="status",
                wb_timestamp=ne.custom(lambda x: int(x), 'created_ts')
            ).filter(
                pf.ts_between(self.begin_dttm, self.end_dttm, 'wb_timestamp')
            )
            #     .project(
            #     ne.all(),
            #     source=ne.const('claims')
            # )
        )


    @cached_method
    def get_special_requirements(self):
        return (
            self._job.table(paths.SPECIAL_REQUIREMENTS).project(
                'special_requirement_id',
                'operation',
                fake_performer_tag='arguments'
            ).project(
                ne.all(),
                source=ne.const('special_requirements')
            )
        )

    @cached_method
    def get_cargo_pricing_calculations (self):
        return(
            self._job.table(paths.CARGO_PRICING_CALCULATIONS).project(
                'recalc_response',
                final_pricing_calc_id=ne.custom(lambda x: 'cargo-pricing/v1/{}'.format(x), 'id'),
                cpc_timestamp=ne.custom(lambda x: int(x), 'created_ts')
            ).filter(
                pf.ts_between(self.begin_dttm, self.end_dttm, 'cpc_timestamp')
            )
        )

    @cached_method
    def get_orders_performers(self):
        return (
            self._job.table(paths.ORDERS_PERFORMERS).project(
                'transport_type',
                # 'created_ts', 'updated_ts',
                'park_name',
                'dist_from_point_a',
                'order_alias_id',
                cargo_order_id='order_id',
                op_tariff_class='tariff_class',
                dbid_uuid=ne.custom(
                    lambda x, y: '{}_{}'.format(x, y), 'park_id', 'driver_id'
                ),
                op_timestamp=ne.custom(lambda x: int(x), 'created_ts')
            ).filter(
                pf.ts_between(self.begin_dttm, self.end_dttm, 'op_timestamp')
            )
        )