
import datetime

from nile.api.v1 import extractors as ne
from nile.api.v1 import filters as nf
from nile.api.v1 import Record
from qb2.api.v1 import filters as qf

from projects.common.decorators import cached_method
from projects.common.nile import filters as pf
from projects.common.nile.dates import range_selector
from projects.common.time_utils import datetime_2_timestamp, parse_timestring
from . import paths


class DataContext:
    def __init__(
            self,
            job,
            begin_dttm: datetime.datetime,
            end_dttm: datetime.datetime,
            sources=None,
    ):
        sources = sources or ['alena_lukina_cube', 'dima_frolov_cube']

        self._cache = dict()
        self._job = job
        self.begin_dttm = begin_dttm
        self.end_dttm = end_dttm

        self._sources = sources
        self._source_logs_map = {
            'dima_frolov_cube': self.get_dima_frolov_cube,
            'alena_lukina_cube': self.get_alena_lukina_cube
        }

        self.time_filter = lambda ts_field: pf.dttm_between(
            begin_dttm, end_dttm, ts_field,
        )

    def get_job(self):
        return self._job


    @cached_method
    def get_sources_logs(self, sources=None):
        sources = sources or self._sources
        sources_logs = [
            self._source_logs_map.get(sources)() for sources in sources
        ]
        return self._job.concat(*sources_logs)

    @cached_method
    def get_dima_frolov_cube(self):
        return (
            self._job.table(paths.DIMA_FROLOV_CUBE.format(
                range_selector(self.begin_dttm, self.end_dttm, '%Y-%m-01')
            )).project(
                'dbid_uuid', 'dt', 'tags',
                timestamp=ne.custom(lambda x: int(x), 'dt')
            )
            # TODO: filter timestamp on input dates - done?
            .filter(
                pf.ts_between(self.begin_dttm, self.end_dttm, 'timestamp')
            )
        )

    @cached_method
    def get_alena_lukina_cube(self):
        return (
            self._job.table(paths.ALENA_LUKINA_CUBE.format(
                range_selector(self.begin_dttm, self.end_dttm, '%Y-%m-01')
            )).project(
                "auto_courier_tag", "autocourier_sh_share", "dbid_uuid",
                "expeditor_flg", "expeditor_tag", "thermobag_tag",
                "thermobox_option_on_tag", "thermobag_confirmed_tag",
                "utc_dt", "multipoints_tag", "multipoints_available",
                'walking_courier_tag',
                timestamp=ne.custom(
                    lambda utc_dt:
                    datetime_2_timestamp(
                        parse_timestring(utc_dt, 'UTC')
                    )
                ),
                fake_dttm=ne.custom(
                    lambda utc_dt: '{} 23:59:59'.format(utc_dt)
                )

            ).filter(
                nf.or_(
                    # TODO [alarm]: drop this filter?
                    # Russia - autocourier_sh_share >= 0.8,
                    # Abroad - auto_courier_tag >= 0.8
                    # walk courier - walking_courier_tag >= 0.8
                    nf.custom(lambda x: x or 0 >= 0.8, 'autocourier_sh_share'),
                    nf.custom(lambda x: x >= 0.8, 'auto_courier_tag'),
                    nf.custom(lambda x: x >= 0.8, 'walking_courier_tag')
                )
            ).filter(
                # date
                self.time_filter('fake_dttm')
            )
        )