
import datetime

from nile.api.v1 import extractors as ne
from nile.api.v1 import filters as nf
from nile.api.v1 import Record
from qb2.api.v1 import filters as qf

from projects.common.decorators import cached_method
from projects.common.nile import filters as pf
from projects.common.nile.dates import range_selector
from projects.common.time_utils import datetime_2_timestamp, parse_timestring
from . import paths


class DataContext:
    def __init__(
            self,
            job,
            begin_dttm: datetime.datetime,
            end_dttm: datetime.datetime,
            sources=None,
    ):
        sources = sources or ['taxi_dispatch']

        self._cache = dict()
        self._job = job
        self.begin_dttm = begin_dttm
        self.end_dttm = end_dttm

        self._sources = sources
        self._source_logs_map = {
            'taxi_dispatch': self.get_taxi_dispatch
        }

        self.time_filter = lambda ts_field: pf.dttm_between(
            begin_dttm, end_dttm, ts_field,
        )

    def get_job(self):
        return self._job


    @cached_method
    def get_sources_logs(self, sources=None):
        sources = sources or self._sources
        sources_logs = [
            self._source_logs_map.get(sources)() for sources in sources
        ]
        return self._job.concat(*sources_logs)

    @cached_method
    def get_taxi_dispatch(self):
        return (
            self._job.table(
                paths.TAXI_DISPATCH.format(
                    range_selector(self.begin_dttm, self.end_dttm, '%Y-%m-%d')
                ),
                ignore_missing=True
            ).project(
                'candidates_meta',
                'created', # when order is created
                'order_id',
                'winner_applied',
                'winner_id',
                'geo_point',
                'draw_id',
                'trace_id',
                # 'f648f3000f394bd58e6b338a39c121cd'
                timestamp_td=ne.custom(lambda x: int(x), '_logfeller_timestamp')
            ).filter(
                pf.ts_between(self.begin_dttm, self.end_dttm, 'timestamp_td')

            ).filter(
                nf.custom(lambda x: x == True, 'winner_applied')
            ).project(
                ne.all(),
                # source=ne.const('taxi_dispatch')
            )
        )

    @cached_method
    def get_taxi_candidates(self):
        return (
            self._job.table(
                paths.TAXI_CANDIDATES_DAILY.format(
                    range_selector(self.begin_dttm, self.end_dttm, '%Y-%m-%d')
                ),
                ignore_missing=True
            ).project(
                'trace_id',
                #'timestamp', #TODO: `timestamp` -- московское время в формате 2020-11-11T00:00:00.000000
                # 'body',
                'text',
                # timestamp=ne.custom(lambda x: int(x), '_logfeller_timestamp')
                timestamp_tc='timestamp'
            )
            #     .filter(
            #     pf.ts_between(self.begin_dttm, self.end_dttm, 'timestamp')
            # )
            .project(
                ne.all(),
                source=ne.const('taxi_candidates')
            )
        )

    @cached_method
    def get_taxi_driver_scoring(self):
        return (
            self._job.table(
                paths.TAXI_DRIVER_SCORING.format(
                    range_selector(self.begin_dttm, self.end_dttm, '%Y-%m-%d')
                ), ignore_missing=True
            ).project(
                'order_id',
                'dbid_uuid', #TODO: `timestamp` -- московское время в формате 2020-11-11T00:00:00.000000
                # 'body',
                'trace_id',
                'utc_timestamp',
                'is_excluded',
                'min_neg',

                'bonuses',
                'score',
                'zone'
                # timestamp=ne.custom(lambda x: int(x), '_logfeller_timestamp')
            )
            #     .filter(
            #     pf.ts_between(self.begin_dttm, self.end_dttm, 'timestamp')
            # )
            .project(
                ne.all(),
                source=ne.const('taxi_driver_scoring')
            )
        )

    @cached_method
    def get_ld_dispatch(self, to_date, cargo_ref_id_is_defined=True):
        if cargo_ref_id_is_defined:
            # print(
            #     range_selector(self.begin_dttm, self.end_dttm, '%Y-%m-%d').replace(
            #     "2021-07-01,2021-07-02,2021-07-03,2021-07-04,2021-07-05,2021-07-06,2021-07-07,2021-07-08,",""))
            #
            # # 2021-07-01,2021-07-02,2021-07-03,2021-07-04,2021-07-05,2021-07-06,2021-07-07,2021-07-08,
            return (
                # self._job.table(
                #     paths.LD_DISPATCH_DAILY.format(
                #         range_selector(self.begin_dttm, self.end_dttm, '%Y-%m-%d')
                #     ),
                #     ignore_missing=True
                # )

# https://yql.yandex-team.ru/Operations/YTZ-RJfFt7V0YaO_E46sJhvlQVMsYnfgCuesLmphCb4=
                # self._job.table('//home/taxi-delivery/analytics/dev/nkozlovskaya/ml_handler/target/tmp___').filter(
                self._job.table(
                    '//home/taxi-delivery/analytics/production/ld_logs/full/{}'.format(to_date) # '2021-10-01'
                    # '//home/taxi-delivery/analytics/production/ld_logs/full/2021-09-21'
                    # '//home/taxi-delivery/analytics/production/ld_logs/full/2021-09-12'
                    # 'home/taxi-delivery/analytics/dev/nkozlovskaya/ml_handler/target/tmp_v1'
                ).filter(
                    # qf.defined('timestamp'),
                    nf.custom(
                        lambda x: x in
                                  ['planner-assignment', 'planner-segment-edges', 'ServiceApi'],
                        'module'
                    ),
                    qf.defined('cargo_ref_id'), #TODO - change it
                    qf.defined('timestamp'),

                )
                    .project(
                    ne.all(),
                    segment_id='cargo_ref_id',

                    timestamp_ld=ne.custom(
                        lambda x:
                        int(x),
                        # TODO: timestamp as float
                        '_logfeller_timestamp'
                    )
                    # timestamp_ld=ne.custom(
                    #         lambda x:
                    #         (datetime_2_timestamp(
                    #             # datetime.datetime.strptime(x, '%Y-%m-%dT%H:%M:%S.%fZ'),
                    #             datetime.datetime.strptime(x, '%Y-%m-%dT%H:%M:%S.%f%z'),
                    #             milliseconds=True
                    #         ) / 1000)
                    #     if (('-' in x[19:]) or '+' in x[19:])
                    #         else
                    #         (datetime_2_timestamp(
                    #             datetime.datetime.strptime(x, '%Y-%m-%dT%H:%M:%S.%fZ'),
                    #             milliseconds=True
                    #         ) / 1000),
                    #         # TODO: timestamp as float
                    #         'timestamp'
                    #         # 'iso_eventtime'
                    #     )
                ).filter(
                    pf.ts_between(self.begin_dttm, self.end_dttm, 'timestamp_ld'),
                    # nf.custom(lambda x: x > 1629500000, 'timestamp_ld'

                ).project(
                    ne.all(),
                    # source=ne.const('ld_dispatch')
                )
            )
        else:
            return(
                self._job.table(
                    paths.LD_DISPATCH_DAILY.format(
                        range_selector(self.begin_dttm, self.end_dttm,
                                       '%Y-%m-%d')
                    ),
                    ignore_missing=True
                ).filter(
                    # qf.defined('timestamp'),
                    nf.custom(
                        lambda x: x in
                                  ['planner-assignment',
                                   'planner-segment-edges', 'ServiceApi'],
                        'module'
                    ),
                    qf.defined('timestamp')
                ).project(
                    ne.all(),
                    segment_id='cargo_ref_id',
                    timestamp_ld=ne.custom(
                        lambda x:
                        datetime_2_timestamp(
                            datetime.datetime.strptime(x,
                                                       '%Y-%m-%dT%H:%M:%S.%fZ'),
                            milliseconds=True
                        ) / 1000,
                        # TODO: timestamp as float
                        'timestamp'
                        # 'iso_eventtime'
                    )
                ).filter(
                    pf.ts_between(self.begin_dttm, self.end_dttm,
                                  'timestamp_ld'),

                ).project(
                    ne.all(),
                    # source=ne.const('ld_dispatch')
                )
            )


    @cached_method
    def get_ld_route_propositions(self):
        return (
            self._job.table(
                paths.ROUTE_PROPOSITIONS.format(
                    range_selector(self.begin_dttm, self.end_dttm, '%Y-%m-%d')
                ),
                ignore_missing=True
            ).project(
                'history_timestamp',
                'order_id',
                'proposition_id',
                'history_action',
                'contractor_id',
                timestamp_rp=ne.custom(lambda x: int(x), 'history_timestamp')
            ).filter(
                pf.ts_between(self.begin_dttm, self.end_dttm, 'timestamp_rp')
            )
        )

    @cached_method
    def get_taxi_priority(self):
        return (
            self._job.table(
                paths.TAXI_PRIORITY.format(
                    range_selector(self.begin_dttm, self.end_dttm, '%Y-%m-%d')
                ),
                ignore_missing=True
            ).project(
                'tags',
                'dbid_uuid',
                'tariff_zone',
                utc_date=ne.custom(
                    lambda x: datetime.datetime.utcfromtimestamp(int(x)).strftime(
                        '%Y-%m-%d'
                    ), '_logfeller_timestamp'
                ),
                timestamp_tp=ne.custom(lambda x: int(x), '_logfeller_timestamp')
            ).filter(
                pf.ts_between(self.begin_dttm, self.end_dttm, 'timestamp_tp')

            ).project(
                ne.all(),
                # source=ne.const('taxi_dispatch')
            )
        )