
=encoding UTF-8

=head1 Название

QBit::Application::Model::API::Yandex::Balance - API для работы с балансом

=head1 Описание

Реализут XMLRPC интерфейс с балансом.

О реализации интерфейса на стороне балансом можно посмотреть на вики
L</Balance/XmlRpc|https://wiki.yandex-team.ru/Balance/XmlRpc> и
L</Balance/BalanceService|http://wiki.yandex-team.ru/Balance/BalanceService>

Баланс доступен пользователем по адресу http://balance.yandex.ru

В балансе есть следующие сущности:

 * Клиент (client) - отдельная сущность, живущая именно в балансе.
   Обладает уникальным идентификатором client_id. Есть несколько параметров. В
   том числе: "Форма собственности" (может быть "Физическое лицо", "ООО", ...)
 * Представитель клиента (user) - пользователи Яндекса (с uid и логином),
   привязываются к клиенту. Уникальный индетификатор uid. У одного клиента
   может быть несколько представителей. Представитель может быть привязан
   только к одному клиенту (что совершенно логично: если было бы иначе,
   пользователю приходилось бы каждый раз выбирать для какого клиента он хочет
   выполнить действия)
 * Плательщики (person) - тоже отдельная сущность баланса. Как раз этой
   сущности Яндекс платит деньги, либо от которой получает деньги. Платильщики
   привязываются к Клиенту. У клиента может быть несколько плательщиков. Один
   и тот же плательщик может быть для разныех клиентов, но с какими-то
   оговорками. Есть 2 типа плательщиков: обычные (которые платят Яндексу) и
   партнерские (которым платит Яндекс). В ПИ мы работает только с
   плательщиками которым платит Яндекс. Плательщиков у клиентам может быть
   несколько, актуального плательщика можно выяснить из текущего договора. В
   договоре может быть только один плательщик.
 * Площадка - рекламная площадка в ПИ, привязывается к клиенту. На момент
   написания у баланса не было API посмотреть площадки, привязанные к
   клиенту, эта возможно сделать через веб интерфейс.
 * Договор - привязывается к client_id. У одного клиента может быть несколько
   договоров, но только один действующий в настоящий момент времени.
 * Оператор (operator)- пользователь Яндекса, который выполняет какие-то
   действия в балансе (например: 1. создает нового клиента, или 2. изменяет
   контактные данные представителя клиента). Это может быть либо сотрудник
   Яндекса (у которого есть необходимые права), либо сам пользователь.
 * площадка (place)

У баланса есть несколько инстансов (сред):

 * продакшн (боевая) среда - к ней привязан проадкш ПИ
 * тестовая среда - к ней привязаны беты ПИ
 * девелоперская среда - в ПИ не используется

В этом API идут обращения к следующим методам xml rpc баланса:

 * Balance.FindClient - по заданому фильтру показывает некоторые данные
   клиентов, которые соответствуют фильтру
 * Balance.GetPartnerContracts - получение договоров, доп. соглашений,
   информации о плательщике и информации о клиенте. Т.е. получение всей
   информации о клиенте, кроме данных о представителе.

 * Balance.GetClientPersons - получение данных о плательщиках клиента
 * Balance.GetClientUsers - получение данных о представителях клиента

 * Balance.RemoveUserClientAssociation - отвязывает предствителя от клиента
 * Balance.CreateOrUpdatePlace - создание или обновление площадки
 * Balance.CreateOrUpdatePartner - если не указан client_id, то создает
   клиента, представителя и плательщика. Если client_id указан, то обновляет
   данные.

=begin comment Проблема API баланса

К очень большому сожалению, в API баланса нет единого формата ключей
в хешах, которые отдает и принимсет API:

Метод Balance.GetClientPersons возвращает данные в виде (uppercase):

    'BANK_TYPE' => '2',
    'FNAME'     => "Иван",
    'CLIENT_ID' => '1010004',

А метод Balance.GetPartnerContracts в блоке о плательщике возвращает
(canonical):

    'bank_type' => '2',
    'fname'     => "Иван",
    'client_id' => '1010004',

При этом метод Balance.CreateOrUpdatePartner хочет получать данные в виде
(lcminus):

    'bank-type' => '2',
    'fname'     => "Иван",
    'client-id' => '1010004',

Внутри ПИ канонической считается считается perlish версия: 'some_thing'.

Для этих трех форматов ключей придуманы названия: uppercase, canonical,
lcminus и есть метод $self->_convert_hash_keys(), который преобразует
ключи к указанному виду.

=end comment

=head1 Исключения


=head2 Exception::Balance

Базовое исключение.

=cut

package Exception::Balance;
use base qw(Exception);

=head2 Exception::Balance::IncorrectAnswer

Исключение возникает в том случае, баланс вернул ответ, который мы не смогли
распарсить.

=cut

package Exception::Balance::IncorrectAnswer;
use base qw(Exception::Balance);

=head2 Exception::Balance::IncorrectAnswer::KnownErrors

Исключение возникает в том случае, баланс вернул ошибку, но у ошибки есть
несколько известных видов.

    try {
        throw Exception::Balance::IncorrectAnswer::KnownErrors { error_id => 11, error_text => "Error message" };

    } catch {
        my ($exception) = @_;

        p $exception->get_error_id();           # 11
        p $exception->get_error_text();         # "Error message"

        p $exception->message();                # "Error ID: 11 Error Text: Error message"
        p $exception->as_string();              # big text will all details
    };

=cut

package Exception::Balance::IncorrectAnswer::KnownErrors;
use base qw(Exception::Balance::IncorrectAnswer);

sub message {
    my ($self) = shift;

    return "Error ID: " . $self->get_error_id . " Error Text: " . $self->get_error_text();
}

sub as_string {
    my ($self) = shift;

    local $self->{'text'} = $self->message;
    $self->SUPER::as_string(@_);
}

sub get_error_id {
    my ($self) = @_;

    return $self->{'text'}->{'error_id'};
}

sub get_error_text {
    my ($self) = @_;

    return $self->{'text'}->{'error_text'};
}

=head2 Exception::Balance::SeveralResults

Исключение возникает в том случае, если в результате есть несколько элементов

=cut

package Exception::Balance::SeveralResults;
use base qw(Exception::Balance);

=begin comment Начало модуля

=end comment

=cut

package QBit::Application::Model::API::Yandex::Balance;

use qbit;
use base qw(QBit::Application::Model::API::XMLRPC);

use XML::Twig;
use XML::Parser;
use Utils::TSV;

=head1 Методы

=head2 init

B<Параметры:> -

B<Возвращаемое значение:> -

Инициализация

TODO - добится от баланса принятия серелизованных строк не только типа string
и убрать хак.

=cut

sub init {
    my ($self) = @_;

    $self->SUPER::init();

    # Баланс не умеет воспринимать только "<value><string>...</", поэтому все
    # типы объектов мы серилизуем в string
    # Но из-за этого возникает warning:
    # Content-Length header value was wrong, fixed at /usr/share/perl5/LWP/Protocol/http.pm line 190.
    $self->{__RPC__}->{_serializer}->{_typelookup} = {string => [2, sub {1}, 'as_string']};

}

=head2 call

B<Параметры:> 1) $self, 2) $func, 3) @opts

B<Возвращаемое значение:> 1) $result - ссылка на хеш с данными

Низкоуровневое обращение к XMLRPC.

В том случае если получиили от билинга ошибку пытаемся воспринять эту ошибку
как xml и отформатировать.

=cut

sub call {
    my ($self, $func, @opts) = @_;

    # Для того чтобы эту штуку можно было тестировать
    return $self->{debug_result} if defined $self->{debug_result};

    my $return;

    try {
        $return = $self->SUPER::call($func, @opts);
    }
    catch Exception::API::XMLRPC with {

        # Пытаюсь отформатировать xml
        my $error = $@->{text};

        my $xml = XML::Twig->new(pretty_print => 'indented');
        eval {$xml->parse($error);};

        if (!$@) {
            open my $fh, ">", \$error;
            $xml->print($fh);
        }
        $error = "\n$error";

        throw Exception::Balance::IncorrectAnswer $error;
    };

    return $return;
}

=head2 find_client

B<Параметры:> 1) % хеш с запросом на поиск (описание ключей в доке
http://wiki.yandex-team.ru/Balance/XmlRpc#balance.findclient)

B<Возвращаемое значение:> 1) ссылка на массив со ссылками на хеш с ответом
баланса

Метод обращается к методу баланса Balance.FindClient. В случае каких-либо
ошибок возвращает исключение Exception::Balance::IncorrectAnswer. Возможна
ситуация что найден более чем один клиент - в этом случае массив будет
состоять более чем из одного элемента. В том случае если по запросу не найден
ни один клиент будет возвращена ссылка на пустой массив. Формат ключей -
canonical.

То данные которые выдает этот метод отличаются от данных в блоке Client метода
Balance.GetPartnerContracts.

    p $app->api_balance->find_client("Login" => "ivanbessarabov");
    {
        agency_id        0,
        city             "",
        client_id        1304746,
        client_type_id   0,
        email            "ivan@bessarabov.ru",
        fax              "",
        is_agency        0,
        name             "Бессарабов Иван",
        phone            "+7 (915) 480 2997",
        url              ""
    }

=cut

sub find_client {
    my ($self, %opts) = @_;

    my $data = $self->call('Balance.FindClient', \%opts,);

    my $result = [];

    throw Exception::Balance::IncorrectAnswer gettext('Could not interpret output from balance')
      if ref $data ne 'ARRAY'
          or scalar $data->[0] != 0;

    foreach my $el (@{$data->[2]}) {
        push @{$result}, $self->_convert_hash_keys($el);
    }

    return $result;
}

=head2 get_client_id_by_uid

B<Параметры:> 1) $self 2) $uid

B<Возвращаемое значение:> 1) $client_id если указаный $uid является
представительем какоего-то клиента, иначе undef.

=cut

sub get_client_id_by_uid {
    my ($self, $uid) = @_;

    my $balance_answer = $self->find_client("PassportID" => $uid);

    if (ref $balance_answer eq 'ARRAY') {
        my $n = @{$balance_answer};

        if ($n == 0) {
            return undef;
        } elsif ($n == 1) {

            if ($balance_answer->[0]->{client_id}) {
                return $balance_answer->[0]->{client_id};
            } else {
                return undef;
            }

        } else {
            throw Exception::Balance::IncorrectAnswer
"For uid '$uid' got incorrect number of Balance.FindClient results: '$n'. It can containg only one result.";
        }

    } else {
        return undef;
    }
}

=head2 get_partner_contracts

B<Параметры:> 1) % хеш с запросом на поиск:

=over

=item B<ClientID> - число, ID клиента в балансе.

=item B<ExternalID> - строка, номер договора.

=back

B<Возвращаемое значение:> 1) ссылка на массив хешей с ответом баланса

Метод обращается к методу баланса Balance.GetPartnerContracts. В случае
каких-либо ошибок выбрасывает исключение.

В данных плательщика удаляются ненужные поля с помощью метода
_delete_unused_person_data

Уведомления не подписываются.

    p $app->api_balance->get_partner_contracts("ClientID" => $client_id);
    {
        Client        { ... }, # клиент
        Collaterals   [ ... ], # дополнительные соглашения
        Contract      { ... }, # договор
        Person        { ... }, # плательщик
    }

=cut

sub get_partner_contracts {
    my ($self, %opts) = @_;

    my $data = $self->call('Balance.GetPartnerContracts', \%opts);

    $data = $data->[0];

    $self->_delete_unused_person_data($_->{Person}) foreach @$data;

    return $data;
}

=head2 get_client_persons

На входе $client_id и $type.

На выходе ARRAYREF с хешами. Если нет плательщиков, то возвращается ARRAYREF
без элементов.

Используется метод баланса Balance.GetClientPersons. Возвращает из баланса
информацию о плательщиках, которые привязаны к указанному client_id.
Следует использовать этот метод, только когда у пользователя нет договора,
так как в договоре есть информация о текущем плательщике.

В случае если не найдены данные, то будет возвращена ссылка на пустой хеш.

Данные которые мы получаем через метод Balance.GetClientPersons несколько
отличаются от того чтоб мы получаем через метод Balance.GetPartnerContracts:

 * GetClientPersons поле dt в формате '2010-12-08 12:44:05', а в
   GetPartnerContracts поел dt в формате '2010-12-08'
 * У GetClientPersons есть значение у поля 'attributes', а у
   GetPartnerContracts нет (но значение никакой ценности не представляет).
 * У GetClientPersons есть значение у поля 'exports', а у
   GetPartnerContracts нет (но значение никакой ценности не представляет).

http://wiki.yandex-team.ru/Balance/XmlRpc#balance.getclientpersons

    # Число-констаната, означающая тип плательщика, которые нужно возвращать
    # 1 - тип плательщика 'partner' - Яндекс платит таким плательщикам
    # 0 - тип плательщика 'обычный' - такие плательщики платит Яндексу
    my $type = 1;

    p $app->api_balance->get_client_persons($client_id, $type);
    {
        account                  40817810004210005493,
        address_postcode         123056,
        authority_doc_details    "",
        ...
    }

=cut

sub get_client_persons {
    my ($self, $client_id, $type) = @_;

    if (!defined($type) || !in_array($type, [0, 1])) {
        throw Exception::BadArguments gettext("Incorrect type");
    }

    my $data = $self->call('Balance.GetClientPersons', $client_id, $type);

    my @persons;

    foreach (@{$data->[0]}) {
        my $fixed_data = $self->_convert_hash_keys($_);
        $self->_delete_unused_person_data($fixed_data);
        push @persons, $fixed_data;
    }

    return \@persons;
}

=head2 get_client_users

Используется метод баланса Balance.GetClientUsers. Возвращает из баланса
ссылку на массив с хешами информации о представителях клиента, которые
привязаны к указанному client_id.

В том случае, если у указанного Client ID нет представителей, то будет
возвращена ссылка на пустой массив.

В том случае, если в балансе нет клиента с указанным Client ID, то так же
будет возвращена ссылка на пустой массив.

Про метод GetClientUsers есть информация на странице
http://wiki.yandex-team.ru/Balance/BalanceService

    p $app->api_balance->get_client_users($client_id);
    [
        [0] {
            gecos         "Бессарабов Иван",
            login         "ivanbessarabov",
            passport_id   35309619
        }
    ]

У одного клиента может быть больше чем один представитель. Например, у
client_id = 943972 есть 2 представителя.

=cut

sub get_client_users {
    my ($self, $client_id) = @_;

    my $data = $self->call('Balance.GetClientUsers', $client_id);

    my $result = [];

    throw Exception::Balance::IncorrectAnswer gettext('Could not interpret output from balance')
      if ref $data ne 'ARRAY';

    foreach my $el (@{$data->[0]}) {
        push @{$result}, $self->_convert_hash_keys($el);
    }

    return $result;
}

=head2 remove_user_client_association

B<Параметры:> 1) $self 2) $uid оператора, который проводит изменение
3) $client_id 4) $uid представителя, которого нужно отвязать от клиента

B<Возвращаемое значение:> 1) true если представитель успешно отвязан

Доступ к методу Balance.RemoveUserClientAssociation - отвязывает предствителя
от клиента

http://wiki.yandex-team.ru/Balance/XmlRpc#balance.removeuserclientassociation

Метод выкидывает исключение, если не удалось отвязать представителя.

Метод умирает, если запустить его не на тестовой версиии баланса.

=cut

sub remove_user_client_association {
    my ($self, $operator_uid, $client_id, $user_uid) = @_;

    if ($self->get_option('url') ne 'http://xmlrpc.balance.greed-ts1f.yandex.ru:8002/xmlrpc') {
        throw gettext("Method RemoveUserClientAssociation should be run only with test balance instance");
    }

    my $data = $self->call('Balance.RemoveUserClientAssociation', $operator_uid, $client_id, $user_uid,);

    throw Exception::Balance::IncorrectAnswer gettext('Could not interpret output from balance')
      if ref $data ne 'ARRAY';
    throw Exception::Balance::IncorrectAnswer join(' ', @{$data}) if scalar $data->[0] != 0;

    return TRUE;
}

=head2 create_or_update_place

B<Параметры:> 1) $self 2) %opts

B<Возвращаемое значение:> 1) TRUE если удалось создать обновить площадку,
иначе будет выброшено исключение.

Доступ к методу Balance.CreateOrUpdatePlace - создание или обновление площадки
в балансе.

http://wiki.yandex-team.ru/Balance/XmlRpc#balance.createorupdateplace

Значения %opts:

=over

=item B<operator_uid> - число, UID пользователя, который создаёт/изменяет
площадку. (обязательный параметр)

=item B<client_id> - число, ID пользователя в балансе - владельца площадки.
Можно менять для ряда клиентов с client_id = (412775, 440062, 469655, 501212)
BALANCE-9853

=item B<page_id> - число, ID площадки в БК. (обязательный параметр)

=item B<domain> - строка, домен площадки.

=item B<campaign_type> - строка, тип площадки. (обязательный параметр)

=item B<viptype> - строка, внутренний тип площадки.

=item B<search_id> - число, ID площадки в ППС

=item B<is_payable> - нужно ли считать и перечислять деньги по влощадке

=item B<pay_type_id> - тип оплаты (для дистрибуции)

=item B<clid_type_id> - тип клида (для дистрибуции)

=back

=cut

sub create_or_update_place {
    my ($self, %opts) = @_;

    my $user2balance = {
        client_id       => 'ClientID',
        page_id         => 'ID',
        domain          => 'URL',
        campaign_type   => 'Type',
        viptype         => 'InternalType',
        search_id       => 'SearchID',
        is_payable      => 'IsPayable',
        pay_type_id     => 'PaymentTypeID',
        clid_type_id    => 'ClidType',
        contract_tag_id => 'TagID',
    };

    my $data = {hash_transform(\%opts, [], $user2balance)};

    my $result = $self->call('Balance.CreateOrUpdatePlace', $opts{operator_uid}, $data);

    throw Exception::Balance::IncorrectAnswer gettext('Could not interpret output from balance')
      if ref $result ne 'ARRAY';
    throw Exception::Balance::IncorrectAnswer join(' ', @{$result}) if scalar $result->[0] != 0;

    return TRUE;
}

=head2 create_or_update_partner

B<Параметры:> 1) $self 2) %opts

B<Возвращаемое значение:> 1) $client_id - это client_id
созданного/измененного клиента (возвращаетс только в случае успеха, в случае
каких-то проблем выбрасывается исключение)

Взаимодействие с методом Balance.CreateOrUpdatePartner - создание клиента,
представителя и плательщика или изменение данных.

Значения %opts:

=over

=item B<operator_uid> - число, UID пользователя, который создаёт/изменяет
площадку. (обязательный параметр)

=item B<mode> - режим отправки 'form' - это создание клиента, 'edit' - это
редактирование клиента.

=item B<...> - куча других параметров, которые вводит пользователь при
заполнении анкеты партнера

В случае ошибки метод Balance.CreateOrUpdatePartner вернет xml с описанием
ошибки.

=back

=cut

sub create_or_update_partner {
    my ($self, %opts) = @_;

    my $operator_uid = delete $opts{operator_uid};

    my $result = $self->call('Balance.CreateOrUpdatePartner', $operator_uid, \%opts,);

    throw Exception::Balance::IncorrectAnswer $result if ref $result ne 'ARRAY';
    throw Exception::Balance::IncorrectAnswer $result if !$result->[0];

    my $client_id = $result->[0];

    return $client_id;
}

=head2 create_or_update_contract_tag

B<Параметры:> 1) $self 2) %opts

B<Возвращаемое значение:> 1) TRUE

Доступ к методу Balance.CreateClient - создание или обновление коммерческих условий

https://wiki.yandex-team.ru/balance/xmlrpc#balance.createorupdatedistributiontag

Значения %opts:

=over

=item B<operator_uid> - число, UID пользователя, который создаёт/изменяет
площадку. (обязательный параметр)

=item B<id> - число, изменяемый/создаваемый идентификатор коммерческих условий.
(Обязательный параметр)

=item B<caption> - строка, изменяемое/создаваемое название коммерческих условий.
(Обязательный параметр)

=back

=cut

sub create_or_update_contract_tag {
    my ($self, %opts) = @_;

    my $contract_tag_2_balance = {
        id        => 'TagID',
        caption   => 'TagName',
        client_id => 'ClientID',
    };

    my $data = {hash_transform(\%opts, [], $contract_tag_2_balance)};

    my $result = $self->call('Balance.CreateOrUpdateDistributionTag', $opts{operator_uid}, $data);

    throw Exception::Balance::IncorrectAnswer Dumper($result)
      if (ref($result) ne 'ARRAY') || (ref($result->[0]) ne 'ARRAY');

    if (($result->[0]->[0] ne '0') or ($result->[0]->[1] ne 'SUCCESS')) {
        throw Exception::Balance::IncorrectAnswer::KnownErrors {
            error_id   => $result->[0]->[0],
            error_text => $result->[0]->[1],
        };
    }

    return TRUE;
}

=head2 create_client

B<Параметры:> 1) $self 2) %opts

B<Возвращаемое значение:> 1) $client_id

Доступ к методу Balance.CreateClient - создание или обновление клиента

http://wiki.yandex-team.ru/Balance/XmlRpc#balance.createclient

Значения %opts:

=over

=item B<operator_uid> - число, UID пользователя, который создаёт/изменяет
площадку. (обязательный параметр)

=back

=cut

sub create_client {
    my ($self, %opts) = @_;

    throw gettext("Expected 'operator_uid'") unless defined $opts{operator_uid};

    my $operator_uid = delete $opts{operator_uid};

    my $result = $self->call('Balance.CreateClient', $operator_uid, \%opts,);

    throw Exception::Balance::IncorrectAnswer Dumper($result) if ref $result  ne 'ARRAY';
    throw Exception::Balance::IncorrectAnswer Dumper($result) if $result->[0] ne '0';
    throw Exception::Balance::IncorrectAnswer Dumper($result) if $result->[1] ne 'SUCCESS';

    my $client_id = $result->[2];
    return $client_id;
}

=head2 create_user_client_association

B<Параметры:> 1) $self 2) %opts

B<Возвращаемое значение:> -

Доступ к методу Balance.CreateUserClientAssociation - привязывание
представителя к клиенту.

Метод ничего не возвращает. В случае успеха метод молча отработает. В случае
какой-либо ошибки метод выбросит исключение.

Про исключения. Если случилось что-то совсем старшное и нерешаемое, то метод
выбрасывает исключение Exception::Balance::IncorrectAnswer. Если же произошла
ошибка про которую есть дополнительная информация, то будет прошено
искключение Exception::Balance::IncorrectAnswer::KnownErrors в котором есть 2
метода, которые позволяют получить дополнительную информацию об ошибке:
$exception->get_error_id() и $exception->get_error_text()

http://wiki.yandex-team.ru/Balance/XmlRpc#balance.createuserclientassociation

Значения %opts:

=over

=item B<operator_uid> - число, UID пользователя, который создает привязку
(обязательный параметр)

=item B<client_id> - число, Client ID к которому нужно привязать представителя
(обязательный параметр)

=item B<user_id> - число, uid пользователя, который будет являтеся
представителем. (обязательный параметр)

=back

Пример работы. В случае какой-то ошибки просто падаем с исключением:

    $app->api_balance->create_user_client_association(
        operator_uid => $operator_uid,
        client_id => $client_id,
        user_id => $user_id,
    );

Пример работы, где отрабатывется исключением:

    try {
        $app->api_balance->create_user_client_association(
            operator_uid => $operator_uid,
            client_id => $client_id,
            user_id => $user_id,
        );
    } catch Exception::Balance::IncorrectAnswer::KnownErrors with {
        my ($exception) = @_;
        p $exception->get_error_id();
        p $exception->get_error_text();
    };

=cut

sub create_user_client_association {
    my ($self, %opts) = @_;

    throw gettext("Expected 'operator_uid'") unless defined $opts{'operator_uid'};
    throw gettext("Expected 'client_id'")    unless defined $opts{'client_id'};
    throw gettext("Expected 'user_id'")      unless defined $opts{'user_id'};

    $self->_call_std('Balance2.CreateUserClientAssociation',
        $opts{'operator_uid'}, $opts{'client_id'}, $opts{'user_id'},);

    return TRUE;
}

=head2 create_person

Доступ к методу Balance.CreatePerson - создание или изменение плательщика.

B<Параметры:> 1) $self 2) %opts - хеш с данными плательщика которого нужно
создать. В обятательном порядке нужно передавать:

=over

=item B<client_id>

=item B<operator_uid> - uid пользователя кто проводит изменение (может быть
как пользользователь-партнер, так и системаный пользователь)

=item B<person_id> - id плательщкика для редактирования. В том случае если
передается -1, то плательщик будет создан (warning: баланс в xmlrpc протоколе
хочет видеть этот -1 как число, поэтому нужно передавать
`SOAP::Data->type(int => -1)`)

=back

B<Возвращаемое значение:> $person_id

    my $person_id = $self->api_balance->create_person(
        operator_uid => $user->{'id'},
        client_id => $user->{'client_id'},
        person_id => SOAP::Data->type(int => -1),  # пример как передавать -1 как число
        type => 'ur',
        %values4balance,
    );

=cut

sub create_person {
    my ($self, %opts) = @_;

    my $operator_uid = delete($opts{'operator_uid'});
    throw gettext("Expected 'operator_uid'") unless defined $operator_uid;
    throw gettext("Expected 'client_id'") unless defined $opts{'client_id'};

    my $result = $self->call('Balance2.CreatePerson', $operator_uid, \%opts);

    throw Exception::Balance::IncorrectAnswer Dumper($result) if ref $result ne 'ARRAY';
    throw Exception::Balance::IncorrectAnswer Dumper($result) if scalar @$result != 1;

    my $person_id = $result->[0];

    return $person_id;
}

=head2 get_currency_rate

Access to method Balance.GetCurrencyRate - currency_rate getting.

B<Options:> 1) $self 2) %opts - hash with date and currency code which rate
is required.

=over

=item B<currency> - Currency code, e. g. USD, EUR.

=item B<date> - (Optional) Currency rate date. 'yyyy-mm-dd'

=back

B<Returning value:> $rate

    my $rate = $self->api_balance->get_currency_rate(
        currency => 'USD',
        date     => '2014-01-20'
    );

=cut

sub get_currency_rate {
    my ($self, %opts) = @_;

    my $currency = delete($opts{'currency'});
    my $date     = delete($opts{'date'});

    throw gettext("Missed required field '%s'", 'currency') unless defined $currency;
    throw gettext("Incorrect %s = %s", 'date', $date) if defined $date && !trdate(db => sec => $date);

    return $self->_call_std('Balance2.GetCurrencyRate', $currency, $date)->{'rate'};
}

=head2 query_catalog

B<Параметры:> 1) $self 2) $tables 3) $where 4) {} (опц.) - указание как привести поля

B<Возвращаемое значение:> 1) $data - структура с данными

http://wiki.yandex-team.ru/Balance/XmlRpc#balance.querycatalog

Например:

    $proj->billing->query_catalog(["T_BRAND"], '');

Это вернет структуру вида:

    $VAR1 = {
              'columns' => [
                           't_brand.bid',
                           't_brand.name',
                           't_brand.ename',
                           't_brand.note',
                           't_brand.status',
                           't_brand.file_id',
                           't_brand.hidden'
                         ],
              'result' => [
                          [
                            '706769.0000000000',
                            "ЭЛЕКТРОННЫЙ СПРАВОЧНИК КОНСТРУКТОРА",
                            'ELEKTRONNY SPRAVOCHNIK KONSTRUKTORA',
                            undef,
                            'A',
                            '2.0000000000',
                            undef
                          ],
                          ...
                          ]

            }

В случае если указан 4 параметр - {t_brand.name => 'name', t_brand.file_id => 'id'}, Вернет структуру вида:

    $VAR1 = [
        { id => '2.0000000000',  name => "ЭЛЕКТРОННЫЙ СПРАВОЧНИК КОНСТРУКТОРА" },
        ...
    ]


=cut

sub query_catalog {
    my ($self, $tables, $where, $map) = @_;

    my $result = $self->call('Balance.QueryCatalog', $tables, $where);

    if (ref $result eq 'ARRAY') {
        $result = $result->[0];
    } else {
        throw Exception::Balance::IncorrectAnswer gettext('Could not interpret output from balance');
    }

    if ($map) {
        my ($list_columns, $list_data) = @{$result}{'columns', 'result'};
        # get fields names
        my @columns_out = map($map->{$_} || '', @$list_columns);
        # convert array into hashes with new f. names
        my $list_data_out = [
            map({
                    my $a = {};
                      @$a{@columns_out} = @$_;
                      delete($a->{''});
                      $a;
                } @$list_data)
        ];
        return $list_data_out;
    } else {
        return $result;
    }
}

# Balance.GetDistributionRevenueShare
# Возвращает структуру с данными
sub get_parsed_distribution_revenue_share {
    my ($self, %params) = @_;

    return $self->_call_method_from_to(
        method => 'Balance.GetDistributionRevenueShare',
        from   => $params{from},
        to     => $params{to},
    );
}

# Balance.GetDistributionRevenueShareFull
# Возвращает структуру с данными
sub get_parsed_distribution_revenue_share_full {
    my ($self, %params) = @_;

    return $self->_call_method_from_to(
        method => 'Balance.GetDistributionRevenueShareFull',
        from   => $params{from},
        to     => $params{to},
    );
}

# Balance.GetDistributionFixed
# Возвращает структуру с данными
sub get_parsed_distribution_fixed {
    my ($self, %params) = @_;

    return $self->_call_method_from_to(
        method => 'Balance.GetDistributionFixed',
        from   => $params{from},
        to     => $params{to},
    );
}

# Balance.GetDistributionDownloads
# Возвращает структуру с данными
sub get_parsed_distribution_downloads {
    my ($self, %params) = @_;

    throw gettext("Don't use Balance.GetDistributionDownloads, use Balance.GetDistributionFixed.");

    return $self->_call_method_from_to(
        method => 'Balance.GetDistributionDownloads',
        from   => $params{from},
        to     => $params{to},
    );
}

# Balance.GetDistributionSerpHits
# Возвращает структуру с данными
sub get_parsed_distribution_serp_hits {
    my ($self, %params) = @_;

    return $self->_call_method_from_to(
        method => 'Balance.GetDistributionSerpHits',
        from   => $params{from},
        to     => $params{to},
    );
}

sub link_dsp_to_client {
    my ($self, %opts) = @_;

    throw Exception::BadArguments("Expected 'user_id'") unless defined $opts{user_id};

    my $user_id = delete $opts{user_id};

    my $result = $self->call('Balance2.LinkDspToClient', $user_id, \%opts)->[0];

    throw Exception::Balance::IncorrectAnswer gettext('Could not interpret output from balance')
      if ref $result ne 'ARRAY';
    throw Exception::Balance::IncorrectAnswer join(' ', @{$result}) if scalar $result->[0] != 0;

    return TRUE;
}

# Balance2.GetDspStat
# Возвращает структуру с данными
sub get_dsp_stat {
    my ($self, %params) = @_;

    return $self->_call_method_from_to(
        method => 'Balance2.GetDspStat',
        from   => $params{from},
        to     => $params{to},
    );
}

sub get_dsp_money_completition {
    my ($self, %params) = @_;

    return $self->_call_method_from_to(
        method => 'Balance2.GetDspMoneyCompletion',
        from   => $params{from},
        to     => $params{to},
    );
}

sub get_dsp_money_completition_with_page_id {
    my ($self, %params) = @_;

    return $self->_call_method_from_to(
        method => 'Balance2.GetDspMoneyCompletionWithPageId',
        from   => $params{from},
        to     => $params{to},
    );
}

sub get_pages_stat {
    my ($self, %params) = @_;

    return $self->_call_method_from_to(
        method => 'Balance.GetPagesStat',
        from   => $params{from},
        to     => $params{to},
    );
}

sub get_pages_tags_stat {
    my ($self, %params) = @_;

    return $self->_call_method_from_to(
        method => 'Balance2.GetPagesTagsStat',
        from   => $params{from},
        to     => $params{to},
    );
}

sub get_internal_pages_stat {
    my ($self, %params) = @_;

    return $self->_call_method_from_to(
        method => 'Balance2.GetInternalPagesStat',
        from   => $params{from},
        to     => $params{to},
    );
}

sub get_internal_pages_tags_stat {
    my ($self, %params) = @_;

    return $self->_call_method_from_to(
        method => 'Balance2.GetInternalPagesTagsStat',
        from   => $params{from},
        to     => $params{to},
    );
}

=begin comment _convert_hash_keys

Что это такое - читай в pod в разделе 'Описание'

=end comment

=cut

sub _convert_hash_keys {
    my ($self, $hash, $format) = @_;

    $format = 'canonical' if !defined($format);
    throw gettext("Incorrect format '%s'", $format) if !in_array($format, [qw(uppercase canonical lcminus)]);

    my $fixed_hash;

    if ($format eq 'uppercase') {

        $fixed_hash = {map {my $new_key = $_; $new_key =~ s/-/_/g; uc $new_key => $hash->{$_}} keys %$hash};

    } elsif ($format eq 'canonical') {

        $fixed_hash = {map {my $new_key = $_; $new_key =~ s/-/_/g; lc $new_key => $hash->{$_}} keys %$hash};

    } elsif ($format eq 'lcminus') {

        $fixed_hash = {map {my $new_key = $_; $new_key =~ s/_/-/g; lc $new_key => $hash->{$_}} keys %$hash};

    }

    return $fixed_hash;
}

=begin comment _delete_unused_person_data

B<Параметры:> 1) $self, 2) $person - ссылка на хеш с данными плательщика

B<Возвращаемое значение:> -

Метод предназанчеен для удаления некоторых полей из данных платильщика,
которые мы получаем в методах GetPartnerContracts и GetClientPersons.

Метод изменяет переданный параметр $person

Цель этого метода - это подготовить ПИ к ситуации, когда баланс отрежет
эти поля. Т.е. заранее отрезаем поля, убеждаемся что все работает, а потом
и баланс у себя убирает эти поля.

В этом методе есть массив полей, которые мы удаляем, по хорошему он должен
быть пуст - что означает что баланс у себя уже все удалил.

=end comment

=cut

sub _delete_unused_person_data {
    my ($self, $person) = @_;

    # В некоторых случаях Баланс в качетсве данных плательщика может вернуть
    # '0'. В этом случае не нужно править данные.
    return if ref($person) ne 'HASH';

    my @what_to_delete_from_person_data = qw(
      bank
      bankcity
      bank_data
      bank_inn
      corraccount

      legal_address_region
      legal_address_building
      legal_address_construction
      legal_address_code
      legal_address_district
      legal_address_home
      legal_address_town
      legal_address_flat
      legal_address_street
      legal_address_gni
      legal_address_city

      address_town
      address_flat
      address_street
      address_construction
      address_district
      address_code
      address_home
      address_building
      address_region
      address_city
      address_gni
      );

    foreach my $k (keys %$person) {
        foreach my $el (@what_to_delete_from_person_data) {
            delete $person->{$k} if $k =~ /^$el$/i;
        }
    }
}

=begin comment _call_method_from_to

=end comment

=cut

sub _call_method_from_to {
    my ($self, %params) = @_;

    throw gettext("Need 'method'") unless defined $params{method};
    throw gettext("Need 'from'")   unless defined $params{from};
    throw gettext("Need 'to'")     unless defined $params{to};

    my $tsv = $self->call($params{method}, $params{from}, $params{to},)->[0];

    if ($params{_get_tsv}) {
        return $tsv;
    } else {
        my $stat = parse_tsv($tsv);
        return $stat;
    }
}

=begin comment _call_std

=end comment

=cut

sub _call_std {
    my ($self, $method, @opts) = @_;

    my $result = $self->call($method, @opts);

    throw Exception::Balance::IncorrectAnswer Dumper($result)
      if (ref($result) ne 'ARRAY') || (ref($result->[0]) ne 'ARRAY');

    if (($result->[0]->[0] ne '0') or ($result->[0]->[1] ne 'SUCCESS')) {
        throw Exception::Balance::IncorrectAnswer::KnownErrors {
            error_id   => $result->[0][0],
            error_text => $result->[0][1],
        };
    }

    return $result->[0][2];
}

TRUE;
