
=head1 Name

QBit::Application::Model::API::Yandex::BlackBox

=head2 Настройки

 * Debug - $, boolean
 * URL - $, string. Урл интерфейса блэкбокса
 * MayOnlySocial - $, boolean. Возможна только социальная авторизация (без логина на Я).

=head2 Исключения

 * ...
 * Exception::BlackBox::NeedResign (url => $pass)- пора(нужно) обновить сессионную куку через PASS
 * Exception::BlackBox::NeedAuth (url => $passport) - нужно авторизоваться на пасспорте
 * Exception::BlackBox::NeedRealLogin (url => $passport) - только соцаальная авторизация. Нет логина на Я. Нужно привязать социальный аккаунт к профилю на Я на пасспорте

=cut

package Exception::BlackBox;
use base qw(Exception);

package Exception::BlackBox::Net;
use base qw(Exception::BlackBox);

package Exception::BlackBox::InternalError;
use base qw(Exception::BlackBox);

package Exception::BlackBox::NeedResign;
use base qw(Exception::BlackBox);

package Exception::BlackBox::NeedAuth;
use base qw(Exception::BlackBox);

package Exception::BlackBox::NeedRealLogin;
use base qw(Exception::BlackBox);

package QBit::Application::Model::API::Yandex::BlackBox;

use qbit;

use base qw(QBit::Application::Model);

use LWP::UserAgent;
use URI::Escape qw(uri_escape_utf8);
use HTTP::Request;
use XML::Simple;

__PACKAGE__->mk_ro_accessors(qw(ua));

our $DEBUG;

sub init {
    my ($self) = @_;

    $self->SUPER::init();

    throw gettext('Need URL of BlackBox in config') unless $self->get_option('URL');

    $self->{'ua'} = LWP::UserAgent->new(agent => "Yandex::Blackbox/perl/1.00",);

    $DEBUG = $self->get_option('Debug');
}

=head2 sessionid

Получаем информацию о сессии из ящика. В случае ошибки 2 типа исключения:

Exception::BlackBox::NeedAuth - не авторизован (совсем или протухло)

Exception::BlackBox::NeedResign - требуется продление (по времени или запрос главного домена)

    my $bb_res = $self->api_blackbox->sessionid(
        session_id      => $self->request->cookie('Session_id'),
        sessionid2      => $self->request->cookie('sessionid2'),        # optional
        remote_addr     => $self->request->remote_addr(),
        server_name     => $self->request->server_name(),
        fields          => [qw(
            accounts.login.uid
            account_info.fio.uid
            subscription.suid.85
        )],
    );

Документация
 http://wiki.yandex-team.ru/AuthLib
 http://wiki.yandex-team.ru/passport/mda/intro

=cut

sub sessionid {
    my ($self, %opts) = @_;

    $self->timelog->start(gettext('BlackBox: Get sessionid'));

    my $session_id = delete $opts{'session_id'};
    my $sessionid2 = delete $opts{'sessionid2'};

    my $remote_addr = delete $opts{'remote_addr'};
    throw Exception::BadArguments gettext("Expected remote_addr") if !defined($remote_addr);

    my $server_name = delete $opts{'server_name'};
    throw Exception::BadArguments gettext("Expected server_name") if !defined($server_name);

    my @fields = @{delete $opts{'fields'}};

    throw Exception::BadArguments gettext("Got unknown parameters: %s", join(", ", keys(%opts))) if %opts;

    my $pass = $self->_get_addr_pass($server_name);

    if (!$session_id) {
        if ($pass->{master_domain}) {
            throw Exception::BlackBox::NeedAuth 'No session', url => $pass->{passport};
        } else {
            throw Exception::BlackBox::NeedResign 'Try to sync session from master passport', url => $pass->{pass};
        }
    }

    my $res = $self->_get(
        method    => 'sessionid',
        sessionid => $session_id,
        (defined($sessionid2) ? (sslsessionid => $sessionid2) : ()),
        userip => $remote_addr,
        host   => $server_name,
        emails => 'getdefault',
        (@fields ? (dbfields => join(',', @fields)) : ()),
    );

    $self->timelog->finish();

    ldump({blackbox => $res}) if $DEBUG;

    throw Exception::BlackBox::InternalError($res->{'error'}
          && $res->{'error'}->{'content'} ? $res->{'error'}->{'content'} : gettext('Unknown error'))
      unless $res->{'status'}
          && defined($res->{'status'}->{'content'});

    if ($res->{'status'}->{'content'} eq 'VALID') {
        # if social - go away and bring back YA.login
        throw Exception::BlackBox::NeedRealLogin 'Yandex login - must have. Only social login doesn\'t matter',
          url => $pass->{passport}
          if (!$res->{'login'}->{'content'} && !$self->get_option('MayOnlySocial'));

        # All ok
        return $res;
    } elsif ($res->{'status'}{'id'} == 5) {
        throw Exception::BlackBox::NeedAuth 'Not authorized', url => $pass->{passport};
    }

    # TODO: think about JSON & POST
    if ($pass->{master_domain}) {
        if ($res->{'status'}->{'content'} eq 'NEED_RESET') {
            throw Exception::BlackBox::NeedResign 'Session wants to be fresh', url => $pass->{pass}, res => $res;
        } else {
            throw Exception::BlackBox::NeedAuth 'Not authorized', url => $pass->{passport};
        }
    } else {
        if ($res->{'status'}->{'content'} eq 'NOAUTH') {
            throw Exception::BlackBox::NeedAuth 'Session too old', url => $pass->{passport};
        } else {
            throw Exception::BlackBox::NeedResign 'Session wants to be fresh or get from master domains',
              url     => $pass->{pass},
              res     => $res,
              cookies => {Cookie_check => 'CheckCookieCheckCookie; domain=.yandex.' . $pass->{tld} . '; path=/;'};
        }
    }

    return $res;
}

=head2 oauth

Получаем информацию о сессии из ящика. В случае ошибки 2 типа исключения:

    my $bb_res = $self->api_blackbox->oauth(
        token       => $self->request->http_header('Authorization'),
        remote_addr => $self->request->remote_addr(),
        userip      => $self->request->remote_addr(),
        fields      => [qw(
            accounts.login.uid
            account_info.fio.uid
        )],
    );

=cut

sub oauth {
    my ($self, %opts) = @_;

    $self->timelog->start(gettext('BlackBox: Get oauth'));

    my @missed_required_fields = grep {!defined($opts{$_})} qw(token userip);
    throw Exception::BadArguments ngettext(
        'Missed required argument %s',
        'Missed required arguments %s',
        scalar(@missed_required_fields),
        join(', ', @missed_required_fields)
    ) if @missed_required_fields;

    my @fields = @{$opts{'fields'} // []};

    my $res = $self->_get(
        method      => 'oauth',
        oauth_token => $opts{'token'},
        userip      => $opts{'userip'},
        (@fields ? (dbfields => join(',', @fields)) : ()),
    );

    $self->timelog->finish();

    ldump({blackbox => $res}) if $DEBUG;

    throw Exception::BlackBox::InternalError($res->{'error'}
          && $res->{'error'}->{'content'} ? $res->{'error'}->{'content'} : gettext('Unknown error'))
      unless $res->{'status'}
          && defined($res->{'status'}->{'content'});

    if ($res->{'status'}->{'content'} ne 'VALID') {
        throw Exception::BlackBox::NeedAuth 'Not authorized';
    }

    return $res;
}

=head2 userinfo

B<Параметры:> %opts

B<Возвращаемое значение:> $ со структорой ответа BlackBox

http://doc.yandex-team.ru/blackbox/reference/MethodUserInfo.xml

=cut

sub userinfo {
    my ($self, %opts) = @_;

    $self->timelog->start(gettext('BlackBox: Get userinfo'));

    my @fields = @{delete $opts{fields}};

    my $data = $self->_get(
        method => 'userinfo',
        userip => ($opts{userip} || '127.0.0.1'),
        %opts,
        (@fields ? (dbfields => join(',', @fields)) : ()),
    );

    $self->timelog->finish();

    return $data;
}

sub _get {
    my ($self, %fields) = @_;

    my $url =
        $self->get_option('URL')
      . '/blackbox/?'
      . join('&', map {uri_escape_utf8($_) . '=' . uri_escape_utf8($fields{$_})} keys(%fields));
    my $req = HTTP::Request->new('GET' => $url);
    l($req->as_string()) if $self->get_option('Debug');

    $self->timelog->start(gettext('Get XML data'));
    my $resp;
    for my $timeout (2, 5) {
        $self->ua->timeout($timeout);
        $resp = $self->ua->request($req);
        if ($resp->is_success()) {
            last;
        } else {
            l("Blackbox request(timeout: $timeout) failed: " . $resp->status_line());
        }
    }
    $self->timelog->finish();

    l($resp->as_string()) if $self->get_option('Debug');

    if ($resp->is_success()) {
        $self->timelog->start(gettext('Parse XML data'));
        my $res = XML::Simple::XMLin($resp->decoded_content(), ForceContent => 1, ForceArray => ['dbfield']);
        throw Exception::BlackBox $res->{'error'}{'content'}
          if ((($res->{'error'}{'content'} || '') ne 'OK') and (!$res->{'login'}{'content'}))
          and exists($res->{'status'})
          and $res->{'status'}{'id'} != 5;

        if ($res->{'dbfield'}) {
            $res->{'dbfield'}{$_} = $res->{'dbfield'}{$_}{'content'} foreach keys(%{$res->{'dbfield'}});
        }
        $self->timelog->finish();
        return $res;
    } else {
        throw Exception::BlackBox::Net $resp->status_line();
    }
}

=head2 _get_addr_pass

Получаем информацию о вхождении домена в главный или подчиненные домены

 tld - $, запрашиваемый top level domain
 master_domain - флаг, принадлежность к главному домену
 pass, passport - $, соответствующие урлы для пасса и пасспорта (включают // и ?)

=cut

sub _get_addr_pass {
    my ($self, $server_name) = @_;

    my $master_domains = {
        'ru'     => ['//passport.yandex.ru/passport?',     '//pass.yandex.ru/?'],
        'com'    => ['//passport.yandex.com/passport?',    '//pass.yandex.com/resign?'],
        'com.tr' => ['//passport.yandex.com.tr/passport?', '//pass.yandex.com.tr/resign?'],
    };

    my $tld = $server_name =~ /\.yandex\.(.+)$/i ? lc($1) : 'ru';
    my ($url_passport, $url_pass) = @{$master_domains->{$tld} // $master_domains->{ru}};

    return {
        tld           => $tld,                         # our top level domain
        master_domain => !!$master_domains->{$tld},    # is our domain is master ?
        pass          => $url_pass,                    # 'pass' url
        passport      => $url_passport,                # 'passport' url
    };
}

TRUE;
