package Exception::BK;
use base qw(Exception);

package QBit::Application::Model::API::Yandex::HTTPBK;

use qbit;

use base qw(QBit::Application::Model::API::HTTP);
use Utils::TSV;

sub _convert_to {
    my ($date, $format) = @_;

    local $qbit::Date::TR_HS{'bs'} = {
        '>' => sub {sprintf('%04d%02d%02d000000', @{$_[0]})},
        '<' => sub {shift =~ /(\d{4})(\d{2})(\d{2})000000/ ? [$1, $2, $3, 0, 0, 0] : [0, 0, 0, 0, 0, 0]},
    };

    return $format eq 'bs' ? trdate('db', 'bs', $date) : trdate('bs', 'db', $date),;
}

sub get_dsp_info {
    my ($self, $dsp_id) = @_;

    my %params = (func => 'info');
    %params = (
        func   => 'get_dsp',
        dsp_id => $dsp_id
    ) if $dsp_id;

    my $data = $self->call('export/dsp-api.cgi', %params);

    my $result = from_json($data);

    throw Exception::BK $result->{'error'} if $result->{'code'} != 0;

    return $result->{'data'};
}

sub start_dsp {
    my ($self, $dsp_id) = @_;

    throw Exception::BadArguments(gettext('Expected that DSP ID is defined')) unless defined($dsp_id);

    my $data = $self->call(
        'export/dsp-api.cgi',
        func   => 'start_dsp',
        dsp_id => $dsp_id
    );

    my $result = from_json($data);

    throw Exception::BK $result->{'error'} if $result->{'code'} != 0;

    return $result;
}

sub stop_dsp {
    my ($self, $dsp_id) = @_;

    throw Exception::BadArguments(gettext('Expected that DSP ID is defined')) unless defined($dsp_id);

    my $data = $self->call(
        'export/dsp-api.cgi',
        func   => 'stop_dsp',
        dsp_id => $dsp_id
    );

    my $result = from_json($data);

    throw Exception::BK $result->{'error'} if $result->{'code'} != 0;

    return $result;
}

sub get_big_search_page_ids {
    my ($self) = @_;

    my $data = $self->call('export/export-tagstat-pages.cgi');

    return [split(/\s+/, $data)];
}

sub get_serp_tag_hits {
    my ($self, $starttime, $stoptime) = @_;

    my $bs_start = _convert_to($starttime, 'bs');
    my $bs_end   = _convert_to($stoptime,  'bs');

    my $data = $self->call(
        'export/export_serptaghits.cgi',
        starttime => $bs_start,
        stoptime  => $bs_end,
    );

    throw Exception::BK gettext('Got no end marker. Starttime: "%s", endtime: "%s".', $bs_start, $bs_end)
      if $data !~ s/#\s*end\r?\n//si;

    my @result = ();

    while ($data =~ /([^\r\n]+)?\r?\n/imsg) {
        my @data = split(/\t/, $1);
        push(
            @result,
            {
                UpdateTime => _convert_to($data[0], 'db'),
                PageID     => $data[1],
                CLID       => $data[2],
                Shows      => $data[3]
            }
        );
    }

    return \@result;
}

sub get_dsp_statistics_responses {
    my ($self, $dsp_id, $from_date, $to_date) = @_;

    my %opts = ();

    $opts{'fromdate'} = $from_date if defined($from_date);
    $opts{'todate'}   = $to_date   if defined($to_date);

    foreach (keys(%opts)) {
        throw gettext('Expected date format YYYY-mm-dd') unless check_date($opts{$_}, iformat => 'db');
        $opts{$_} =~ s/-//g;
    }

    $opts{'dspid'} = $dsp_id if defined($dsp_id);

    my $data = $self->call('export/export-dsppageerror.cgi', %opts);

    $data =~ m/^([^\r\n]+)?\r?\n(.*)?\r?\n?$/ms;

    if ($2) {
        $data = $2;
    } elsif ($1 && $1 =~ m/^DSPID/) {
        return [];
    } else {
        throw Exception::BK($1);
    }

    my @result = ();

    while ($data =~ /([^\r\n]+)?\r?\n/imsg) {
        my @data = split(/\t/, $1);
        $data[2] =~ s/([0-9]{4})([0-9]{2})([0-9]{2})/$1-$2-$3/;
        push(
            @result,
            {
                DSPID      => $data[0],
                PageID     => $data[1],
                UpdateDate => $data[2],
                ErrorType  => $data[3],
                ErrorsSum  => $data[4]
            }
        );
    }

    return \@result;
}

sub set_skipnoud {
    my ($self, $dsp_id, $skipnoud) = @_;

    throw Exception::BadArguments(gettext('Expected that DSP ID is defined')) unless defined($dsp_id);

    my $data = $self->call(
        'export/dsp-api.cgi',
        func     => 'set_skipnoud',
        dsp_id   => $dsp_id,
        skipnoud => $skipnoud ? 'true' : 'false'
    );

    my $result = from_json($data);

    throw Exception::BK $result->{'error'} if $result->{'code'} != 0;

    return $result;
}

sub add_dsp {
    my ($self, $dsp_id, $dsp_tag, $url, $disabled, %opts) = @_;

    throw Exception::BadArguments(gettext('Expected that DSP ID is defined'))  unless defined($dsp_id);
    throw Exception::BadArguments(gettext('Expected that DSP TAG is defined')) unless defined($dsp_tag);
    throw Exception::BadArguments(gettext('Expected that DSP URL is defined')) unless defined($url);

    if (exists($opts{'dsp_types'})) {
        my $bk_type;
        $bk_type |= 2**$_ foreach @{$opts{'dsp_types'}};

        $opts{'dsp_types'} = $bk_type;
    }

    my $data = $self->call(
        'export/dsp-api.cgi',
        func     => 'add_dsp',
        dsp_id   => $dsp_id,
        dsp_tag  => $dsp_tag,
        url      => $url,
        disabled => $disabled ? 1 : 0,
        hash_transform(\%opts, ['skipnoud', 'show_probability'], {test_url => 'testurl', dsp_types => 'dsp_type'})
    );

    my $result = from_json($data);

    throw Exception::BK $result->{'error'} if $result->{'code'} != 0;

    return $result;
}

=head1 get_partner_pages

    Cм. подробнее http://wiki.yandex-team.ru/partner/w/get-partner-pages

=cut

sub get_partner_pages {
    my ($self, $page_ids) = @_;

    my $data = $self->call('export/get-partner-pages.cgi', ($page_ids ? (pages => join(',', @$page_ids)) : ()));

    $data =~ s/# //;
    throw Exception::BK gettext('no end of data marker received') unless $data =~ s/# end//;

    my $result = parse_tsv($data);

    return $result;
}

sub get_apfraudstat {
    my ($self, $date) = @_;

    my $bs_date = _convert_to($date, 'bs');

    my $data = $self->call('export/export_apfraudstat.cgi', date => $bs_date,);

    return unless defined($data);

    my @data = ();

    push(
        @data,
        map {
            {
                PageID      => $_->[0],
                UpdateTime  => _convert_to($_->[1], 'db'),
                Shows       => $_->[2],
                FraudShows  => $_->[3],
                Clicks      => $_->[4],
                FraudClicks => $_->[5],
                Cost        => $_->[6],
                FraudCost   => $_->[7],
                AdShows     => $_->[8],
                AdClicks    => $_->[9],
            }
          }
          map {
            [split("\t", $_)]
          }
          grep {
            $_
          }
          split("\n", $data)
        );

    return \@data;
}

sub get_page_block_shows {
    my ($self, $date) = @_;

    my $bs_date = _convert_to($date, 'bs');

    my $data = $self->call('export/export_pageblockshows.cgi', date => $bs_date,);

    throw Exception::BK gettext('BK returned undef') unless defined($data) && $data =~ s/\s+#End\s+$//msi;

    my @data = ();

    push(
        @data,
        map {
            {
                UpdateTime => _convert_to($_->[0], 'db'),
                PageID     => $_->[1],
                PlaceID    => $_->[2],
                Flag       => $_->[3],
                PageAdShow => $_->[4],
            }
          }
          map {
            [split("\t", $_)]
          }
          grep {
            $_
          }
          split("\n", $data)
        );

    return \@data;
}

sub edit_dsp {
    my ($self, $dsp_id, %opts) = @_;

    throw Exception::BadArguments(gettext('Expected that DSP ID is defined')) unless defined($dsp_id);

    if (exists($opts{'dsp_types'})) {
        my $bk_type;
        $bk_type |= 2**$_ foreach @{$opts{'dsp_types'}};

        $opts{'dsp_types'} = $bk_type;
    }

    my %params = hash_transform(
        \%opts,
        [qw(dsp_id dsp_tag url disabled skipnoud show_probability)],
        {test_url => 'testurl', dsp_types => 'dsp_type'}
    );

    foreach (qw(skipnoud disabled)) {
        $params{$_} = $params{$_} ? 1 : 0 if exists($params{$_});
    }

    my $data = $self->call(
        'export/dsp-api.cgi',
        func   => 'edit_dsp',
        dsp_id => $dsp_id,
        %params
    );

    my $result = from_json($data);

    throw Exception::BK $result->{'error'} if $result->{'code'} != 0;

    return $result;
}

=head1 all_pages_hits_stat_day

L<https://wiki.yandex-team.ru/users/mllnr/Подневнаястатистикапопейджам>

    ->all_pages_hits_stat_day('2014-07-16')

=cut

sub all_pages_hits_stat_day {
    my ($self, $date) = @_;

    my $bs_date = _convert_to($date, 'bs');

    my $data = $self->call('export/all-pages-hits-stat-day.cgi', POSTDATA => $bs_date,);

    throw Exception::BK gettext('BK returned undef') unless defined($data);

    my $result = from_json($data);

    throw Exception::BK $result->{'ErrorMessage'} if ref($result) eq 'HASH' && $result->{'Error'} == 1;

    return [
        map {
            {%$_, UpdateTime => _convert_to($_->{'UpdateTime'}, 'db')}
          } grep {
            $_->{'UpdateTime'} eq $bs_date
          } @{$result}
    ];
}

sub get_dsp_page_stat {
    my ($self, %opts) = @_;

    foreach (qw(starttime stoptime)) {
        throw gettext('Expected date format YYYY-mm-dd') unless check_date($opts{$_}, iformat => 'db');
        $opts{$_} =~ s/-//g;
    }

    my $data = $self->call('export/export-dsppagestat.cgi', hash_transform(\%opts, [qw(starttime stoptime)]));

    my @result = ();

    while ($data =~ /([^\r\n]+)?\r?\n/imsg) {
        my @data = split(/\t/, $1);
        push(
            @result,
            {
                Date            => $data[0],
                PageID          => $data[1],
                ImpID           => $data[2],
                DSPID           => $data[3],
                WinHits         => $data[4],
                WinPrice        => $data[5],
                WinPartnerPrice => $data[6],
                Shows           => $data[7],
                AllHits         => $data[8],
                AllRealPrice    => $data[9],
            }
        );
    }

    return \@result;
}

TRUE;
