
=encoding UTF-8

=head1 Name

QBit::Application::Model::Yandex::Geobase - Модель для работы с геобазой

=head1 Description

Скроманя документация по geobase3 http://wiki.yandex-team.ru/rimzaidullin/libgeobase3
Реализованные методы можно смотреть в коде либы (тесты - geobase3.t и в geobase3.xs)

=cut

package QBit::Application::Model::Yandex::Geobase;

use qbit;

use base qw(QBit::Application::Model);

use geobase3;
use Data::Validate::IP qw(is_ipv4 is_ipv6);

our $LOOKUPS;    # { file => '/foo/bar', lastcheck => 12345, fmodif => 12300, obj => bless(...) }, {,,,}
our $LOOKUP_REFRESH_INTERVAL = 20;

our $REG_ID_ROOT = 10000;    # earth

# Прелоад списка стра (saved_list_countries) выпилен
#   ест память (держит старый файл) при апдейте геобазы в родительском процессе fcgi демона
our %saved_list_countries;

=head1 Methods

=cut

=head2 to_hash

B<Arguments:>

=over

=item B<$obj> - объект geobase3::region

=back

B<Return value:> указатель на хеш, с ключами id, parent_id, type, is_main_region, name, english_name.

=cut

sub to_hash {
    my ($self, $obj) = @_;

    my $ret;
    if (ref($obj) eq 'geobase3::region' && $obj->id) {
        # get fields
        $ret = {map(($_ => $obj->$_), qw/id parent_id type is_main_region name english_name/)};

        # unicode symbols
        map(utf8::decode($ret->{$_}), qw/name english_name/);

        # espand hash with extra fields
        $ret->{childs} = $self->_get_lookup()->children($obj->id) || [];

        # ... add 'path' ???
    }

    return $ret;
}

=head2 get_region_by_id

B<Arguments:>

=over

=item B<$id> - число, ID региона

=back

B<Return value:> указатель на хеш или undef, если не найден. Хеш аналогичен хешу метода L</to_hash>.

=cut

sub get_region_by_id {
    my ($self, $id) = @_;

    my $reg = geobase3::region->new();
    $self->_get_lookup()->region_by_id($id, $reg) || return undef;

    return $self->to_hash($reg);
}

=head2 get_region_parents_by_id

B<Arguments:>

=over

=item B<$id> - число, ID региона

=back

B<Return value:> указатель на массив чисел, ID родительских регионов.

=cut

sub get_region_parents_by_id {
    my ($self, $id) = @_;

    return $self->_get_lookup()->parents($id || 10000);
}

=head2 get_geo_slice_type

B<Arguments:>

=over

=item B<$id> - число, id стартового региона;

=item B<$type_limit> - число, максимальный уровень.

=back

B<Return value:> указатель на массив, дерево регионов.

=cut

sub get_geo_slice_type {
    my ($self, $id, $type_limit) = @_;

    our $REG_ID_ROOT;
    $id //= $REG_ID_ROOT;
    $type_limit //= 4;

    my $reg = $self->get_region_by_id($id);

    return [] if (!$reg || $reg->{type} > $type_limit);
    return [$reg] if ($reg->{type} == $type_limit);
    return [map(@{$self->get_geo_slice_type($_, $type_limit)}, @{$reg->{childs}})];
}

=head2 get_geo_name

B<Arguments:>

=over

=item B<$ref> - число (ID региона) или объект типа geobase3::region;

=item B<$locale> - строка, локаль, по умолчанию текущая локаль приложения или ru_RU.

=back

B<Return value:> строка, название региона.

=cut

sub get_geo_name {
    my ($self, $ref, $locale) = @_;

    my $reg = ref($ref) ? $ref : $self->get_region_by_id($ref);
    $locale = $self->_get_locale($locale);
    my $name;

    if ($locale =~ /^en_/) {
        $name = $reg->{'english_name'};
    } elsif ($locale =~ /^ru_/) {
        $name = $reg->{'name'};
    } elsif ($locale =~ /^([a-z]{2})_/) {
        my $res_ling = geobase3::linguistics->new();
        $self->_get_lookup()->linguistics_for_region($reg->{'id'}, $1, $res_ling);
        $name = $res_ling->nominative_case || $reg->{'english_name'};
        utf8::decode($name);
    } else {
        throw gettext('Bad locale "%s"', $locale);
    }

    return $name;
}

=head2 list_countries

B<Arguments:>

=over

=item B<$locale> - строка, аналогично методу L</get_geo_name>.

=back

B<Return value:> указатель на массив хешей.

=cut

sub list_countries {
    my ($self, $locale) = @_;

    my $countries;
    $locale = $self->_get_locale($locale);
    my $fname = $self->get_option('filename', '/var/cache/geobase/geodata-local3.bin');

    if ($saved_list_countries{$fname}->{$locale}) {
        return $saved_list_countries{$fname}->{$locale};
    } else {
        foreach my $reg (@{$self->get_geo_slice_type(undef, 3)}) {
            push(
                @$countries,
                {
                    id   => $reg->{id},
                    name => $self->get_geo_name($reg, $locale),
                }
            );
        }

        return $saved_list_countries{$fname}->{$locale} = $countries;
    }
}

=head2 get_region_by_ip

B<Arguments:>

=over

=item B<$ip> - строка, понимает IPv4 и IPv6.

=back

B<Return value:> указатель на хеш. Хеш как в методе L</to_hash>.

=cut

sub get_region_by_ip {
    my ($self, $ip) = @_;

    my $reg = geobase3::region->new();

    $self->_get_lookup()->region_by_ip($ip, $reg);

    return undef unless $reg && $reg->id;

    return $self->to_hash($reg);
}

sub is_child {
    my ($self, $child, $parent) = @_;

    return $self->_get_lookup->region_in_region($child, $parent);
}

=head2 get_region_parents_by_ip

B<Arguments:>

=over

=item B<$ip> - строка, понимает IPv4 и IPv6.

=back

B<Return value:> указатель на массив чисел, ID родительских регионов.

=cut

sub get_region_parents_by_ip {
    my ($self, $ip) = @_;

    my $region = geobase3::region->new();
    $self->_get_lookup()->region_by_ip($ip, $region);

    return $self->get_region_parents_by_id($region->id);
}

=head2 is_yandex_internal_ip

B<Arguments:>

=over

=item B<$ip> - строка, понимает IPv4, IPv6

=back

B<Return value:> boolean значение является ли указанный ip адрес адресом
внутренней сети яндекса.

=cut

sub is_yandex_internal_ip {
    my ($self, $ip) = @_;

    if (is_ipv4($ip) || is_ipv6($ip)) {
        my $reg = $self->get_region_by_ip($ip);

        return $reg && $reg->{'id'} == 9999;
    } else {
        throw gettext('Incorrect ip "%s"', $ip);
    }
}

sub _get_lookup {
    my ($self) = @_;

    my $fname = $self->get_option('filename', '/var/cache/geobase/geodata3.bin');

    # return global obj
    return $LOOKUPS->{$fname}->{'obj'} if (exists($LOOKUPS->{$fname}->{'obj'}) && _is_fresh_lookup($fname));

    throw gettext('Geodata file "%s" does not exists', $fname) unless -f $fname;

    $self->timelog->start('Loading geobase') if $self->timelog;
    my $l_tmp = {
        file      => $fname,
        fmodif    => (stat($fname))[9],
        obj       => geobase3::lookup->new($fname, 0),
        lastcheck => time(),
    };

    # TODO: estimate data quality (records in base?)
    if ($l_tmp->{'obj'}) {
        $LOOKUPS->{$fname} = $l_tmp;
    } else {
        warn("geodata file is wrong [$l_tmp->{file}]\n") if (!$l_tmp->{obj});
    }
    $self->timelog->finish() if $self->timelog;

    return $LOOKUPS->{$fname}->{'obj'};
}

sub _is_fresh_lookup {
    my ($fname) = @_;

    return FALSE unless exists($LOOKUPS->{$fname}->{'obj'});

    if ((time() - $LOOKUPS->{$fname}->{'lastcheck'}) > $LOOKUP_REFRESH_INTERVAL) {
        $LOOKUPS->{$fname}->{'lastcheck'} = time();
        my $fmodif = (stat($fname))[9];
        return FALSE if $fmodif > $LOOKUPS->{$fname}->{'fmodif'};
    }

    return TRUE;
}

sub _get_locale {
    my ($self, $locale) = @_;

    my $locales = $self->get_option('locales', {ru => {code => 'ru_RU'}});
    my %codes = map {$_->{'code'} => 1} values(%$locales);

    if (defined($locale)) {
        throw gettext('Unknown locale "%s"', $locale) unless $codes{$locale};
        return $locale;
    }

    my $cur_locale = $self->get_option('locale') // return 'ru_RU';
    return $locales->{$cur_locale}{'code'};
}

1;
