package ru.yandex.direct.api.v5.clientcurrencyconversion;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.ws.context.MessageContext;
import org.springframework.ws.server.EndpointInterceptor;

import ru.yandex.direct.api.v5.ApiFaultTranslations;
import ru.yandex.direct.api.v5.context.ApiContextHolder;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.security.SecurityErrors;
import ru.yandex.direct.core.entity.client.service.ClientCurrencyConversionTeaserService;
import ru.yandex.direct.core.security.AccessDeniedException;
import ru.yandex.direct.dbutil.model.ClientId;

/**
 * Перехватчик узнаёт, затронут ли клиент сейчас переводом в валюту. Если да, ему
 * нельзя ходить в API, так что он получает сообщение об ошибке.
 * <p>
 * Какие клиенты считаются затронутыми:
 * А: которые прямо сейчас или очень скоро будут переводиться в валюту
 * Б: которые ещё в у. е. и должны в интерфейсе согласиться с тем, что их переведут в валюту
 * <p>
 * Клиенты (А) и клиенты (Б) получают разные сообщения об ошибке.
 */
@ParametersAreNonnullByDefault
@Component
public class ClientCurrencyConversionTeaserInterceptor implements EndpointInterceptor {
    private final ClientCurrencyConversionTeaserService clientCurrencyConversionTeaserService;
    private final ApiContextHolder apiContextHolder;
    private final ApiAuthenticationSource apiAuthenticationSource;

    @Autowired
    public ClientCurrencyConversionTeaserInterceptor(
            ClientCurrencyConversionTeaserService clientCurrencyConversionTeaserService,
            ApiContextHolder apiContextHolder,
            ApiAuthenticationSource apiAuthenticationSource) {
        this.clientCurrencyConversionTeaserService = clientCurrencyConversionTeaserService;
        this.apiContextHolder = apiContextHolder;
        this.apiAuthenticationSource = apiAuthenticationSource;
    }

    @Override
    public boolean handleRequest(MessageContext messageContext, Object endpoint) throws Exception {
        ClientId clientId = apiAuthenticationSource.getSubclient().getClientId();

        if (clientCurrencyConversionTeaserService.isClientConvertingSoon(clientId)) {
            apiContextHolder.get().setShouldChargeUnitsForRequest(false);
            throw SecurityErrors.newAccessToApiDeniedWhileConvertingCurrency();
        }

        if (clientCurrencyConversionTeaserService.doesClientHaveToConvert(clientId)) {
            apiContextHolder.get().setShouldChargeUnitsForRequest(false);
            throw new AccessDeniedException("Клиент обязан согласиться, чтобы его перевели в валюту",
                    ApiFaultTranslations.INSTANCE.clientHasToConvertToCurrency());
        }

        return true;
    }

    @Override
    public boolean handleResponse(MessageContext messageContext, Object endpoint) throws Exception {
        return true;
    }

    @Override
    public boolean handleFault(MessageContext messageContext, Object endpoint) throws Exception {
        return true;
    }

    @Override
    public void afterCompletion(MessageContext messageContext, Object endpoint, Exception ex) throws Exception {
        /* всю работу делает handleRequest, здесь делать нечего */
    }
}
