package ru.yandex.direct.api.v5.common;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.general.AutotargetingCategoriesEnum;
import com.yandex.direct.api.v5.general.AutotargetingCategory;
import com.yandex.direct.api.v5.general.AutotargetingCategoryArray;
import com.yandex.direct.api.v5.general.ObjectFactory;
import com.yandex.direct.api.v5.general.YesNoEnum;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatchCategory;

import static ru.yandex.direct.core.entity.relevancematch.Constants.AD_GROUP_TYPES_ALLOWED_FOR_RELEVANCE_MATCH_CATEGORIES;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.filterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class RelevanceMatchCategoriesConverter {

    private static final ObjectFactory FACTORY = new ObjectFactory();

    private static final Map<AutotargetingCategoriesEnum, Boolean> DEFAULT_CATEGORIES = Map.of(
            AutotargetingCategoriesEnum.EXACT, true,
            AutotargetingCategoriesEnum.ACCESSORY, true,
            AutotargetingCategoriesEnum.ALTERNATIVE, true,
            AutotargetingCategoriesEnum.BROADER, true,
            AutotargetingCategoriesEnum.COMPETITOR, true
    );

    private static final Map<AutotargetingCategoriesEnum, RelevanceMatchCategory> AUTOTARGETING_CATEGORIES_TO_RELEVANCE_MATCH_CATEGORIES = Map.of(
            AutotargetingCategoriesEnum.EXACT, RelevanceMatchCategory.exact_mark,
            AutotargetingCategoriesEnum.ACCESSORY, RelevanceMatchCategory.accessory_mark,
            AutotargetingCategoriesEnum.ALTERNATIVE, RelevanceMatchCategory.alternative_mark,
            AutotargetingCategoriesEnum.BROADER, RelevanceMatchCategory.broader_mark,
            AutotargetingCategoriesEnum.COMPETITOR, RelevanceMatchCategory.competitor_mark
    );

    private static final Map<RelevanceMatchCategory, AutotargetingCategoriesEnum> RELEVANCE_MATCH_CATEGORIES_TO_AUTOTARGETING_CATEGORIES = Map.of(
            RelevanceMatchCategory.exact_mark, AutotargetingCategoriesEnum.EXACT,
            RelevanceMatchCategory.accessory_mark, AutotargetingCategoriesEnum.ACCESSORY,
            RelevanceMatchCategory.alternative_mark, AutotargetingCategoriesEnum.ALTERNATIVE,
            RelevanceMatchCategory.broader_mark, AutotargetingCategoriesEnum.BROADER,
            RelevanceMatchCategory.competitor_mark, AutotargetingCategoriesEnum.COMPETITOR
    );

    private RelevanceMatchCategoriesConverter() {
        // no instantiation
    }

    @Nullable
    public static Set<RelevanceMatchCategory> autotargetingCategoriesToCore(
            List<AutotargetingCategory> autotargetingCategories, AdGroupType adGroupType) {
        if (!AD_GROUP_TYPES_ALLOWED_FOR_RELEVANCE_MATCH_CATEGORIES.contains(adGroupType)) {
            return isEmpty(autotargetingCategories)
                    ? null
                    : listToSet(autotargetingCategories,
                    ac -> AUTOTARGETING_CATEGORIES_TO_RELEVANCE_MATCH_CATEGORIES.get(ac.getCategory()));
        }

        return autotargetingCategoriesToCore(autotargetingCategories);
    }

    public static Set<RelevanceMatchCategory> autotargetingCategoriesToCore(
            List<AutotargetingCategory> autotargetingCategories) {
        Map<AutotargetingCategoriesEnum, Boolean> valueByCategory = new HashMap<>();

        for (AutotargetingCategory ac : autotargetingCategories) {
            valueByCategory.put(ac.getCategory(), ac.getValue() == YesNoEnum.YES);
        }

        return filterAndMapToSet(
                Arrays.asList(AutotargetingCategoriesEnum.values()),
                category -> valueByCategory.getOrDefault(category, DEFAULT_CATEGORIES.get(category)),
                AUTOTARGETING_CATEGORIES_TO_RELEVANCE_MATCH_CATEGORIES::get);
    }

    @Nullable
    public static AutotargetingCategoryArray autotargetingCategoriesFromCore(
            @Nullable Set<RelevanceMatchCategory> relevanceMatchCategories,
            AdGroupType adGroupType) {
        if (!AD_GROUP_TYPES_ALLOWED_FOR_RELEVANCE_MATCH_CATEGORIES.contains(adGroupType)) {
            return null;
        }

        return autotargetingCategoriesFromCore(relevanceMatchCategories);
    }

    public static AutotargetingCategoryArray autotargetingCategoriesFromCore(
            @Nullable Set<RelevanceMatchCategory> relevanceMatchCategories) {
        if (isEmpty(relevanceMatchCategories)) {
            //пустое поле - это дефолт. В данный момент это означает включенность всех категорий
            relevanceMatchCategories = filterToSet(RELEVANCE_MATCH_CATEGORIES_TO_AUTOTARGETING_CATEGORIES.keySet(),
                    rmc -> DEFAULT_CATEGORIES.get(RELEVANCE_MATCH_CATEGORIES_TO_AUTOTARGETING_CATEGORIES.get(rmc)));
        }

        Map<AutotargetingCategoriesEnum, YesNoEnum> valueByCategory = StreamEx.of(AutotargetingCategoriesEnum.values())
                .mapToEntry(category -> YesNoEnum.NO)
                .toMap();

        for (RelevanceMatchCategory rmc : relevanceMatchCategories) {
            valueByCategory.put(RELEVANCE_MATCH_CATEGORIES_TO_AUTOTARGETING_CATEGORIES.get(rmc), YesNoEnum.YES);
        }

        return FACTORY.createAutotargetingCategoryArray()
                .withItems(mapList(valueByCategory.entrySet(),
                        entry -> new AutotargetingCategory().withCategory(entry.getKey()).withValue(entry.getValue())));
    }

    @Nullable
    public static Set<RelevanceMatchCategory> mergeAutotargetingCategoriesFromRequestWithDb(
            List<AutotargetingCategory> autotargetingCategories,
            @Nullable Set<RelevanceMatchCategory> relevanceMatchCategories,
            AdGroupType adGroupType) {
        if (!AD_GROUP_TYPES_ALLOWED_FOR_RELEVANCE_MATCH_CATEGORIES.contains(adGroupType)) {
            return isEmpty(autotargetingCategories)
                    ? null
                    : listToSet(autotargetingCategories,
                    ac -> AUTOTARGETING_CATEGORIES_TO_RELEVANCE_MATCH_CATEGORIES.get(ac.getCategory()));
        }

        return mergeAutotargetingCategoriesFromRequestWithDb(autotargetingCategories, relevanceMatchCategories);
    }

    public static Set<RelevanceMatchCategory> mergeAutotargetingCategoriesFromRequestWithDb(
            List<AutotargetingCategory> autotargetingCategories,
            @Nullable Set<RelevanceMatchCategory> relevanceMatchCategories) {
        if (isEmpty(relevanceMatchCategories)) {
            //пустое поле - это дефолт. В данный момент это означает включенность всех категорий
            relevanceMatchCategories = filterToSet(RELEVANCE_MATCH_CATEGORIES_TO_AUTOTARGETING_CATEGORIES.keySet(),
                    rmc -> DEFAULT_CATEGORIES.get(RELEVANCE_MATCH_CATEGORIES_TO_AUTOTARGETING_CATEGORIES.get(rmc)));
        }

        Map<AutotargetingCategoriesEnum, YesNoEnum> valueByCategory = new HashMap<>();

        for (AutotargetingCategory ac : autotargetingCategories) {
            valueByCategory.put(ac.getCategory(), ac.getValue());
        }

        for (RelevanceMatchCategory rmc : relevanceMatchCategories) {
            valueByCategory.putIfAbsent(RELEVANCE_MATCH_CATEGORIES_TO_AUTOTARGETING_CATEGORIES.get(rmc), YesNoEnum.YES);
        }

        return filterAndMapToSet(
                valueByCategory.entrySet(),
                entry -> entry.getValue() == YesNoEnum.YES,
                entry -> AUTOTARGETING_CATEGORIES_TO_RELEVANCE_MATCH_CATEGORIES.get(entry.getKey()));
    }
}
