package ru.yandex.direct.api.v5.common.validation;

import java.util.HashMap;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Function;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableMap;

import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectId;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;

/**
 * Класс для получения представлений дефектов {@link ApiDefectPresentation} по {@link DefectId}
 */
public class DefectPresentationsHolder {

    private final DefectPresentationsHolder fallback;
    private final ImmutableMap<DefectId, ApiDefectPresentation> presentationMap;

    private DefectPresentationsHolder(@Nullable DefectPresentationsHolder fallback,
                                      ImmutableMap<DefectId, ApiDefectPresentation> presentationMap) {
        this.fallback = fallback;
        this.presentationMap = presentationMap;
    }

    /**
     * @return {@code true}, если для {@code defectId} в этом {@link DefectPresentationsHolder}'е есть
     * соответствующее представление.
     */
    public boolean contains(DefectId defectId) {
        boolean found = presentationMap.containsKey(defectId);
        if (!found && fallback != null) {
            found = fallback.contains(defectId);
        }
        return found;
    }

    /**
     * @return {@link ApiDefectPresentation}, соответствующее указанному {@code defectId} или {@code null},
     * если соответствия в этом {@link DefectPresentationsHolder}'е нет
     */
    @Nullable
    public ApiDefectPresentation get(DefectId defectId) {
        ApiDefectPresentation result = presentationMap.get(defectId);
        if (result == null && fallback != null) {
            result = fallback.get(defectId);
        }
        return result;
    }

    public Map<DefectId, ApiDefectPresentation> getPresentationMap() {
        return presentationMap;
    }

    /**
     * Удобный builder для {@link DefectPresentationsHolder}
     */
    public static Builder builder() {
        return new Builder(null);
    }

    /**
     * Этим {@link Builder}'ом в итоге создаётся экземпляр {@link DefectPresentationsHolder},
     * который будет обращатсья к {@code fallbackHolder}, если не найдёт маппинг для нужного {@link DefectId} у себя.
     */
    public static Builder builderWithFallback(DefectPresentationsHolder fallbackHolder) {
        return new Builder(fallbackHolder);
    }

    public static class Builder {
        private final DefectPresentationsHolder fallback;
        private final Map<DefectId, ApiDefectPresentation> presentationMap;
        private DefectType defaultDefectType;

        private Builder(@Nullable DefectPresentationsHolder fallback) {
            this.fallback = fallback;
            presentationMap = new HashMap<>();
            defaultDefectType = null;
        }

        /**
         * Указывает, какой {@link DefectType} должен быть использован как default'ный для {@link DefectId},
         * зарегистрированных черех {@link #registerWithDefault(DefectId)}. Полезно, например, при заполнении
         * fallback presentation holder'а.
         *
         * @param defaultDefectType {@link DefectType} "по умолчанию".
         */
        public Builder useDefaultDefectType(@Nullable DefectType defaultDefectType) {
            this.defaultDefectType = defaultDefectType;
            return this;
        }

        /**
         * Позволяет зарегистрировать {@link DefectId} с некоторым представлением "по умолчанию".
         * default {@link DefectType} задаётся в {@link #useDefaultDefectType(DefectType)}.
         *
         * @throws IllegalStateException если не был задан default {@link DefectType}.
         * @see #useDefaultDefectType(DefectType)
         */
        public <P> Builder registerWithDefault(DefectId<P> defectId) {
            checkState(defaultDefectType != null,
                    "Cannot register defectId '%s' with default defectType. Default defectType is not set",
                    defectId.getCode());
            register(defectId, new ApiDefectPresentation<>(defectId, t -> defaultDefectType));
            return this;
        }

        /**
         * Регистрирует представление {@code defectPresentation} для {@code defectId}
         */
        public <P> Builder register(DefectId<P> defectId, ApiDefectPresentation<P> defectPresentation) {
            checkArgument(!presentationMap.containsKey(defectId),
                    "Cannot register multiple defect presentations for %s", defectId);
            presentationMap.put(defectId, defectPresentation);
            return this;
        }

        /**
         * Регистрирует представление для {@code defectId} через функцию получения {@link DefectType}
         * по параметру {@link Defect}
         */
        public <P> Builder register(DefectId<P> defectId, Function<P, DefectType> defectTypeProvider) {
            register(defectId, new ApiDefectPresentation<>(defectId, defectTypeProvider));
            return this;
        }

        /**
         * Регистрирует представление для {@code defectId} через {@link DefectType}
         */
        public <P> Builder register(DefectId<P> defectId, DefectType defectType) {
            register(defectId, t -> defectType);
            return this;
        }

        /**
         * Делегировать часть конфигурирования стороннему коду
         */
        public Builder configure(Consumer<Builder> configurer) {
            configurer.accept(this);
            return this;
        }

        /**
         * @return готовый {@link DefectPresentationsHolder}
         */
        public DefectPresentationsHolder build() {
            return new DefectPresentationsHolder(fallback, ImmutableMap.copyOf(presentationMap));
        }
    }
}
