package ru.yandex.direct.api.v5.common.validation;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.function.Consumer;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;

import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.validation.result.DefectId;
import ru.yandex.direct.validation.result.Path;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;

@ParametersAreNonnullByDefault
public class DefectPresentationsHolderRespectingPathImpl implements DefectPresentationsHolderRespectingPath {

    private final DefectPresentationsHolder fallback;
    private final ImmutableMap<Key, ApiDefectPresentation> presentationMap;

    private DefectPresentationsHolderRespectingPathImpl(
            @Nullable DefectPresentationsHolder fallback,
            ImmutableMap<Key, ApiDefectPresentation> presentationMap) {
        this.fallback = fallback;
        this.presentationMap = presentationMap;
    }

    @Override
    public boolean contains(DefectId defectId, Path path) {
        var key = new Key(defectId, path);
        boolean found = presentationMap.containsKey(key);
        if (!found && fallback != null) {
            found = fallback.contains(defectId);
        }
        return found;
    }

    @Override
    @Nullable
    public ApiDefectPresentation get(DefectId defectId,
                                     Path path) {
        var key = new Key(defectId, path);
        ApiDefectPresentation result = presentationMap.get(key);
        if (result == null && fallback != null) {
            result = fallback.get(defectId);
        }
        return result;
    }

    public Map<Key, ApiDefectPresentation> getPresentationMap() {
        return presentationMap;
    }

    public static DefectPresentationsHolderRespectingPathImpl.Builder builderWithFallback(
            DefectPresentationsHolder fallbackHolder) {
        return new DefectPresentationsHolderRespectingPathImpl.Builder(fallbackHolder);
    }

    public static class Builder {
        private final DefectPresentationsHolder fallback;
        private final Map<Key, ApiDefectPresentation> presentationMap;
        private DefectType defaultDefectType;

        private Builder(@Nullable DefectPresentationsHolder fallback) {
            this.fallback = fallback;
            presentationMap = new HashMap<>();
            defaultDefectType = null;
        }

        public DefectPresentationsHolderRespectingPathImpl.Builder useDefaultDefectType(@Nullable DefectType defaultDefectType) {
            this.defaultDefectType = defaultDefectType;
            return this;
        }

        public <P> DefectPresentationsHolderRespectingPathImpl.Builder registerWithDefault(DefectId<P> defectId,
                                                                                           Path path) {
            checkState(defaultDefectType != null,
                    "Cannot register defectId '%s' with default defectType. Default defectType is not set",
                    defectId.getCode());
            register(defectId, path, new ApiDefectPresentation<>(defectId, t -> defaultDefectType));
            return this;
        }

        /**
         * Регистрирует представление {@code defectPresentation} для {@code defectId}
         */
        public <P> Builder register(DefectId<P> defectId,
                                    Path path,
                                    ApiDefectPresentation<P> defectPresentation) {
            var key = new Key(defectId, path);
            checkArgument(!presentationMap.containsKey(key),
                    "Cannot register multiple defect presentations for %s", defectId);
            presentationMap.put(key, defectPresentation);
            return this;
        }

        public <P> Builder register(DefectId<P> defectId,
                                    Path path,
                                    Function<P, DefectType> defectTypeProvider) {
            register(defectId, path, new ApiDefectPresentation<>(defectId, defectTypeProvider));
            return this;
        }

        public <P> DefectPresentationsHolderRespectingPathImpl.Builder register(DefectId<P> defectId,
                                                                                Path path,
                                                                                DefectType defectType) {
            register(defectId, path, t -> defectType);
            return this;
        }

        public DefectPresentationsHolderRespectingPathImpl.Builder configure(
                Consumer<DefectPresentationsHolderRespectingPathImpl.Builder> configurer) {
            configurer.accept(this);
            return this;
        }

        public DefectPresentationsHolderRespectingPathImpl build() {
            return new DefectPresentationsHolderRespectingPathImpl(fallback, ImmutableMap.copyOf(presentationMap));
        }
    }

    private static class Key {

        public Key(DefectId defectId, Path path) {
            this.defectId = defectId;
            this.path = path;
        }

        DefectId defectId;
        Path path;

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Key key = (Key) o;
            return Objects.equals(defectId, key.defectId) && Objects.equals(path, key.path);
        }

        @Override
        public int hashCode() {
            return Objects.hash(defectId, path);
        }
    }

}
