package ru.yandex.direct.api.v5.configuration;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.xml.JacksonXmlModule;
import com.fasterxml.jackson.dataformat.xml.XmlMapper;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.direct.common.jackson.jaxbmodule.JaxbModule;

import static com.fasterxml.jackson.databind.DeserializationFeature.ACCEPT_FLOAT_AS_INT;
import static ru.yandex.direct.api.v5.ws.WsConstants.JSON_MESSAGE_OBJECT_READER_BEAN_NAME;
import static ru.yandex.direct.api.v5.ws.WsConstants.JSON_MESSAGE_OBJECT_WRITER_BEAN_NAME;
import static ru.yandex.direct.api.v5.ws.WsConstants.SOAP_MESSAGE_OBJECT_MAPPER_BEAN_NAME;

/**
 * Конфигурация основных форматов поддерживаемых API (json + xml)
 */
@Configuration
public class MessageMapperConfiguration {
    /*
    Парсинг запроса проходит в два этапа:
    - сначала парсится весь json и параметры попадают в Map
       jsonMessageObjectMapper.readValue(inputStream, JsonRequestEnvelope.class)
    - потом делается convertValue(param, ***Request.class)

    Реально, на втором этапе Jackson сначала сериализует значение в TokenBuffer, а потом десериализует.
    Чтобы настройки сериализации не мешали десерализации - делаем два ObjectMapper-а.
     */
    @Bean(name = JSON_MESSAGE_OBJECT_READER_BEAN_NAME)
    public ObjectMapper jsonMessageObjectReader() {
        ObjectMapper mapper = new ObjectMapper();
        mapper.registerModule(new JaxbModule());

        // Запрещаем преобразовывать float-литералы в int, long, BigInteger поля (См. DIRECT-69208)
        mapper.configure(ACCEPT_FLOAT_AS_INT, false);

        return mapper;
    }

    @Bean(name = JSON_MESSAGE_OBJECT_WRITER_BEAN_NAME)
    public ObjectMapper jsonMessageObjectWriter() {
        ObjectMapper mapper = new ObjectMapper();
        mapper.registerModule(new JaxbModule());

        mapper.setDefaultPropertyInclusion(
                JsonInclude.Value.construct(
                        JsonInclude.Include.NON_NULL,
                        JsonInclude.Include.ALWAYS));

        // ActionResult.getWarnings() делает автооживление ж-((, поэтому просим jackson не использовать геттеры
        mapper.setVisibility(PropertyAccessor.ALL, JsonAutoDetect.Visibility.NONE);
        mapper.setVisibility(PropertyAccessor.FIELD, JsonAutoDetect.Visibility.ANY);
        return mapper;
    }

    @Bean(name = SOAP_MESSAGE_OBJECT_MAPPER_BEAN_NAME)
    public XmlMapper soapMessageObjectMapper() {
        JacksonXmlModule module = new JacksonXmlModule();
        module.setDefaultUseWrapper(false);
        XmlMapper mapper = new XmlMapper(module);
        mapper.registerModule(new JaxbModule());
        return mapper;
    }
}
