package ru.yandex.direct.api.v5.configuration;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import javax.servlet.Filter;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.FilterType;
import org.springframework.context.annotation.Import;
import org.springframework.core.annotation.Order;
import org.springframework.web.context.annotation.RequestScope;
import org.springframework.web.filter.CompositeFilter;
import org.springframework.web.servlet.LocaleResolver;
import org.springframework.ws.config.annotation.EnableWs;
import org.springframework.ws.config.annotation.WsConfigurerAdapter;
import org.springframework.ws.server.EndpointInterceptor;
import org.springframework.ws.server.MessageDispatcher;
import org.springframework.ws.server.endpoint.adapter.method.MethodArgumentResolver;
import org.springframework.ws.server.endpoint.adapter.method.MethodReturnValueHandler;
import org.springframework.ws.soap.server.SoapMessageDispatcher;
import org.springframework.ws.transport.http.MessageDispatcherServlet;

import ru.yandex.direct.api.v5.clientcurrencyconversion.ClientCurrencyConversionTeaserInterceptor;
import ru.yandex.direct.api.v5.context.ApiContextFilter;
import ru.yandex.direct.api.v5.logging.ApiLoggingFilter;
import ru.yandex.direct.api.v5.logging.ApiLoggingInterceptor;
import ru.yandex.direct.api.v5.ratelimit.RateLimitInterceptor;
import ru.yandex.direct.api.v5.security.AuthenticationFilter;
import ru.yandex.direct.api.v5.security.AuthenticationInterceptor;
import ru.yandex.direct.api.v5.security.AuthorizationFilter;
import ru.yandex.direct.api.v5.security.ServiceAuthInterceptor;
import ru.yandex.direct.api.v5.semaphore.ApiLockInterceptor;
import ru.yandex.direct.api.v5.semaphore.jvm.ApiJvmSemaphoreInterceptor;
import ru.yandex.direct.api.v5.service.ApiNoEmptyErrorListsInterceptor;
import ru.yandex.direct.api.v5.timeout.TimeoutInterceptor;
import ru.yandex.direct.api.v5.units.ApiUnitsService;
import ru.yandex.direct.api.v5.units.UnitsFilter;
import ru.yandex.direct.api.v5.units.UnitsInterceptor;
import ru.yandex.direct.api.v5.ws.ApiLocaleResolver;
import ru.yandex.direct.api.v5.ws.ApiObjectsArgumentAndReturnValueResolver;
import ru.yandex.direct.api.v5.ws.ApiWsHandlerAdapter;
import ru.yandex.direct.api.v5.ws.HandlerAdapterDelegateRule;
import ru.yandex.direct.api.v5.ws.SetApiResponseToContextFilter;
import ru.yandex.direct.api.v5.ws.enpointmapping.ApiMethodEndpointMapping;
import ru.yandex.direct.api.v5.ws.exceptionresolver.ApiExceptionResolver;
import ru.yandex.direct.api.v5.ws.json.JsonMessageFactory;
import ru.yandex.direct.api.v5.ws.soap.SoapMessageFactory;
import ru.yandex.direct.common.tracing.TraceContextFilter;
import ru.yandex.direct.core.entity.uac.grut.GrutContext;
import ru.yandex.direct.core.entity.uac.grut.RequestScopeGrutContext;
import ru.yandex.grut.client.GrutClient;

import static ru.yandex.direct.api.v5.ApiConstants.DEFAULT_LOCALE;
import static ru.yandex.direct.api.v5.ApiConstants.SUPPORTED_LOCALES;

@Configuration
@EnableWs
@Import({
        ApiConfiguration.class,
        SecurityConfiguration.class,
        WsdlConfiguration.class
})
@ComponentScan(
        basePackages = "ru.yandex.direct.api.v5.ws",
        excludeFilters = @ComponentScan.Filter(value = Configuration.class, type = FilterType.ANNOTATION)
)
public class WebServiceConfiguration extends WsConfigurerAdapter {
    private final AuthenticationInterceptor authenticationInterceptor;
    private final ApiLockInterceptor apiLockInterceptor;
    private final ServiceAuthInterceptor serviceAuthInterceptor;
    private final ApiLoggingInterceptor apiLoggingInterceptor;
    private final RateLimitInterceptor rateLimitInterceptor;
    private final UnitsInterceptor unitsInterceptor;
    private final ClientCurrencyConversionTeaserInterceptor clientCurrencyConversionTeaserInterceptor;
    private final ApiNoEmptyErrorListsInterceptor apiNoEmptyErrorListsInterceptor;
    private final TimeoutInterceptor timeoutInterceptor;

    private final ApiJvmSemaphoreInterceptor apiJvmSemaphoreInterceptor;
    private final ApiObjectsArgumentAndReturnValueResolver apiObjectRequestMethodArgumentResolver;

    @Autowired
    public WebServiceConfiguration(
            AuthenticationInterceptor authenticationInterceptor,
            ApiLockInterceptor apiLockInterceptor,
            ServiceAuthInterceptor serviceAuthInterceptor,
            ApiLoggingInterceptor apiLoggingInterceptor,
            RateLimitInterceptor rateLimitInterceptor,
            UnitsInterceptor unitsInterceptor,
            ClientCurrencyConversionTeaserInterceptor clientCurrencyConversionTeaserInterceptor,
            ApiNoEmptyErrorListsInterceptor apiNoEmptyErrorListsInterceptor,
            TimeoutInterceptor timeoutInterceptor,
            ApiJvmSemaphoreInterceptor apiJvmSemaphoreInterceptor,
            ApiObjectsArgumentAndReturnValueResolver apiObjectRequestMethodArgumentResolver) {
        this.authenticationInterceptor = authenticationInterceptor;
        this.apiLockInterceptor = apiLockInterceptor;
        this.serviceAuthInterceptor = serviceAuthInterceptor;
        this.apiLoggingInterceptor = apiLoggingInterceptor;
        this.rateLimitInterceptor = rateLimitInterceptor;
        this.unitsInterceptor = unitsInterceptor;
        this.clientCurrencyConversionTeaserInterceptor = clientCurrencyConversionTeaserInterceptor;
        this.apiNoEmptyErrorListsInterceptor = apiNoEmptyErrorListsInterceptor;
        this.timeoutInterceptor = timeoutInterceptor;
        this.apiJvmSemaphoreInterceptor = apiJvmSemaphoreInterceptor;
        this.apiObjectRequestMethodArgumentResolver = apiObjectRequestMethodArgumentResolver;
    }

    @Bean
    LocaleResolver localeResolver() {
        return new ApiLocaleResolver(DEFAULT_LOCALE, SUPPORTED_LOCALES);
    }

    @Bean
    ApiMethodEndpointMapping apiMethodEndpointMapping() {
        ApiMethodEndpointMapping endpointMapping = new ApiMethodEndpointMapping();
        endpointMapping.setInterceptors(apiInterceptors());
        return endpointMapping;
    }

    @Override
    public void addArgumentResolvers(List<MethodArgumentResolver> argumentResolvers) {
        argumentResolvers.add(apiObjectRequestMethodArgumentResolver);
    }

    @Override
    public void addReturnValueHandlers(List<MethodReturnValueHandler> returnValueHandlers) {
        returnValueHandlers.add(apiObjectRequestMethodArgumentResolver);
    }

    @Bean
    public EndpointInterceptor[] apiInterceptors() {
        // выполняются интерсепторы в том же порядке, в котором они указаны
        return new EndpointInterceptor[]{
                apiNoEmptyErrorListsInterceptor,
                apiLoggingInterceptor,
                unitsInterceptor,
                authenticationInterceptor,
                // интерсепторы ниже должны использовать ApiAuthenticationSource для получения данных аутентификации
                apiLockInterceptor,
                serviceAuthInterceptor,
                rateLimitInterceptor,
                clientCurrencyConversionTeaserInterceptor,
                apiJvmSemaphoreInterceptor,
                // timeoutInterceptor должен быть последним,
                // так как может прервать по таймауту любую работу,
                // которая выполняется в рамках его жизненного цикла
                timeoutInterceptor
        };
    }

    @Bean(name = "api5Filter")
    @Autowired
    public Filter api5Filter(TraceContextFilter traceContextFilter,
                             ApiContextFilter apiContextFilter,
                             ApiLoggingFilter apiLoggingFilter,
                             AuthenticationFilter authenticationFilter,
                             AuthorizationFilter authorizationFilter,
                             UnitsFilter unitsFilter,
                             SetApiResponseToContextFilter setApiResponseToContextFilter) {
        CompositeFilter compositeFilter = new CompositeFilter();
        /*
        Порядок важен:
        - trace должен засечь время как можно раньше
        - ApiContext должен быть доступен из ApiLoggingFilter
         */
        compositeFilter.setFilters(Arrays.asList(
                traceContextFilter,
                apiContextFilter,
                setApiResponseToContextFilter,
                apiLoggingFilter,
                authenticationFilter,
                authorizationFilter,
                unitsFilter
        ));
        return compositeFilter;
    }

    @Bean
    @Autowired
    @Order(1) // json запросы приходят чаще, поэтому ставим это правило первым
    public HandlerAdapterDelegateRule jsonApiWsHandlerAdapter(JsonMessageFactory jsonMessageFactory,
                                                              ApiExceptionResolver apiExceptionResolver,
                                                              ApiUnitsService apiUnitsService) {
        return new HandlerAdapterDelegateRule(
                "/json/v5/", new ApiWsHandlerAdapter(jsonMessageFactory, apiExceptionResolver, apiUnitsService));
    }

    @Bean
    @Autowired
    @Order(2)
    public HandlerAdapterDelegateRule soapApiWsHandlerAdapter(SoapMessageFactory soapMessageFactory,
                                                              ApiExceptionResolver apiExceptionResolver,
                                                              ApiUnitsService apiUnitsService) {
        return new HandlerAdapterDelegateRule(
                "/v5/", new ApiWsHandlerAdapter(soapMessageFactory, apiExceptionResolver, apiUnitsService));
    }

    @Bean(name = MessageDispatcherServlet.DEFAULT_MESSAGE_RECEIVER_BEAN_NAME)
    public MessageDispatcher messageReceiver() {
        SoapMessageDispatcher messageReceiver = new SoapMessageDispatcher();
        messageReceiver.setEndpointMappings(Collections.singletonList(
                apiMethodEndpointMapping()
        ));
        return messageReceiver;
    }

    @Bean
    @RequestScope
    public GrutContext grutContext(GrutClient grutClient) {
        return new RequestScopeGrutContext(grutClient);
    }
}
