package ru.yandex.direct.api.v5.context;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.servlet.http.HttpServletResponse;

import org.springframework.util.Assert;

import ru.yandex.direct.api.v5.context.units.UnitsContext;
import ru.yandex.direct.api.v5.logging.ApiLogRecord;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.security.DirectApiAuthentication;
import ru.yandex.direct.api.v5.security.DirectApiPreAuthentication;
import ru.yandex.direct.api.v5.security.internal.DirectApiInternalAuthRequest;
import ru.yandex.direct.core.TranslatableException;
import ru.yandex.direct.core.units.OperationCosts;

import static com.google.common.base.Preconditions.checkState;

/**
 * Типизированный контекст текущего запроса в API
 * Сюда можно добавлять объекты, которые иначе пришлось бы пробрасывать через MessageContext/RequestContext
 * Доступен через {@link ApiContextHolder}, создаётся через {@link ApiContextFilter}
 */
public class ApiContext {
    /**
     * Информация о запросе, которая в итоге будет сброшена в логи АПИ
     */
    private final ApiLogRecord apiLogRecord;

    /**
     * Надо ли за этот запрос списывать баллы. Флаг по умолчанию "да", но его можно
     * поменять при обработке запроса, например, в interceptor-е. После обработки
     * списыватель баллов ({@link ru.yandex.direct.api.v5.entity.GenericApiService#withdrawUnits})
     * его учтёт и не будет ничего делать, если здесь "нет".
     */
    private boolean shouldChargeUnitsForRequest = true;

    /**
     * Описание стоимостей для API-операции. Задаётся при обработке запроса
     */
    private OperationCosts operationCosts;

    /**
     * Объект содержащий информацию о балансах, с которых будут списаны баллы за запрос.
     */
    private UnitsContext unitsContext;

    /**
     * Аутентификация, запомненная фильтром, может быть null
     */
    private DirectApiPreAuthentication preAuthentication;

    /**
     * Исключение, возникшее при авторизации
     */
    private TranslatableException authorizationException;

    /**
     * HTTP ответ API. Храним здесь на случай, если нужно дозаписать что-то после того, как отработал сервис.
     */
    private HttpServletResponse apiResponse;

    /**
     * Код серверной ошибки. Передаётся через {@link ApiContext}, поскольку уровень обнаружения ошибки
     * может не совпадать с уровнем, на котором код потребуется.
     */
    private int appErrorCode;

    /**
     * Информация об авторизации, полученная из токена
     */
    private DirectApiInternalAuthRequest authRequest;

    public ApiContext() {
        this.apiLogRecord = new ApiLogRecord();
    }

    @Nonnull
    public ApiLogRecord getApiLogRecord() {
        return apiLogRecord;
    }

    /**
     * @return ранее сохранённое в {@link ApiContext} значение {@link OperationCosts}
     */
    @Nullable
    public OperationCosts getOperationCosts() {
        return operationCosts;
    }

    /**
     * Сохранить {@link OperationCosts} &ndash; описание стоимостей текущей операции, необходимое для расчёта баллов
     *
     * @param operationCosts {@link OperationCosts} описание стоимостей текущей операции
     * @throws IllegalStateException если {@code operationCosts} уже задан в текущем {@link ApiContext}
     */
    public void setOperationCosts(OperationCosts operationCosts) {
        Assert.state(this.operationCosts == null,
                "Only one OperationSummary can be set to ApiContext during single request");
        this.operationCosts = operationCosts;
    }

    public UnitsContext getUnitsContext() {
        return unitsContext;
    }

    public void setUnitsContext(UnitsContext unitsContext) {
        this.unitsContext = unitsContext;
    }

    /**
     * Получить объект предварительной аутентификации.
     * <p>
     * Может не содержать целевого пользователя, для которого выполняется запрос, если его не удалось определить.
     * Нужен в случае, если необходимо получить данные об аутентификации до окончательной аутентификации/авторизации,
     * например, в фильтрах, для логирования и т.п.
     * <p>
     * Если же нужны окончательные данные об аутентификации, то следует использовать {@link ApiAuthenticationSource},
     * где они становятся достпными после отработки {@link ru.yandex.direct.api.v5.security.AuthenticationInterceptor}
     * <p>
     * В прикладном коде следует использовать {@link ApiAuthenticationSource}
     */
    @Nullable
    public DirectApiPreAuthentication getPreAuthentication() {
        return preAuthentication;
    }

    public void setPreAuthentication(DirectApiPreAuthentication preAuthentication) {
        this.preAuthentication = preAuthentication;
    }

    @Nullable
    public TranslatableException getAuthorizationException() {
        return authorizationException;
    }

    public void setAuthorizationException(TranslatableException authorizationException) {
        this.authorizationException = authorizationException;
    }

    /**
     * Получить объект, содержащий данные о полностью аутентифицированном запросе или сохранённую ошибку аутентификации.
     * Внимание! В прикладном коде следует использовать {@link ApiAuthenticationSource}
     */
    public DirectApiAuthentication getDirectApiAuthenticationOrFail() {
        if (authorizationException != null) {
            throw authorizationException;
        }
        checkState(preAuthentication != null && preAuthentication.isFullyAuthorized(),
                "either preAuthentication or authenticationException "
                        + "must be defined in ApiContext");
        return preAuthentication.toDirectApiAuthentication();
    }

    public boolean shouldChargeUnitsForRequest() {
        return shouldChargeUnitsForRequest;
    }

    public void setShouldChargeUnitsForRequest(boolean shouldChargeUnitsForRequest) {
        this.shouldChargeUnitsForRequest = shouldChargeUnitsForRequest;
    }

    public HttpServletResponse getApiResponse() {
        return apiResponse;
    }

    public void setApiResponse(HttpServletResponse apiResponse) {
        this.apiResponse = apiResponse;
    }

    public int getAppErrorCode() {
        return appErrorCode;
    }

    public void setAppErrorCode(int appErrorCode) {
        this.appErrorCode = appErrorCode;
    }

    public DirectApiInternalAuthRequest getAuthRequest() {
        return authRequest;
    }

    public void setAuthRequest(DirectApiInternalAuthRequest authRequest) {
        this.authRequest = authRequest;
    }
}
