package ru.yandex.direct.api.v5.context;

import java.util.Optional;

import javax.annotation.Nonnull;

import org.springframework.stereotype.Component;

import static com.google.common.base.Preconditions.checkNotNull;
import static org.springframework.web.context.request.RequestAttributes.SCOPE_REQUEST;
import static org.springframework.web.context.request.RequestContextHolder.currentRequestAttributes;

/**
 * Утилитный класс для получения {@link ApiContext}
 * Создавать {@link ApiContext} должен {@link ApiContextFilter}
 * По сути, у него статические методы, но в тестах их было бы нельзя мокать
 */
@Component
public class ApiContextHolder {
    private static final String REQUEST_ATTRIBUTE_KEY = ApiContextHolder.class.getName() + ".CONTEXT";

    /**
     * Получение объекта {@link ApiContext}, относящегося к текущему запросу,
     *
     * @throws IllegalStateException если контекст на был ранее сохранён
     */
    @Nonnull
    public ApiContext get() {
        return Optional
                .ofNullable((ApiContext) currentRequestAttributes().getAttribute(REQUEST_ATTRIBUTE_KEY, SCOPE_REQUEST))
                .orElseThrow(() -> new IllegalStateException("ApiContext is not initialized yet"));
    }

    void set(@Nonnull ApiContext ctx) {
        checkNotNull(ctx);

        currentRequestAttributes().setAttribute(REQUEST_ATTRIBUTE_KEY, ctx, SCOPE_REQUEST);
    }
}
