package ru.yandex.direct.api.v5.converter;

import java.util.function.Function;

import javax.xml.bind.JAXBElement;

import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;

public class ModelChangesHelper {
    private ModelChangesHelper() {
    }

    /**
     * Добавить в ModelChanges изменение, описанное в JaxbElement.
     * <p>
     * Значение в JaxbElement может быть в 3-х состояниях.
     * 1. Отсутствовать -- это значит изменений нет
     * 2. Быть nil -- целевое значение нужно сделать равным null
     * 3. Присутствовать -- целевое значение нужно сделать равным указанному
     *
     * @param changes       набор изменений, в которое нужно добавить изменение для указанного свойства
     * @param jaxbElement   JaxbElement, в котором предана информация об измнении свойства
     * @param modelProperty изменяемое свойство
     * @param <M>           изменяемая модель
     * @param <T>           тип значения в свойстве и в JaxbElement'е
     */
    public static <M extends ModelWithId, T> void processJaxbElement(
            ModelChanges<M> changes,
            JAXBElement<T> jaxbElement,
            ModelProperty<? super M, T> modelProperty) {
        if (jaxbElement != null) {
            if (jaxbElement.isNil()) {
                changes.process(null, modelProperty);
            } else {
                changes.process(jaxbElement.getValue(), modelProperty);
            }
        }
    }

    /**
     * Добавить в ModelChanges изменение, описанное в JaxbElement, предварительно его сконвертировав
     * <p>
     * Значение в JaxbElement может быть в 3-х состояниях.
     * 1. Отсутствовать -- это значит изменений нет
     * 2. Быть nil -- целевое значение нужно сделать равным null
     * 3. Присутствовать -- целевое значение нужно сделать равным указанному
     *
     * @param changes       набор изменений, в которое нужно добавить изменение для указанного свойства
     * @param jaxbElement   JaxbElement, в котором предана информация об измнении свойства
     * @param modelProperty изменяемое свойство
     * @param convert       функция конвертации из значения в JaxbElement в значение свойства
     * @param <M>           изменяемая модель
     * @param <I>           тип значения в JaxbElement'е
     * @param <T>           тип значения в свойстве
     */
    public static <M extends ModelWithId, I, T> void processJaxbElement(
            ModelChanges<M> changes,
            JAXBElement<I> jaxbElement,
            ModelProperty<? super M, T> modelProperty,
            Function<I, T> convert) {
        if (jaxbElement != null) {
            if (jaxbElement.isNil()) {
                changes.process(null, modelProperty);
            } else {
                changes.process(convert.apply(jaxbElement.getValue()), modelProperty);
            }
        }
    }

    /**
     * Добавить в ModelChanges изменение, описанное в JaxbElement, предварительно его сконвертировав
     * <p>
     * Значение в JaxbElement может быть в 3-х состояниях.
     * 1. Отсутствовать -- это значит изменений нет
     * 2. Быть nil -- целевое значение нужно сделать равным defaultValueForNil
     * 3. Присутствовать -- целевое значение нужно сделать равным указанному
     *
     * @param changes            набор изменений, в которое нужно добавить изменение для указанного свойства
     * @param jaxbElement        JaxbElement, в котором предана информация об измнении свойства
     * @param modelProperty      изменяемое свойство
     * @param convert            функция конвертации из значения в JaxbElement в значение свойства
     * @param defaultValueForNil дефолтное значение для конвертации, если пришедший элемент nil
     * @param <M>                изменяемая модель
     * @param <I>                тип значения в JaxbElement'е
     * @param <T>                тип значения в свойстве
     */
    public static <M extends ModelWithId, I, T> void processJaxbElement(
            ModelChanges<M> changes,
            JAXBElement<I> jaxbElement,
            ModelProperty<? super M, T> modelProperty,
            Function<I, T> convert,
            T defaultValueForNil) {
        if (jaxbElement != null) {
            if (jaxbElement.isNil()) {
                changes.process(defaultValueForNil, modelProperty);
            } else {
                changes.process(convert.apply(jaxbElement.getValue()), modelProperty);
            }
        }
    }
}
