package ru.yandex.direct.api.v5.converter;

import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.advideos.VideoActionResult;
import com.yandex.direct.api.v5.bids.BidActionResult;
import com.yandex.direct.api.v5.general.ActionResult;
import com.yandex.direct.api.v5.general.ActionResultBase;
import com.yandex.direct.api.v5.general.ClientsActionResult;
import com.yandex.direct.api.v5.general.ExceptionNotification;
import com.yandex.direct.api.v5.general.MultiIdsActionResult;
import com.yandex.direct.api.v5.general.SetBidsActionResult;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.common.ResultConverterException;
import ru.yandex.direct.api.v5.common.validation.ApiDefectPresentation;
import ru.yandex.direct.api.v5.common.validation.DefectPresentationService;
import ru.yandex.direct.api.v5.common.validation.DefectPresentationsHolder;
import ru.yandex.direct.api.v5.common.validation.DefectPresentationsHolderRespectingPath;
import ru.yandex.direct.api.v5.result.ApiMassResult;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.result.ApiResultState;
import ru.yandex.direct.api.v5.validation.ApiDefect;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.bids.container.SetBidItem;
import ru.yandex.direct.core.entity.creative.service.CreativeService;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.result.ResultState;
import ru.yandex.direct.utils.converter.Converter;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.PathConverter;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class ResultConverter {
    private static final Logger logger = LoggerFactory.getLogger(ResultConverter.class);
    private final TranslationService translationService;
    private final DefectPresentationService defectPresentationService;

    @Autowired
    public ResultConverter(TranslationService translationService,
                           DefectPresentationService defectPresentationService) {
        this.translationService = translationService;
        this.defectPresentationService = defectPresentationService;
    }

    public <T> ApiResult<T> toApiResult(Result<T> coreResult) {
        return toApiResult(coreResult, null);
    }

    public <T> ApiResult<T> toApiResult(Result<T> coreResult,
                                        @Nullable DefectPresentationsHolder overridingPresentations) {
        return new ApiResult<>(coreResult.getResult(),
                mapList(coreResult.getErrors(), t -> toDefectInfo(t, overridingPresentations)),
                mapList(coreResult.getWarnings(), t -> toDefectInfo(t, overridingPresentations)),
                convertState(coreResult.getState()));
    }

    public <T> ApiResult<T> toApiResultRespectingPath(
            Result<T> coreResult,
            @Nullable DefectPresentationsHolderRespectingPath overridingPresentations) {
        return new ApiResult<>(coreResult.getResult(),
                mapList(coreResult.getErrors(), t -> toDefectInfoRespectingPath(t, overridingPresentations)),
                mapList(coreResult.getWarnings(), t -> toDefectInfoRespectingPath(t, overridingPresentations)),
                convertState(coreResult.getState()));
    }

    public <T> ApiMassResult<T> toApiMassResult(MassResult<T> coreMassResult) {
        return toApiMassResult(coreMassResult, null);
    }

    public <T> ApiMassResult<T> toApiMassResult(MassResult<T> coreMassResult,
                                                @Nullable DefectPresentationsHolder overridingPresentations) {
        List<ApiResult<T>> result = mapList(coreMassResult.getResult(), r -> toApiResult(r, overridingPresentations));

        ApiMassResult<T> apiMassResult = new ApiMassResult<>(result,
                mapList(coreMassResult.getErrors(), t -> toDefectInfo(t, overridingPresentations)),
                mapList(coreMassResult.getWarnings(), t -> toDefectInfo(t, overridingPresentations)),
                convertState(coreMassResult.getState()));
        apiMassResult.withOperationMeta(coreMassResult.getOperationMeta().orElse(null));

        return apiMassResult.withCounts(coreMassResult.getSuccessfulCount(), coreMassResult.getErrorCount());
    }

    public <T> ApiMassResult<T> toApiMassResultRespectingPath(
            MassResult<T> coreMassResult,
            @Nullable DefectPresentationsHolderRespectingPath overridingPresentations) {
        List<ApiResult<T>> result = mapList(coreMassResult.getResult(),
                r -> toApiResultRespectingPath(r, overridingPresentations));

        ApiMassResult<T> apiMassResult = new ApiMassResult<>(result,
                mapList(coreMassResult.getErrors(), t -> toDefectInfoRespectingPath(t, overridingPresentations)),
                mapList(coreMassResult.getWarnings(), t -> toDefectInfoRespectingPath(t, overridingPresentations)),
                convertState(coreMassResult.getState()));
        apiMassResult.withOperationMeta(coreMassResult.getOperationMeta().orElse(null));

        return apiMassResult.withCounts(coreMassResult.getSuccessfulCount(), coreMassResult.getErrorCount());
    }

    private ApiResultState convertState(ResultState state) {
        switch (state) {
            case SUCCESSFUL:
                return ApiResultState.SUCCESSFUL;
            case BROKEN:
                return ApiResultState.BROKEN;
            default:
                throw new IllegalStateException("Unexpected result state");
        }
    }

    public <T> ValidationResult<T, DefectType> convertValidationResult(
            ValidationResult<T, Defect> validationResult,
            @Nullable DefectPresentationsHolder overridingPresentations) {
        ValidationResult<T, DefectType> newResult = new ValidationResult<>(validationResult.getValue());
        convertAndTransferValidationTree(newResult, validationResult, overridingPresentations);
        return newResult;
    }

    public <T> ValidationResult<T, DefectType> convertValidationResult(
            ValidationResult<T, Defect> validationResult) {
        ValidationResult<T, DefectType> newResult = new ValidationResult<>(validationResult.getValue());
        convertAndTransferValidationTree(newResult, validationResult, null);
        return newResult;
    }

    private <T> void convertAndTransferValidationTree(ValidationResult<T, DefectType> destinationVr,
                                                      ValidationResult<?, Defect> sourceVr,
                                                      @Nullable DefectPresentationsHolder overridingPresentations) {
        destinationVr.getErrors().addAll(toDefectTypeList(sourceVr.getErrors(), overridingPresentations));
        destinationVr.getWarnings().addAll(toDefectTypeList(sourceVr.getWarnings(), overridingPresentations));
        sourceVr.getSubResults().forEach((pathNode, subResult) -> {
            ValidationResult<?, DefectType> newSubResult =
                    destinationVr.getOrCreateSubValidationResult(pathNode, subResult.getValue());
            convertAndTransferValidationTree(newSubResult, subResult, overridingPresentations);
        });
    }

    public DefectInfo<DefectType> toDefectInfo(DefectInfo<Defect> defectInfo,
                                               @Nullable DefectPresentationsHolder overridingPresentations) {
        //noinspection unchecked
        return new DefectInfo<DefectType>(defectInfo.getPath(),
                defectInfo.getValue(),
                toDefectType(defectInfo.getDefect(), overridingPresentations));
    }

    public DefectInfo<DefectType> toDefectInfoRespectingPath(
            DefectInfo<Defect> defectInfo,
            @Nullable DefectPresentationsHolderRespectingPath overridingPresentations) {
        //noinspection unchecked
        return new DefectInfo<DefectType>(defectInfo.getPath(),
                defectInfo.getValue(),
                toDefectTypeRespectingPath(defectInfo.getDefect(),
                        overridingPresentations,
                        defectInfo.getPath()));
    }


    private List<DefectType> toDefectTypeList(List<Defect> defects,
                                              @Nullable DefectPresentationsHolder overridingPresentations) {
        //noinspection unchecked
        return StreamEx.of(defects)
                .map(t -> toDefectType(t, overridingPresentations))
                .toList();
    }

    private <P> DefectType toDefectType(Defect<P> defect,
                                        @Nullable DefectPresentationsHolder overridingPresentations) {
        ApiDefectPresentation presentation =
                defectPresentationService.getPresentationFor(defect.defectId(), overridingPresentations);
        //noinspection unchecked
        return presentation.toDefectType(defect.params());
    }

    private <P> DefectType toDefectTypeRespectingPath(
            Defect<P> defect,
            @Nullable DefectPresentationsHolderRespectingPath overridingPresentations,
            Path path) {
        ApiDefectPresentation presentation =
                defectPresentationService.getPresentationFor(defect.defectId(), overridingPresentations, path);
        //noinspection unchecked
        return presentation.toDefectType(defect.params());
    }

    public ActionResult convertToActionResult(ApiResult<Long> source, PathConverter apiPathConverter) {
        return toActionResult(source, apiPathConverter, ActionResult::new, ActionResult::setId);
    }

    public List<ActionResult> toActionResults(ApiResult<? extends Collection<ApiResult<Long>>> results,
                                              PathConverter apiPathConverter) {
        return results.getResult()
                .stream()
                .map(r -> convertToActionResult(r, apiPathConverter))
                .collect(toList());
    }

    public BidActionResult convertToBidActionResult(ApiResult<SetBidItem> result,
                                                    PathConverter apiPathConverter) {
        return toBidActionResult(convertToSetBidsActionResult(result, apiPathConverter));
    }

    public SetBidsActionResult convertToSetBidsActionResult(ApiResult<SetBidItem> result,
                                                            PathConverter apiPathConverter) {
        return toActionResult(result, apiPathConverter, SetBidsActionResult::new,
                (actionResult, bid) -> actionResult.withId(bid.getId())
                        .withAdGroupId(bid.getAdGroupId())
                        .withCampaignId(bid.getCampaignId()));
    }

    /**
     * Возвращает {@link ActionResultBase} с ошибками и предупреждениями из {@code source}.
     * <p>
     * Метод оказывается полезен, когда необходимо сконвертировать ошибки и предупреждения в формат ответа API,
     * но при этом целевой тип результата отличен от {@link ActionResult}.
     */
    public <T> ActionResultBase convertToGenericActionResult(ApiResult<T> source, PathConverter pathConverter) {
        return toActionResult(source, pathConverter, ActionResultBase::new, null);
    }

    private BidActionResult toBidActionResult(SetBidsActionResult setBidsActionResult) {
        BidActionResult bidActionResult = new BidActionResult()
                .withKeywordId(setBidsActionResult.getId())
                .withCampaignId(setBidsActionResult.getCampaignId())
                .withAdGroupId(setBidsActionResult.getAdGroupId());
        // getErrors() и getWarnings() под капотом лениво инициализируют списки ошибок и предупреждений
        // проверяем списки на наличие элементов, чтобы не отдавать "Errors" и "Warnings", если ошибок не было
        if (!setBidsActionResult.getErrors().isEmpty()) {
            bidActionResult.setErrors(setBidsActionResult.getErrors());
        }
        if (!setBidsActionResult.getWarnings().isEmpty()) {
            bidActionResult.setWarnings(setBidsActionResult.getWarnings());
        }
        return bidActionResult;
    }

    public VideoActionResult convertToAdVideoActionResult(ApiResult<CreativeService.VideoItem> result,
                                                          PathConverter apiPathConverter) {
        return toActionResult(result, apiPathConverter, VideoActionResult::new,
                (actionResult, videoItem) -> actionResult.withId(videoItem.getExternalVideoId()));
    }

    /**
     * Конвертирует ответ сервиса из внутренних сущностей в ClientsActionResult для сервисов по работе с клиентами
     *
     * @param result           результат выполнения операции внутренним сервисом
     * @param apiPathConverter конвертер путей
     * @return ответ во внешнем представлении
     */
    public ClientsActionResult convertToClientsActionResult(ApiResult<Long> result,
                                                            PathConverter apiPathConverter) {
        return toActionResult(result, apiPathConverter, ClientsActionResult::new,
                ClientsActionResult::withClientId);
    }

    /**
     * Преобразовать результат выполнения {@link ApiResult} в результат,
     * выдваваемый в API (подклассы {@link ActionResultBase}), попутно конвертируя пути ошибок.
     *
     * @param source               результат, который нужно преобразовать
     * @param pathConverter        конвертор путей, который нужно применить к ошибка и предупреждениям в результате
     * @param actionResultSupplier создватель API результата
     * @param filler               заполнятель API результата содержимым результата выполнения
     *                             {@link ApiResult#getResult()}
     * @param <T>                  тип содержимого результата выполнения
     * @param <R>                  тип подкласса {@link ActionResultBase} -- результата, выдаваемого в API
     * @return результат, для выдачи в API
     */
    public <T, R extends ActionResultBase> R toActionResult(
            ApiResult<T> source, PathConverter pathConverter,
            Supplier<R> actionResultSupplier, @Nullable BiConsumer<R, T> filler) {
        R actionResult = actionResultSupplier.get();
        if (source.isSuccessful()) {
            if (filler != null) {
                filler.accept(actionResult, source.getResult());
            }
        } else {
            addErrors(actionResult, source.getErrors(), pathConverter);
        }
        addWarnings(actionResult, source.getWarnings(), pathConverter);

        return actionResult;
    }

    public void addErrors(ActionResultBase actionResult, Collection<DefectInfo<DefectType>> errors,
                          PathConverter apiPathConverter) {
        if (!errors.isEmpty()) {
            actionResult.withErrors(convertDefectInfosToExceptionNotifications(errors, apiPathConverter));
        }
    }

    public void addWarnings(ActionResultBase actionResult, Collection<DefectInfo<DefectType>> warnings,
                            PathConverter apiPathConverter) {
        if (!warnings.isEmpty()) {
            actionResult.withWarnings(convertDefectInfosToExceptionNotifications(warnings, apiPathConverter));
        }
    }

    private Collection<ExceptionNotification> convertDefectInfosToExceptionNotifications(
            Collection<DefectInfo<DefectType>> results, PathConverter apiPathConverter)
            throws ResultConverterException {
        try {
            return results.stream()
                    .map(r -> r.convertPath(apiPathConverter))
                    .map(ApiDefect::new)
                    .map(this::defectToNotification)
                    .collect(toList());
        } catch (IllegalStateException e) {
            logger.error("Error in path conversion", e);
            throw new ResultConverterException(e);
        }
    }

    private ExceptionNotification defectToNotification(ApiDefect apiDefect) {
        return new ExceptionNotification()
                .withCode(apiDefect.getCode())
                .withMessage(translationService.translate(apiDefect.getShortMessage()))
                .withDetails(
                        Optional.ofNullable(apiDefect.getDetailedMessage())
                                .map(translationService::translate)
                                .orElse(null));
    }

    public List<MultiIdsActionResult> convertToMultiIdsActionResults(
            ApiResult<List<ApiResult<List<Long>>>> results,
            PathConverter apiPathConverter) {
        return results.getResult()
                .stream()
                .map(r -> convertToMultiIdsActionResult(r, apiPathConverter))
                .collect(toList());

    }

    private MultiIdsActionResult convertToMultiIdsActionResult(ApiResult<List<Long>> source,
                                                               PathConverter apiPathConverter) {
        return toActionResult(source, apiPathConverter, MultiIdsActionResult::new,
                MultiIdsActionResult::setIds);
    }

    public static <T1, T2> Converter<ApiMassResult<T1>, ApiMassResult<T2>> apiMassResultConverter(
            Converter<ApiResult<T1>, ApiResult<T2>> resultConverter) {
        return source -> new ApiMassResult<>(
                resultConverter.convertList(source.getResult()),
                source.getErrors(),
                source.getWarnings(),
                source.getState()
        );
    }
}
