package ru.yandex.direct.api.v5.entity;

import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;

import ru.yandex.direct.api.v5.entity.ads.validation.AdTypeNames;
import ru.yandex.direct.api.v5.entity.bidmodifiers.Constants;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.ApiDefect;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.TranslatableException;
import ru.yandex.direct.core.entity.adgroup.container.AccessibleAdGroupTypes;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignAccessibiltyChecker;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.api5.Api5CampaignAccessibilityChecker;
import ru.yandex.direct.core.units.OperationSummary;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.direct.validation.result.PathConverter;
import ru.yandex.direct.validation.result.ValidationResult;

public abstract class BaseApiServiceDelegate<Req, Resp, IntReq, IntResp> {

    protected final PathConverter apiPathConverter;
    protected final ApiAuthenticationSource auth;

    public BaseApiServiceDelegate(PathConverter apiPathConverter,
                                  ApiAuthenticationSource auth) {
        this.apiPathConverter = apiPathConverter;
        this.auth = auth;
    }

    /**
     * Валидирует запрос на предмет критичных ошибок, не позволяющих выполнить запрос
     *
     * @param externalRequest запрос
     * @return результат валидации запроса; null, если валидация не выполнялась
     */
    @Nullable
    public ValidationResult<Req, DefectType> validateRequest(Req externalRequest) {
        return null;
    }

    /**
     * Преобразовывает запрос из внешнего представления во внутреннее
     *
     * @param externalRequest запрос
     * @return внутреннее представление запроса для передачи в {@link #processRequest(Object)}
     */
    public abstract IntReq convertRequest(Req externalRequest);

    /**
     * Обрабатывает запрос, подготовленный с помощью {@link #convertRequest(Object)}
     *
     * @param internalRequest запрос
     * @return {@link ApiResult} с внутренним представлением результата выполнения запроса
     */
    public abstract ApiResult<List<IntResp>> processRequest(IntReq internalRequest);


    /**
     * Исправить информацию о текущей операции
     *
     * @param internalRequest запрос
     * @param apiResult       результат текущего запроса из {@link #processRequest(Object)}
     * @return исправленная информация о запросе
     * <p>
     * Например в API5:GetKeywordsDelegate возникла необходимость передачи флага specialProcessingObjectCost
     * на основании информации о запрашиваемых полях.
     */
    public OperationSummary correctOperationSummary(IntReq internalRequest, ApiResult<List<IntResp>> apiResult) {
        return apiResult.getOperationSummary();
    }

    /**
     * Преобразовывает результат во внешний ответ.
     *
     * @param result результат выполнения {@link #processRequest(Object)}
     * @return ответ во внешнем представлении
     */
    public abstract Resp convertResponse(ApiResult<List<IntResp>> result);

    public ApiAuthenticationSource getAuth() {
        return auth;
    }

    public TranslatableException createValidationException(ApiResult<?> result, boolean convertPath,
                                                           boolean isUnitsSpent) {
        DefectInfo<DefectType> criticalErrorInfo = result.getErrors().iterator().next();
        DefectInfo<DefectType> newCriticalErrorInfo = criticalErrorInfo.convertPath(
                convertPath ? apiPathConverter : PathConverter.identity());
        ApiDefect criticalError = new ApiDefect(newCriticalErrorInfo);
        return new ApiValidationException(criticalError, isUnitsSpent);
    }

    public CampaignAccessibiltyChecker getCampaignAccessibiltyChecker() {
        return Api5CampaignAccessibilityChecker.getApi5AccessibilityChecker(auth.getRequestSource());
    }

    public Set<AdGroupType> getAllowedAdGroupTypes() {
        return AccessibleAdGroupTypes.API5_ALLOWED_AD_GROUP_TYPES;
    }

    public String getAllowedAdTypes() {
        return AdTypeNames.getApi5AllowedAdTypes();
    }

    public Set<BidModifierType> getAllowedBidModifierTypes() {
        return Constants.getApi5AllowedBidModifierTypes();
    }
}
