package ru.yandex.direct.api.v5.entity;

import java.util.List;

import com.yandex.direct.api.v5.general.ApiExceptionMessage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.api.v5.context.ApiContext;
import ru.yandex.direct.api.v5.context.ApiContextHolder;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.service.accelinfo.AccelInfoHeaderSetter;
import ru.yandex.direct.api.v5.units.ApiUnitsService;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.api.v5.ws.exceptionresolver.ApiExceptionResolver;
import ru.yandex.direct.core.TranslatableException;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.RequestCampaignAccessibilityCheckerProvider;
import ru.yandex.direct.core.units.OperationSummary;
import ru.yandex.direct.validation.result.ValidationResult;

/**
 * Большинство операций API имеют одинаковый workflow. Отдельные шаги этого процесса в точности повторяются разными
 * компонентами. Введение данного сервиса позволяет достигнуть двух целей. Во-первых, в него был вынесен дублирующийся
 * в компонентах код. Во-вторых, будущие изменения, требующие работы с внутренним представлением {@link ApiResult},
 * будет удобно реализовывать в этом сервисе.
 *
 * @see BaseApiServiceDelegate
 */
@Service
public class GenericApiService {
    private static final Logger logger = LoggerFactory.getLogger(GenericApiService.class);

    private final ApiContextHolder apiContextHolder;
    private final ApiUnitsService apiUnitsService;
    private final AccelInfoHeaderSetter accelInfoHeaderSetter;
    private final RequestCampaignAccessibilityCheckerProvider requestAccessibleCampaignTypes;

    @Autowired
    public GenericApiService(ApiContextHolder apiContextHolder,
                             ApiUnitsService apiUnitsService,
                             AccelInfoHeaderSetter accelInfoHeaderSetter,
                             RequestCampaignAccessibilityCheckerProvider requestAccessibleCampaignTypes) {
        this.apiContextHolder = apiContextHolder;
        this.apiUnitsService = apiUnitsService;
        this.accelInfoHeaderSetter = accelInfoHeaderSetter;
        this.requestAccessibleCampaignTypes = requestAccessibleCampaignTypes;
    }

    /**
     * Вызывает методы переданного {@code internalDelegate}, попутно выполняя некоторую утилитарную работу, не связанную
     * напрямую с обработкой запроса {@code request}
     *
     * @param internalDelegate сервис, реализующий интерфейс {@link BaseApiServiceDelegate}
     * @param request          запрос для обработки по шагам интерфейса {@link BaseApiServiceDelegate}
     * @return Response во внешнем формате
     */
    public final <Req, Resp, IntReq, IntResp> Resp doAction(
            BaseApiServiceDelegate<Req, Resp, IntReq, IntResp> internalDelegate, Req request) {
        requestAccessibleCampaignTypes.setCustom(internalDelegate.getCampaignAccessibiltyChecker());

        ValidationResult<Req, DefectType> vr = internalDelegate.validateRequest(request);

        ApiResult<List<IntResp>> result;
        IntReq internalRequest = null;
        boolean convertPath = true;
        if (vr != null && vr.hasAnyErrors()) {
            result = ApiResult.broken(vr.flattenErrors(), vr.flattenWarnings());
            // ошибка произошла при валидации исходного запроса, значит конвертировать пути в ошибках не нужно
            convertPath = false;
        } else {
            internalRequest = internalDelegate.convertRequest(request);
            result = internalDelegate.processRequest(internalRequest);
        }

        OperationSummary operationSummary;
        if (internalRequest != null) {
            operationSummary = internalDelegate.correctOperationSummary(internalRequest, result);
        } else {
            operationSummary = result.getOperationSummary();
        }
        withdrawUnits(operationSummary);

        if (!result.isSuccessful()) {
            logger.debug("Can't process request due to operational level errors");
            TranslatableException validationException =
                    internalDelegate.createValidationException(result, convertPath, true);

            saveCodeAndRethrow(validationException);
        }

        accelInfoHeaderSetter.setAccelInfoHeaderToHttpResponse();

        return internalDelegate.convertResponse(result);
    }

    private void withdrawUnits(OperationSummary operationSummary) {
        if (!apiContextHolder.get().shouldChargeUnitsForRequest()) {
            return;
        }
        apiUnitsService.withdraw(operationSummary);
    }

    /**
     * Сохранить код ошибки в {@link ApiContext} на случай, если {@link ApiExceptionResolver} не сможет
     * преобразовать её в {@link ApiExceptionMessage}.
     *
     * @param operationalError ошибка валидации запроса, пришедшая из сервиса.
     */
    private void saveCodeAndRethrow(TranslatableException operationalError) {
        setErrorCodeToContext(operationalError.getCode());
        throw operationalError;
    }

    private void setErrorCodeToContext(int errorCode) {
        logger.debug("Setting application error code to API context: {}", errorCode);
        apiContextHolder.get().setAppErrorCode(errorCode);
    }

}
