package ru.yandex.direct.api.v5.entity;

import java.util.Collections;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.general.GetRequestGeneral;

import ru.yandex.direct.api.v5.common.ConverterUtils;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.result.ApiResultState;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.validation.result.PathConverter;

import static java.util.Collections.emptyList;

/**
 * Абстрактная реализация общего воркфлоу для get-запросов.
 * Наследуется от базового делегата {@link BaseApiServiceDelegate} и делает его воркфлоу
 * более адаптированным для реализации get-методов API, инкапсулирует
 * в себе инкрементирование лимита и вычисление limitedBy,
 * а так же делает конечные реализации get-методов более стандартизованными.
 * <p>
 * Воркфлоу состоит из следующих шагов, которые должны быть реализованы в потомках: <ul>
 * <li>валидация запроса уровня API: {@link BaseApiServiceDelegate#validateRequest(Object)}</li>
 * <li>извлечение списка требуемых в ответе полей: {@link #extractFieldNames(GetRequestGeneral)}</li>
 * <li>извлечение условия выборки: {@link #extractSelectionCriteria(GetRequestGeneral)}</li>
 * <li>выполнение непосредственно выборки: {@link #get(GenericGetRequest)}</li>
 * <li>конвертация полученных результатов во внутреннем формате в ответ уровня API:
 * {@link #convertGetResponse(List, Set, Long)}</li>
 * </ul>
 *
 * @param <Req>         тип запроса уровня API
 * @param <Resp>        тип ответа уровня API
 * @param <F>           тип перечисления требуемых полей
 * @param <S>           тип условия выборки (в простейшем случае - список id)
 * @param <IntRespItem> тип внутреннего представления выходного запроса
 */
@ParametersAreNonnullByDefault
public abstract class GetApiServiceDelegate<Req extends GetRequestGeneral, Resp, F extends Enum<F>, S, IntRespItem>
        extends BaseApiServiceDelegate<Req, Resp, GenericGetRequest<F, S>, IntRespItem> {

    public GetApiServiceDelegate(PathConverter apiPathConverter,
                                 ApiAuthenticationSource auth) {
        super(apiPathConverter, auth);
    }

    /**
     * Реализация convertRequest, адаптированная для get-сервисов.
     * Реализует извлечение limit и offset, а извлечение
     * набора полей и условия выборки должно реализовываться в потомках
     * (смотреть методы {@link #extractFieldNames(GetRequestGeneral)}
     * и {@link #extractSelectionCriteria(GetRequestGeneral)}
     *
     * @param externalRequest запрос уровня API
     * @return объект GenericGetRequest, содержащий извлеченные
     * limit/offset, список требуемых полей и условие выборки
     */
    @Override
    public final GenericGetRequest<F, S> convertRequest(Req externalRequest) {
        Set<F> requestedFieldNames = extractFieldNames(externalRequest);
        S selectionCriteria = extractSelectionCriteria(externalRequest);
        LimitOffset limitOffset = ConverterUtils.convertLimitOffset(externalRequest.getPage());
        return new GenericGetRequest<>(requestedFieldNames, selectionCriteria, limitOffset);
    }

    /**
     * Метод должен быть реализован в потомке для извлечения из запроса набора требуемых в ответе полей.
     *
     * @param externalRequest запрос уровня API
     * @return набор требуемых в ответе полей
     */
    public abstract Set<F> extractFieldNames(Req externalRequest);

    /**
     * Метод должен быть реализован в потомке для извлечения условия выборки.
     *
     * @param externalRequest запрос уровня API
     * @return условие выборки
     */
    public abstract S extractSelectionCriteria(Req externalRequest);

    /**
     * Реализация обработки запроса, адаптированная для get-сервисов.
     * <p>
     * Вызывает метод {@link #get(GenericGetRequest)} потомка с лимитом, увеличенным на 1.
     * Далее на основе количества выбранных результатов вычисляет значение
     * поля ответа {@code limitedBy}, а так же при необходимости обрезает лишний
     * выбранный элемент перед передачей в метод конвертации, реализованный
     * так же в потомке {@link #convertGetResponse(List, Set, Long)}.
     * <p>
     * Внимание: в {@link #get}-метод потомка передается лимит, увеличенный на 1.
     * Это нужно для вычисления поля ответа {@code limitedBy}. Если количество выбираемых элементов
     * можно посчитать до выборки или обращения во внешние сервисы и лишний элемент критичен,
     * можно сделать адаптированный для этого кейса делегат (можно обращаться к @mexicano).
     *
     * @param internalRequest запрос
     * @return результат во внутреннем формате, содержащий в себе помимо извлеченных методом
     * {@link #get} результатов дополнительную информацию, которую необходимо передать между
     * этим методом и методом {@link #convertResponse(ApiResult)}.
     */
    @Override
    public final GetResult<F, IntRespItem> processRequest(GenericGetRequest<F, S> internalRequest) {
        GenericGetRequest<F, S> getRequestWithIncrementedLimit =
                GenericGetRequest.copyAndIncrementLimit(internalRequest);
        List<IntRespItem> fetchedItems = get(getRequestWithIncrementedLimit);

        Integer limitedBy = calcLimitedBy(internalRequest.getLimitOffset(), fetchedItems);
        List<IntRespItem> cutFetchedItems = limitedBy != null ?
                fetchedItems.subList(0, internalRequest.getLimitOffset().limit()) :
                fetchedItems;

        executeAdditionalActionsOverFetchedItems(cutFetchedItems);

        return new GetResult<>(cutFetchedItems, internalRequest.getRequestedFields(), limitedBy);
    }

    /**
     * Метод должен быть реализован в потомке для выборки элементов,
     * которые в дальнейшем конвертируются в ответ API.
     * <p>
     * Внимание: в метод передается лимит, увеличенный на 1.
     * Это нужно для вычисления поля ответа {@code limitedBy}. Если количество выбираемых элементов
     * можно посчитать до выборки или обращения во внешние сервисы и лишний элемент критичен,
     * можно сделать адаптированный для этого кейса делегат (можно обращаться к @mexicano).
     *
     * @param getRequest объект, представляющий собой запрос, он содержит
     *                   набор требуемых в ответе полей, условие выборки, limit/offset
     * @return список выбранных объектов
     */
    public abstract List<IntRespItem> get(GenericGetRequest<F, S> getRequest);

    /**
     * Метод может быть реализован в потомке для выполнения некоторых дополнительных
     * действий над полученными результатами.
     * <p>
     * В данный метод всегда передается список результатов,
     * размер которого не превышает указанный в запросе лимит
     * (смотреть javadoc к методам {@link #processRequest(GenericGetRequest)}
     * и {@link #get(GenericGetRequest)}. Если размер списка результатов,
     * выбранных методом {@link #get} больше, чем указанный в запросе limit,
     * то список результатов обрезается и только потом передается в данный метод.
     *
     * @param fetchedItems список выбранных элементов.
     */
    public void executeAdditionalActionsOverFetchedItems(@SuppressWarnings("unused") List<IntRespItem> fetchedItems) {
    }

    /**
     * Метод реализует специфичную для get-сервисов конвертацию результатов.
     * Внутри непосредственно конвертация делегируется методу потомка
     * {@link #convertGetResponse(List, Set, Long)}, в который помимо
     * списка результатов передается набор требуемых полей и значение {@code limitedBy}.
     *
     * @param result результат выполнения {@link #processRequest(GenericGetRequest)}
     * @return ответ уровня API.
     */
    @Override
    public final Resp convertResponse(ApiResult<List<IntRespItem>> result) {
        GetResult<F, IntRespItem> getResult = (GetResult<F, IntRespItem>) result;
        Long longLimitedBy = getResult.limitedBy != null ? Long.valueOf(getResult.limitedBy) : null;
        return convertGetResponse(result.getResult(), getResult.requestedFields, longLimitedBy);
    }

    /**
     * Метод должен быть реализован в потомке для конвертации выбранных элементов
     * в ответ уровня API.
     *
     * @param result          список выбранных элементов
     * @param requestedFields набор требуемых в ответе полей
     * @param limitedBy       значение поля ответа {@code limitedBy}
     * @return ответ уровня API.
     */
    public abstract Resp convertGetResponse(List<IntRespItem> result, Set<F> requestedFields, @Nullable Long limitedBy);

    /**
     * Возвращает список видимых юзеру ID кампаний, которые или элементы которых вернулись в ответе ядра.
     *
     * @param result ответа ядра
     * @return список ID кампаний
     */
    public List<Long> returnedCampaignIds(ApiResult<List<IntRespItem>> result) {
        return Collections.emptyList();
    }


    @Nullable
    private Integer calcLimitedBy(LimitOffset limitOffset, List<IntRespItem> retrievedItems) {
        return retrievedItems.size() > limitOffset.limit() ?
                limitOffset.offset() + limitOffset.limit() :
                null;
    }

    /**
     * Внутренний объект, необходимый для передачи дополнительной информации
     * между методами делегата {@link #processRequest(GenericGetRequest)} )} и
     * {@link #convertResponse(ApiResult)}.
     *
     * @param <F> тип перечисления для списка требуемых в ответе полей
     * @param <I> тип выбираемого элемента
     */
    private static final class GetResult<F extends Enum<F>, I> extends ApiResult<List<I>> {

        private Set<F> requestedFields;
        private Integer limitedBy;

        public GetResult(List<I> result, Set<F> requestedFields, @Nullable Integer limitedBy) {
            super(result, emptyList(), emptyList(), ApiResultState.SUCCESSFUL);
            this.requestedFields = requestedFields;
            this.limitedBy = limitedBy;
        }
    }
}
