package ru.yandex.direct.api.v5.entity;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.StreamEx;

import ru.yandex.direct.api.v5.result.ApiMassResult;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.result.ApiResultState;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.units.OperationSummary;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.validation.result.PathHelper.index;

public class OperationOnListUtils {
    private OperationOnListUtils() {}

    /**
     * Метод, позволяющий не списывать баллы в случае возникнования ошибок с определёнными кодами.
     * Вызывается в переопределениях метода {@link BaseApiServiceDelegate#correctOperationSummary}.
     * @param errorCodes коды ошибок, баллы за которые не нужно списывать.
     */
    public static <El> OperationSummary notCountErrorsWithCodes(ApiResult<List<ApiResult<El>>> apiResultList,
                                                                Set<Integer> errorCodes) {
        if (!apiResultList.isSuccessful()) {
            return apiResultList.getOperationSummary();
        }
        int badStatusErrors = (int) StreamEx.of(apiResultList.getResult())
                .filter(result -> result.getState() == ApiResultState.BROKEN)
                .map(ApiResult::getErrors)
                .filter(defectInfos -> StreamEx.of(defectInfos)
                        .allMatch(defectInfo -> errorCodes.contains(defectInfo.getDefect().getCode())))
                .count();
        checkState(badStatusErrors >= 0 && badStatusErrors <= apiResultList.getUnsuccessfulObjectsCount(),
                "Количество отфильтрованных ошибок не может быть меньше нуля или " +
                        "больше, чем общее количество результатов с ошибками.");
        return OperationSummary.successful(apiResultList.getSuccessfulObjectsCount(),
                apiResultList.getUnsuccessfulObjectsCount() - badStatusErrors);
    }

    public static <IntReqEl, IntRespEl> List<ApiResult<IntRespEl>> unionValidateAndProcessResult(
            ValidationResult<List<IntReqEl>, DefectType> validationResult,
            ApiMassResult<IntRespEl> resultForValidItems,
            List<IntReqEl> internalRequest) {
        Iterator<ApiResult<IntRespEl>> resultIterator = resultForValidItems.getResult().iterator();
        List<ApiResult<IntRespEl>> results = new ArrayList<>(internalRequest.size());
        Map<PathNode, ValidationResult<?, DefectType>> elementVrMap = validationResult.getSubResults();
        for (int i = 0; i < internalRequest.size(); i++) {
            ValidationResult<?, DefectType> elementVr = elementVrMap.get(index(i));
            final ApiResult<IntRespEl> res;
            if (elementVr != null && elementVr.hasAnyErrors()) {
                res = ApiResult.broken(elementVr.flattenErrors(), elementVr.flattenWarnings());
            } else if (elementVr != null && elementVr.hasAnyWarnings()) {

                ApiResult<IntRespEl> next = resultIterator.next();

                List<DefectInfo<DefectType>> warnings = elementVr.flattenWarnings();
                warnings.addAll(next.getWarnings());

                List<DefectInfo<DefectType>> errors = next.getErrors();
                res = errors.isEmpty() ? ApiResult.successful(next.getResult(), warnings) :
                        ApiResult.broken(errors, warnings);
            } else {
                res = resultIterator.next();
            }
            results.add(res);
        }
        checkState(results.size() == internalRequest.size(),
                "кол-во объединённых результатов должно быть равно кол-ву исходных элементов");
        return results;
    }
}
