package ru.yandex.direct.api.v5.entity;

import java.util.List;

import javax.annotation.Nonnull;

import ru.yandex.direct.api.v5.result.ApiMassResult;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.result.ApiResultState;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.validation.result.PathConverter;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.api.v5.entity.OperationOnListUtils.unionValidateAndProcessResult;

/**
 * Делегат, внутреннее представление запроса в котором это объект со списком, а внутреннее представление ответа -- список
 * результатов {@link ApiResult}. Обычно подходит для методов add, update, delete, выполняющих некоторое действие
 * над набором входных элементов. Знание структуры внутренних запроса и ответа позволяет этому делегату
 * валидировать каждый элемент списка в запросе, и, в случае ошибки, выдавать результат для этого элемента без
 * вызова основной операции {@link #processRequestWithList(RequestWithList)}}.
 *
 * @param <Req>       внешенее представление запроса (из wsdl)
 * @param <Resp>      внешнее представление ответа (из wsdl)
 * @param <IntReqEl>  элемент списка во внутреннем представлении запроса
 * @param <IntReq>    внутреннее представление запроса со списком
 * @param <IntRespEl> параметр элемента {@link ApiResult} в результирующем списке
 */
public abstract class OperationOnRequestWithListDelegate<Req, Resp, IntReqEl, IntReq extends RequestWithList<IntReqEl>, IntRespEl>
        extends BaseApiServiceDelegate<Req, Resp, IntReq, ApiResult<IntRespEl>> {

    public OperationOnRequestWithListDelegate(PathConverter apiPathConverter,
                                              ApiAuthenticationSource apiAuthenticationSource) {
        super(apiPathConverter, apiAuthenticationSource);
    }

    /**
     * Валидация списка из внутреннего представления запроса.
     * Невалидные элементы не будут переданы в {@link #processRequestWithList(RequestWithList)}.
     * Если после валидации в списке не останется валидных элементов,
     * то {@link #processRequestWithList(RequestWithList)} вообще не будет выполнен.
     */
    @Nonnull
    public ValidationResult<List<IntReqEl>, DefectType> validateInternalElements(List<IntReqEl> requestElements) {
        return new ValidationResult<>(requestElements);
    }

    @Override
    public final ApiResult<List<ApiResult<IntRespEl>>> processRequest(IntReq internalRequest) {
        List<IntReqEl> requestItems = internalRequest.getItems();
        ValidationResult<List<IntReqEl>, DefectType> validationResult = validateInternalElements(requestItems);
        List<IntReqEl> validItems = ValidationResult.getValidItems(validationResult);
        int invalidsNum = requestItems.size() - validItems.size();
        checkState(invalidsNum >= 0);

        ApiMassResult<IntRespEl> resultForValidItems;
        if (!validItems.isEmpty()) {
            internalRequest.setItems(validItems);
            resultForValidItems = processRequestWithList(internalRequest);
        } else {
            // для пустого списка валидных элементов -- пустой результат
            resultForValidItems = new ApiMassResult<>(emptyList(), emptyList(), emptyList(), ApiResultState.SUCCESSFUL);
        }
        if (!resultForValidItems.isSuccessful()) {
            // ошибка операции -- можно сразу возвращать
            return resultForValidItems;
        }
        checkState(resultForValidItems.getResult().size() == validItems.size(),
                "Должен быть возвращен результат для каждого входного элемента");

        // объединяем результаты валидации с результатами, полученными из операции
        List<ApiResult<IntRespEl>> results =
                unionValidateAndProcessResult(validationResult, resultForValidItems, requestItems);
        return new ApiMassResult<>(results, emptyList(), emptyList(), ApiResultState.SUCCESSFUL)
                .withCounts(resultForValidItems.getSuccessfulObjectsCount(),
                        resultForValidItems.getUnsuccessfulObjectsCount() + invalidsNum);
    }

    /**
     * Выполнить операцию над набором корректных входных элементов
     * <p>
     * Внимание! Если валидация списка элементов {@link #validateInternalElements(List)} найдёт ошибку
     * в каждом элементе списка, то этот метод {@code processList} не будет вызван вообще.
     *
     * @param requestWithValidElements запрос со списком валидных элементов
     * @return результат {@link ApiResult} операции надо списком
     */
    @SuppressWarnings("WeakerAccess")
    public abstract ApiMassResult<IntRespEl> processRequestWithList(IntReq requestWithValidElements);
}
