package ru.yandex.direct.api.v5.entity.adextensions.converter;

import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import com.yandex.direct.api.v5.adextensions.AdExtensionFieldEnum;
import com.yandex.direct.api.v5.adextensions.AdExtensionsSelectionCriteria;
import com.yandex.direct.api.v5.adextensions.CalloutFieldEnum;
import com.yandex.direct.api.v5.adextensions.GetRequest;
import com.yandex.direct.api.v5.adextensiontypes.AdExtensionStateSelectionEnum;
import com.yandex.direct.api.v5.general.ExtensionStatusSelectionEnum;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.entity.adextensions.container.GetFieldName;
import ru.yandex.direct.core.entity.addition.callout.container.CalloutSelection;
import ru.yandex.direct.core.entity.addition.callout.model.CalloutsStatusModerate;

import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
@Component
public class GetRequestConverter {
    private final ZoneId localTimezone;

    public GetRequestConverter(@Value("${timezone}") String timezone) {
        localTimezone = ZoneId.of(timezone);
    }

    public CalloutSelection extractSelectionCriteria(GetRequest externalRequest) {
        AdExtensionsSelectionCriteria sc = externalRequest.getSelectionCriteria();

        Boolean selectDeleted = null; // неопределено, значит выбирать удалённые и нет
        List<AdExtensionStateSelectionEnum> states = sc.getStates();
        if (sc.getIds().isEmpty() && states.isEmpty()) {
            // Из документации: "Если параметры Ids и States оба не заданы, метод не возвращает расширений
            // в состоянии DELETED."
            selectDeleted = Boolean.FALSE;
        } else if (!states.isEmpty() && (
                !states.contains(AdExtensionStateSelectionEnum.DELETED)
                        || !states.contains(AdExtensionStateSelectionEnum.ON))) {
            // States содержит только одно запрашиваемое состояние
            selectDeleted = states.contains(AdExtensionStateSelectionEnum.DELETED);
        }

        return new CalloutSelection()
                .withIds(sc.getIds())
                .withDeleted(selectDeleted)
                .withStatuses(convertSelectionCriteriaStatuses(sc.getStatuses()))
                .withLastChangeGreaterOrEqualThan(parseDate(sc.getModifiedSince()));
    }

    public Set<GetFieldName> extractFieldNames(GetRequest getRequest) {
        ImmutableSet.Builder<GetFieldName> sb = ImmutableSet.builder();
        sb.addAll(mapList(getRequest.getFieldNames(), GetRequestConverter::convertFieldName));
        if (!getRequest.getCalloutFieldNames().isEmpty()) {
            // все Callout поля находятся в поле Callout верхнего уровня
            sb.add(GetFieldName.CALLOUT);
        }
        sb.addAll(mapList(getRequest.getCalloutFieldNames(), GetRequestConverter::convertCalloutFieldNames));
        return sb.build();
    }

    @Nullable
    private LocalDateTime parseDate(@Nullable String modifiedSince) {
        if (modifiedSince == null) {
            return null;
        }
        ZonedDateTime zdt = ZonedDateTime.parse(modifiedSince, DateTimeFormatter.ISO_OFFSET_DATE_TIME);
        return zdt.withZoneSameInstant(localTimezone).toLocalDateTime();
    }

    private static GetFieldName convertFieldName(AdExtensionFieldEnum requestFieldName) {
        switch (requestFieldName) {
            case ID:
                return GetFieldName.ID;
            case TYPE:
                return GetFieldName.TYPE;
            case STATE:
                return GetFieldName.STATE;
            case STATUS:
                return GetFieldName.STATUS;
            case STATUS_CLARIFICATION:
                return GetFieldName.STATUS_CLARIFICATION;
            case ASSOCIATED:
                return GetFieldName.ASSOCIATED;
            default:
                throw new IllegalArgumentException("unknown field");
        }
    }

    private static GetFieldName convertCalloutFieldNames(CalloutFieldEnum requestCalloutFieldName) {
        switch (requestCalloutFieldName) {
            case CALLOUT_TEXT:
                return GetFieldName.CALLOUT_TEXT;
            default:
                throw new IllegalArgumentException("unknown field");
        }
    }

    private static Set<CalloutsStatusModerate> convertSelectionCriteriaStatuses(
            Collection<ExtensionStatusSelectionEnum> statues) {
        return statues.stream().flatMap(s -> convertSelectionCriteriaStatus(s).stream()).collect(Collectors.toSet());
    }

    private static Collection<CalloutsStatusModerate> convertSelectionCriteriaStatus(ExtensionStatusSelectionEnum status) {
        switch (status) {
            case DRAFT:
                return singletonList(CalloutsStatusModerate.NEW);
            case MODERATION:
                return asList(CalloutsStatusModerate.SENT, CalloutsStatusModerate.SENDING, CalloutsStatusModerate.READY);
            case ACCEPTED:
                return singletonList(CalloutsStatusModerate.YES);
            case REJECTED:
                return singletonList(CalloutsStatusModerate.NO);
            default:
                throw new IllegalArgumentException("unknown status");
        }
    }
}
