package ru.yandex.direct.api.v5.entity.adextensions.delegate;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.adextensions.AddRequest;
import com.yandex.direct.api.v5.adextensions.AddResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.api.v5.common.ApiPathConverter;
import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.BaseApiServiceDelegate;
import ru.yandex.direct.api.v5.result.ApiMassResult;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.entity.addition.callout.model.Callout;
import ru.yandex.direct.core.entity.addition.callout.service.CalloutService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.api.v5.entity.adextensions.Constants.MAX_ELEMENTS_PER_ADD;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxElementsPerRequest;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.maxListSize;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class AddAdExtensionsDelegate
        extends BaseApiServiceDelegate<AddRequest, AddResponse, List<Callout>, ApiResult<Long>> {

    private static final Logger logger = LoggerFactory.getLogger(AddAdExtensionsDelegate.class);

    private final CalloutService service;
    private final ResultConverter resultConverter;

    public AddAdExtensionsDelegate(
            CalloutService service,
            ApiAuthenticationSource auth,
            ResultConverter resultConverter) {
        super(ApiPathConverter.forAdExtensions(), auth);
        this.service = service;
        this.resultConverter = resultConverter;
    }

    @Override
    public ValidationResult<AddRequest, DefectType> validateRequest(AddRequest request) {
        ItemValidationBuilder<AddRequest, DefectType> builder = ItemValidationBuilder.of(request);
        builder.item(request.getAdExtensions(), "AdExtensions")
                .check(maxListSize(MAX_ELEMENTS_PER_ADD), maxElementsPerRequest(MAX_ELEMENTS_PER_ADD));
        return builder.getResult();
    }

    @Override
    public List<Callout> convertRequest(AddRequest externalRequest) {
        logger.debug("convert request: {}", externalRequest);
        return mapList(externalRequest.getAdExtensions(), ae -> convert(ae.getCallout()));
    }

    private Callout convert(com.yandex.direct.api.v5.adextensiontypes.Callout externalRequestItem) {
        ClientId clientId = getClientId();
        return new Callout()
                .withClientId(clientId.asLong())
                .withText(externalRequestItem.getCalloutText());
    }

    @Override
    public ApiMassResult<Long> processRequest(List<Callout> internalRequest) {
        logger.debug("process request: {}", internalRequest);
        return resultConverter.toApiMassResult(
                service.addCalloutsPartial(getClientId(), internalRequest));
    }

    @Override
    public AddResponse convertResponse(ApiResult<List<ApiResult<Long>>> result) {
        logger.debug("convert result: {}", result);
        return new AddResponse().withAddResults(resultConverter.toActionResults(result, apiPathConverter));
    }

    private ClientId getClientId() {
        return auth.getChiefSubclient().getClientId();
    }

}
