package ru.yandex.direct.api.v5.entity.adextensions.delegate;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.adextensions.AdExtensionGetItem;
import com.yandex.direct.api.v5.adextensions.GetRequest;
import com.yandex.direct.api.v5.adextensions.GetResponse;
import com.yandex.direct.api.v5.adextensiontypes.AdExtensionBase;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.api.v5.common.EnumPropertyFilter;
import ru.yandex.direct.api.v5.entity.GenericGetRequest;
import ru.yandex.direct.api.v5.entity.GetApiServiceDelegate;
import ru.yandex.direct.api.v5.entity.adextensions.container.GetFieldName;
import ru.yandex.direct.api.v5.entity.adextensions.container.InternalGetResponse;
import ru.yandex.direct.api.v5.entity.adextensions.converter.GetRequestConverter;
import ru.yandex.direct.api.v5.entity.adextensions.converter.InternalGetResponseConverter;
import ru.yandex.direct.api.v5.entity.adextensions.validation.GetRequestValidator;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.common.util.PropertyFilter;
import ru.yandex.direct.core.entity.addition.callout.container.CalloutSelection;
import ru.yandex.direct.core.entity.addition.callout.model.Callout;
import ru.yandex.direct.core.entity.addition.callout.service.CalloutService;
import ru.yandex.direct.core.entity.banner.type.banneradditions.BannerAdditionsRepository;
import ru.yandex.direct.core.entity.client.service.checker.ClientAccessCheckerTypeSupportFacade;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiag;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType;
import ru.yandex.direct.core.entity.moderationreason.service.ModerationReasonService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.validation.result.PathConverter;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class GetAdExtensionsDelegate extends GetApiServiceDelegate<GetRequest, GetResponse, GetFieldName,
        CalloutSelection, InternalGetResponse> {
    private final CalloutService calloutService;
    private final ShardHelper shardHelper;
    private final BannerAdditionsRepository bannerAdditionsRepository;
    private final ModerationReasonService moderationReasonService;
    private final EnumPropertyFilter<GetFieldName> propertyFilter;
    private final GetRequestConverter getRequestConverter;
    private final ClientAccessCheckerTypeSupportFacade clientAccessCheckerTypeSupportFacade;

    @Autowired
    public GetAdExtensionsDelegate(
            CalloutService calloutService,
            ShardHelper shardHelper,
            BannerAdditionsRepository bannerAdditionsRepository,
            ModerationReasonService moderationReasonService,
            ApiAuthenticationSource auth,
            PropertyFilter propertyFilter,
            GetRequestConverter getRequestConverter,
            ClientAccessCheckerTypeSupportFacade clientAccessCheckerTypeSupportFacade) {
        super(PathConverter.identity(), auth);
        this.calloutService = calloutService;
        this.shardHelper = shardHelper;
        this.bannerAdditionsRepository = bannerAdditionsRepository;
        this.moderationReasonService = moderationReasonService;
        this.propertyFilter = EnumPropertyFilter.from(GetFieldName.class, propertyFilter);
        this.getRequestConverter = getRequestConverter;
        this.clientAccessCheckerTypeSupportFacade = clientAccessCheckerTypeSupportFacade;
    }

    @Override
    public ValidationResult<GetRequest, DefectType> validateRequest(GetRequest externalRequest) {
        return GetRequestValidator.validateRequest(externalRequest);
    }

    @Override
    public Set<GetFieldName> extractFieldNames(GetRequest externalRequest) {
        return getRequestConverter.extractFieldNames(externalRequest);
    }

    @Override
    public CalloutSelection extractSelectionCriteria(GetRequest externalRequest) {
        return getRequestConverter.extractSelectionCriteria(externalRequest);
    }

    @Override
    public List<InternalGetResponse> get(GenericGetRequest<GetFieldName, CalloutSelection> getRequest) {
        ClientId clientId = auth.getSubclient().getClientId();
        List<ru.yandex.direct.core.entity.addition.callout.model.Callout> callouts = calloutService.getCallouts(
                clientId, getRequest.getSelectionCriteria(), getRequest.getLimitOffset());

        List<Long> calloutIds = mapList(callouts, Callout::getId);
        Set<Long> verifiedCalloutIds =
                clientAccessCheckerTypeSupportFacade.sendToCheck(Map.of(Callout.class, calloutIds),
                        clientId)
                        .getOrDefault(Callout.class, Collections.emptySet());
        List<InternalGetResponse> responses = mapList(callouts, InternalGetResponse::new);

        if (getRequest.getRequestedFields().contains(GetFieldName.ASSOCIATED)) {
            int shard = shardHelper.getShardByClientIdStrictly(clientId);
            Set<Long> linkedBannersAdditions = bannerAdditionsRepository.getLinkedBannersAdditions(shard,
                    verifiedCalloutIds);
            responses.forEach(r -> r.setAssociated(linkedBannersAdditions.contains(r.getCallout().getId())));
        }

        if (getRequest.getRequestedFields().contains(GetFieldName.STATUS_CLARIFICATION)) {
            Map<Long, List<ModerationDiag>> rejectReasonDiags = moderationReasonService
                    .getRejectReasonDiags(clientId, ModerationReasonObjectType.CALLOUT, verifiedCalloutIds);
            responses.forEach(r -> {
                List<ModerationDiag> diags =
                        rejectReasonDiags.getOrDefault(r.getCallout().getId(), Collections.emptyList());
                r.setStatusClarification(StreamEx.of(diags).map(ModerationDiag::getDiagText).joining("\n"));
            });
        }
        return responses;
    }

    @Override
    public GetResponse convertGetResponse(List<InternalGetResponse> result, Set<GetFieldName> requestedFields,
                                          @Nullable Long limitedBy) {
        GetResponse response = new GetResponse().withLimitedBy(limitedBy);
        if (!result.isEmpty()) {
            List<AdExtensionGetItem> getItems = mapList(result, InternalGetResponseConverter::convert);
            propertyFilter.filterProperties(mapList(getItems, AdExtensionBase::getCallout), requestedFields);
            propertyFilter.filterProperties(getItems, requestedFields);
            response.setAdExtensions(getItems);
        }
        return response;
    }
}
