package ru.yandex.direct.api.v5.entity.adgroups;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupWithType;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupWithTypeAndGeo;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.api.v5.entity.ads.AdsDefectTypes.notSupportedAdGroupType;

@ParametersAreNonnullByDefault
@Component
public class AdGroupTypeValidationService {
    private final ApiAuthenticationSource auth;
    private final AdGroupService service;

    @Autowired
    public AdGroupTypeValidationService(ApiAuthenticationSource auth, AdGroupService service) {
        this.auth = auth;
        this.service = service;
    }

    public ValidationResult<List<Long>, DefectType> validate(List<Long> adGroupIds,
                                                             Set<AdGroupType> allowedAdGroupTypes) {
        Map<Long, AdGroupWithType> adGroupTypes =
                service.getAdGroupsWithType(auth.getSubclient().getClientId(), adGroupIds);
        return ListValidationBuilder.<Long, DefectType>of(adGroupIds)
                .checkEach(allowedType(adGroupTypes, allowedAdGroupTypes))
                .getResult();
    }

    private static Constraint<Long, DefectType> allowedType(Map<Long, AdGroupWithType> types,
                                                            Set<AdGroupType> allowedAdGroupTypes) {
        return Constraint.fromPredicate(id -> !types.containsKey(id)
                        || allowedAdGroupTypes.contains(types.get(id).getType()),
                notSupportedAdGroupType());
    }

    public static void checkHasNoCpmPriceAdGroup(Collection<? extends AdGroupWithTypeAndGeo> adGroups) {
        boolean hasCpmPriceAdGroup = adGroups.stream()
                .anyMatch(adGroup -> adGroup.getType() == AdGroupType.CPM_YNDX_FRONTPAGE);
        checkState(!hasCpmPriceAdGroup, "Cpm Price adGroup is not supported in API5 yet. " +
                "Read more https://wiki.yandex-team.ru/users/i-pokusaev/Podderzhka-prajjsovojj-gruppy-v-api5/");
    }
}
