package ru.yandex.direct.api.v5.entity.adgroups;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.adgroups.AdGroupsPort;
import com.yandex.direct.api.v5.adgroups.AddRequest;
import com.yandex.direct.api.v5.adgroups.AddResponse;
import com.yandex.direct.api.v5.adgroups.DeleteRequest;
import com.yandex.direct.api.v5.adgroups.DeleteResponse;
import com.yandex.direct.api.v5.adgroups.GetRequest;
import com.yandex.direct.api.v5.adgroups.GetResponse;
import com.yandex.direct.api.v5.adgroups.UpdateRequest;
import com.yandex.direct.api.v5.adgroups.UpdateResponse;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.api.v5.common.ApiPathConverter;
import ru.yandex.direct.api.v5.common.validation.DefectPresentationsHolder;
import ru.yandex.direct.api.v5.entity.GenericApiService;
import ru.yandex.direct.api.v5.entity.adgroups.delegate.AddAdGroupsDelegate;
import ru.yandex.direct.api.v5.entity.adgroups.delegate.DeleteAdGroupsDelegate;
import ru.yandex.direct.api.v5.entity.adgroups.delegate.GetAdGroupsDelegate;
import ru.yandex.direct.api.v5.entity.adgroups.delegate.UpdateAdGroupsDelegate;
import ru.yandex.direct.api.v5.validation.DefectTypes;
import ru.yandex.direct.api.v5.ws.annotation.ApiMethod;
import ru.yandex.direct.api.v5.ws.annotation.ApiRequest;
import ru.yandex.direct.api.v5.ws.annotation.ApiResponse;
import ru.yandex.direct.api.v5.ws.annotation.ApiServiceEndpoint;
import ru.yandex.direct.api.v5.ws.annotation.ApiServiceType;
import ru.yandex.direct.api.v5.ws.annotation.ServiceType;
import ru.yandex.direct.core.entity.adgroup.container.ComplexPerformanceAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.DynamicTextAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.PerformanceAdGroup;
import ru.yandex.direct.core.entity.banner.model.PerformanceBannerMain;
import ru.yandex.direct.core.security.authorization.PreAuthorizeRead;
import ru.yandex.direct.validation.defect.ids.StringDefectIds;
import ru.yandex.direct.validation.result.DefectIds;
import ru.yandex.direct.validation.result.MappingPathConverter;
import ru.yandex.direct.validation.result.PathConverter;

@ApiServiceEndpoint
@ApiServiceType(type = ServiceType.CLIENT)
@ParametersAreNonnullByDefault
public class AdGroupsEndpoint implements AdGroupsPort {
    public static final PathConverter ADD_PATH_CONVERTER =
            getAdGroupsCommonPathConverterBuilder(AddAdGroupsDelegate.class, "")
                    .add(AdGroup.CAMPAIGN_ID.name(), "CampaignId")
                    .add(MobileContentAdGroup.STORE_URL.name(), "StoreUrl")
                    .add(PerformanceAdGroup.FEED_ID.name(), "FeedId")
                    .add(ContentPromotionAdGroup.CONTENT_PROMOTION_TYPE.name(), "PromotedContentType")
                    .ignore(AdGroup.TYPE.name())
                    .build();
    public static final PathConverter UPDATE_PATH_CONVERTER =
            getAdGroupsCommonPathConverterBuilder(UpdateAdGroupsDelegate.class, "")
                    .add(AdGroup.ID.name(), "Id")
                    .ignore(AdGroup.TYPE.name())
                    .build();

    public static final DefectPresentationsHolder DEFECT_PRESENTATIONS = DefectPresentationsHolder.builder()
            .register(StringDefectIds.LENGTH_CANNOT_BE_MORE_THAN_MAX,
                    t -> DefectTypes.lengthOfFieldValueMustNotExceed(t.getMaxLength()))
            .register(DefectIds.OBJECT_NOT_FOUND, t -> AdGroupDefectTypes.notFound())
            .register(DefectIds.NO_RIGHTS, t -> AdGroupDefectTypes.notFound())
            .build();

    private static final String SERVICE_NAME = "adgroups";

    private final GenericApiService genericApiService;
    private final GetAdGroupsDelegate getAdGroupsDelegate;
    private final AddAdGroupsDelegate addAdGroupsDelegate;
    private final UpdateAdGroupsDelegate updateAdGroupsDelegate;
    private final DeleteAdGroupsDelegate deleteAdGroupsDelegate;

    @Autowired
    public AdGroupsEndpoint(
            GenericApiService genericApiService,
            GetAdGroupsDelegate getAdGroupsDelegate,
            AddAdGroupsDelegate addAdGroupsDelegate,
            UpdateAdGroupsDelegate updateAdGroupsDelegate,
            DeleteAdGroupsDelegate deleteAdGroupsDelegate) {
        this.genericApiService = genericApiService;
        this.getAdGroupsDelegate = getAdGroupsDelegate;
        this.addAdGroupsDelegate = addAdGroupsDelegate;
        this.updateAdGroupsDelegate = updateAdGroupsDelegate;
        this.deleteAdGroupsDelegate = deleteAdGroupsDelegate;
    }

    private static MappingPathConverter.Builder getAdGroupsCommonPathConverterBuilder(Class<?> clazz, String name) {
        return MappingPathConverter.builder(clazz, name)
                .add(ApiPathConverter.getCommonDict())

                .add(AdGroup.NAME.name(), "Name")
                .add(AdGroup.GEO.name(), "RegionIds")
                .add(AdGroup.MINUS_KEYWORDS.name(), "NegativeKeywords")
                .add(AdGroup.LIBRARY_MINUS_KEYWORDS_IDS.name(), "NegativeKeywordSharedSetIds")
                .add(AdGroup.TRACKING_PARAMS.name(), "TrackingParams")

                .add(MobileContentAdGroup.DEVICE_TYPE_TARGETING.name(), "TargetDeviceType")
                .add(MobileContentAdGroup.MINIMAL_OPERATING_SYSTEM_VERSION.name(), "TargetOperatingSystemVersion")
                .add(MobileContentAdGroup.NETWORK_TARGETING.name(), "TargetCarrier")

                .add(DynamicTextAdGroup.DOMAIN_URL.name(), "DomainUrl")

                .ignore(ComplexPerformanceAdGroup.AD_GROUP.name())
                .add(PerformanceAdGroup.FIELD_TO_USE_AS_NAME.name(), "AdTitleSource")
                .add(PerformanceAdGroup.FIELD_TO_USE_AS_BODY.name(), "AdBodySource")
                .ignore(ComplexPerformanceAdGroup.BANNERS.name())
                .ignoreIndex(ComplexPerformanceAdGroup.BANNERS.name())
                .add(PerformanceBannerMain.LOGO_IMAGE_HASH.name(), "LogoExtensionHash");
    }

    @PreAuthorizeRead
    @ApiMethod(service = SERVICE_NAME, operation = "get")
    @ApiResponse
    @Override
    public GetResponse get(@ApiRequest GetRequest getRequest) {
        return genericApiService.doAction(getAdGroupsDelegate, getRequest);
    }

    @PreAuthorizeRead
    @ApiMethod(service = SERVICE_NAME, operation = "add")
    @ApiResponse
    @Override
    public AddResponse add(@ApiRequest AddRequest addRequest) {
        return genericApiService.doAction(addAdGroupsDelegate, addRequest);
    }

    @PreAuthorizeRead
    @ApiMethod(service = SERVICE_NAME, operation = "update")
    @ApiResponse
    @Override
    public UpdateResponse update(@ApiRequest UpdateRequest updateRequest) {
        return genericApiService.doAction(updateAdGroupsDelegate, updateRequest);
    }

    @PreAuthorizeRead
    @ApiMethod(service = SERVICE_NAME, operation = "delete")
    @ApiResponse
    @Override
    public DeleteResponse delete(@ApiRequest DeleteRequest deleteRequest) {
        return genericApiService.doAction(deleteAdGroupsDelegate, deleteRequest);
    }
}
