package ru.yandex.direct.api.v5.entity.adgroups.converter;

import java.util.Collections;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.adgroups.AdGroupAddItem;
import com.yandex.direct.api.v5.adgroups.DynamicTextFeedAdGroup;
import com.yandex.direct.api.v5.adgroups.MobileAppAdGroupAdd;
import com.yandex.direct.api.v5.adgroups.PromotedContentTypeEnum;
import com.yandex.direct.api.v5.adgroups.SmartAdGroupAdd;
import com.yandex.direct.api.v5.adgroups.TextAdGroupFeedParamsAdd;
import com.yandex.direct.api.v5.general.ArrayOfLong;
import com.yandex.direct.api.v5.general.YesNoEnum;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.entity.adgroups.AdGroupDefectTypes;
import ru.yandex.direct.api.v5.entity.adgroups.container.AdGroupsContainer;
import ru.yandex.direct.api.v5.entity.adgroups.container.AdGroupsValidationSignalContainer;
import ru.yandex.direct.api.v5.entity.adgroups.container.AddAdGroupsComplexPerformanceContainer;
import ru.yandex.direct.api.v5.entity.adgroups.container.AddAdGroupsSimpleContainer;
import ru.yandex.direct.api.v5.validation.DefectTypes;
import ru.yandex.direct.core.entity.adgroup.container.ComplexPerformanceAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdgroupType;
import ru.yandex.direct.core.entity.adgroup.model.CpmBannerAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmVideoAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CriterionType;
import ru.yandex.direct.core.entity.adgroup.model.DynamicAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.DynamicFeedAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.DynamicTextAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.PerformanceAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.TextAdGroup;
import ru.yandex.direct.core.entity.banner.model.PerformanceBannerMain;

import static ru.yandex.direct.api.v5.common.RelevanceMatchCategoriesConverter.autotargetingCategoriesToCore;
import static ru.yandex.direct.api.v5.entity.adgroups.converter.ConverterUtils.convertLibraryMinusKeywords;
import static ru.yandex.direct.api.v5.entity.adgroups.converter.ConverterUtils.convertMinusKeywords;
import static ru.yandex.direct.api.v5.entity.adgroups.converter.ConverterUtils.convertTargetCarrier;
import static ru.yandex.direct.api.v5.entity.adgroups.converter.ConverterUtils.convertTargetDeviceType;
import static ru.yandex.direct.api.v5.entity.adgroups.converter.ConverterUtils.isMixedGroup;
import static ru.yandex.direct.utils.CommonUtils.ifNotNullOrDefault;

@Component
@ParametersAreNonnullByDefault
public class AdGroupsAddRequestConverter {
    public AdGroupsContainer convertItem(AdGroupAddItem adGroupItem) {
        if (isMixedGroup(adGroupItem)) {
            return new AdGroupsValidationSignalContainer(DefectTypes.possibleOnlyOneField());
        } else if (adGroupItem.getMobileAppAdGroup() != null) {
            return convertMobileAppAdGroup(adGroupItem);
        } else if (adGroupItem.getDynamicTextAdGroup() != null) {
            return convertDynamicTextAdGroup(adGroupItem);
        } else if (adGroupItem.getDynamicTextFeedAdGroup() != null) {
            return convertDynamicTextFeedAdGroup(adGroupItem);
        } else if (adGroupItem.getCpmBannerKeywordsAdGroup() != null) {
            return new AddAdGroupsSimpleContainer(
                    convertCommonAdGroup(adGroupItem, new CpmBannerAdGroup())
                            .withType(AdGroupType.CPM_BANNER)
                            .withCriterionType(CriterionType.KEYWORD));
        } else if (adGroupItem.getCpmBannerUserProfileAdGroup() != null) {
            return new AddAdGroupsSimpleContainer(
                    convertCommonAdGroup(adGroupItem, new CpmVideoAdGroup())
                            .withType(AdGroupType.CPM_BANNER)
                            .withCriterionType(CriterionType.USER_PROFILE));
        } else if (adGroupItem.getCpmVideoAdGroup() != null) {
            boolean isNonSkippable = adGroupItem.getCpmVideoAdGroup().getIsNonSkippable() == YesNoEnum.YES;
            return new AddAdGroupsSimpleContainer(
                    convertCommonAdGroup(adGroupItem, new CpmVideoAdGroup())
                            .withType(AdGroupType.CPM_VIDEO)
                            .withIsNonSkippable(isNonSkippable));
        } else if (adGroupItem.getSmartAdGroup() != null) {
            return convertSmartAdGroup(adGroupItem);
        } else if (adGroupItem.getContentPromotionAdGroup() != null) {
            return new AddAdGroupsSimpleContainer(
                    convertCommonAdGroup(adGroupItem, new ContentPromotionAdGroup())
                            .withType(AdGroupType.CONTENT_PROMOTION)
                            .withContentPromotionType(convertContentPromotionType(
                                    adGroupItem.getContentPromotionAdGroup().getPromotedContentType())));
        } else {
            TextAdGroup group = convertCommonAdGroup(adGroupItem, new TextAdGroup())
                    .withType(AdGroupType.BASE);
            if (adGroupItem.getTextAdGroupFeedParams() != null) {
                TextAdGroupFeedParamsAdd textAdGroupFeedParamsAdd = adGroupItem.getTextAdGroupFeedParams();
                group.withOldFeedId(textAdGroupFeedParamsAdd.getFeedId())
                        .withFeedFilterCategories(ifNotNullOrDefault(textAdGroupFeedParamsAdd.getFeedCategoryIds(),
                                ArrayOfLong::getItems, Collections.emptyList()));
            }
            return new AddAdGroupsSimpleContainer(group);
        }
    }

    private static <T extends AdGroup> T convertCommonAdGroup(AdGroupAddItem adGroupItem, T adGroup) {
        adGroup
                .withName(adGroupItem.getName())
                .withCampaignId(adGroupItem.getCampaignId())
                .withGeo(adGroupItem.getRegionIds())
                .withMinusKeywords(convertMinusKeywords(adGroupItem.getNegativeKeywords()))
                .withLibraryMinusKeywordsIds(convertLibraryMinusKeywords(adGroupItem
                        .getNegativeKeywordSharedSetIds()))
                .withTrackingParams(adGroupItem.getTrackingParams());
        return adGroup;
    }

    private static <T extends DynamicAdGroup> T convertDynamicAdGroup(AdGroupAddItem adGroupItem,
            com.yandex.direct.api.v5.adgroups.DynamicAdGroup dynamicAdGroupItem, T adGroup) {
        convertCommonAdGroup(adGroupItem, adGroup)
                .withType(AdGroupType.DYNAMIC)
                .withRelevanceMatchCategories(autotargetingCategoriesToCore(
                        dynamicAdGroupItem.getAutotargetingCategories()));
        return adGroup;
    }

    static AdGroupsContainer convertMobileAppAdGroup(AdGroupAddItem adGroupItem) {
        MobileAppAdGroupAdd mobileAppAdGroup = adGroupItem.getMobileAppAdGroup();
        return new AddAdGroupsSimpleContainer(convertCommonAdGroup(adGroupItem, new MobileContentAdGroup())
                .withType(AdGroupType.MOBILE_CONTENT)
                .withStoreUrl(mobileAppAdGroup.getStoreUrl())
                .withDeviceTypeTargeting(convertTargetDeviceType(mobileAppAdGroup.getTargetDeviceType()))
                .withNetworkTargeting(convertTargetCarrier(mobileAppAdGroup.getTargetCarrier()))
                .withMinimalOperatingSystemVersion(mobileAppAdGroup.getTargetOperatingSystemVersion()));
    }

    static AdGroupsContainer convertDynamicTextAdGroup(AdGroupAddItem adGroupItem) {
        com.yandex.direct.api.v5.adgroups.DynamicTextAdGroup dynamicTextAdGroup = adGroupItem.getDynamicTextAdGroup();
        return new AddAdGroupsSimpleContainer(
                convertDynamicAdGroup(adGroupItem, dynamicTextAdGroup, new DynamicTextAdGroup())
                        .withDomainUrl(dynamicTextAdGroup.getDomainUrl()));
    }

    static AdGroupsContainer convertDynamicTextFeedAdGroup(AdGroupAddItem adGroupItem) {
        DynamicTextFeedAdGroup dynamicTextFeedAdGroup = adGroupItem.getDynamicTextFeedAdGroup();
        return new AddAdGroupsSimpleContainer(
                convertDynamicAdGroup(adGroupItem, dynamicTextFeedAdGroup, new DynamicFeedAdGroup()
                        .withFeedId(dynamicTextFeedAdGroup.getFeedId())));
    }

    private static AdGroupsContainer convertSmartAdGroup(AdGroupAddItem adGroupItem) {
        SmartAdGroupAdd smartAdGroupItem = adGroupItem.getSmartAdGroup();
        PerformanceAdGroup adGroup = convertCommonAdGroup(adGroupItem, new PerformanceAdGroup())
                .withType(AdGroupType.PERFORMANCE)
                .withFeedId(smartAdGroupItem.getFeedId())
                .withFieldToUseAsName(smartAdGroupItem.getAdTitleSource())
                .withFieldToUseAsBody(smartAdGroupItem.getAdBodySource());
        if (Boolean.TRUE.equals(smartAdGroupItem.isNoCreatives())) {
            return new AddAdGroupsComplexPerformanceContainer(
                    new ComplexPerformanceAdGroup()
                            .withAdGroup(adGroup)
                            .withBanners(List.of(new PerformanceBannerMain()
                                    .withLogoImageHash(smartAdGroupItem.getLogoExtensionHash()))));
        } else if (smartAdGroupItem.getLogoExtensionHash() != null) {
            return new AdGroupsValidationSignalContainer(AdGroupDefectTypes.logoIsOnlyForBannersWithoutCreative());
        } else {
            return new AddAdGroupsSimpleContainer(adGroup);
        }
    }

    private static ContentPromotionAdgroupType convertContentPromotionType(
            @Nullable PromotedContentTypeEnum promotedContentTypeEnum) {
        if (promotedContentTypeEnum == null) {  // валидация на null проиходит далее в core
            return null;
        }
        switch (promotedContentTypeEnum) {
            case VIDEO:
                return ContentPromotionAdgroupType.VIDEO;
            case COLLECTION:
                return ContentPromotionAdgroupType.COLLECTION;
            case SERVICE:
                return ContentPromotionAdgroupType.SERVICE;
            case EDA:
                return ContentPromotionAdgroupType.EDA;
            default:
                throw new IllegalArgumentException("Not supported content promotion type: " + promotedContentTypeEnum);
        }
    }
}
