package ru.yandex.direct.api.v5.entity.adgroups.converter

import com.yandex.direct.api.v5.adgroups.AdGroupUpdateItem
import com.yandex.direct.api.v5.adgroups.DynamicAdGroup
import org.springframework.stereotype.Component
import ru.yandex.direct.api.v5.common.RelevanceMatchCategoriesConverter.mergeAutotargetingCategoriesFromRequestWithDb
import ru.yandex.direct.api.v5.common.buildModelChanges
import ru.yandex.direct.api.v5.common.extractValueOrDefault
import ru.yandex.direct.api.v5.common.processJaxbElement
import ru.yandex.direct.api.v5.entity.adgroups.container.AdGroupsContainer
import ru.yandex.direct.api.v5.entity.adgroups.container.AdGroupsValidationSignalContainer
import ru.yandex.direct.api.v5.entity.adgroups.container.UpdateAdGroupsComplexPerformanceContainer
import ru.yandex.direct.api.v5.entity.adgroups.container.UpdateAdGroupsSimpleContainer
import ru.yandex.direct.api.v5.entity.adgroups.converter.ConverterUtils.convertTargetCarrier
import ru.yandex.direct.api.v5.entity.adgroups.converter.ConverterUtils.convertTargetDeviceType
import ru.yandex.direct.api.v5.entity.adgroups.converter.ConverterUtils.isMixedGroup
import ru.yandex.direct.api.v5.validation.DefectTypes.possibleOnlyOneField
import ru.yandex.direct.core.entity.adgroup.container.ComplexPerformanceAdGroup
import ru.yandex.direct.core.entity.adgroup.model.AdGroup
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType
import ru.yandex.direct.core.entity.adgroup.model.DynamicFeedAdGroup
import ru.yandex.direct.core.entity.adgroup.model.DynamicTextAdGroup
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroup
import ru.yandex.direct.core.entity.adgroup.model.PerformanceAdGroup
import ru.yandex.direct.core.entity.adgroup.model.TextAdGroup
import ru.yandex.direct.core.entity.banner.model.PerformanceBannerMain
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.core.entity.adgroup.model.DynamicAdGroup as CoreDynamicAdGroup

private fun isGeoChanged(geo: List<Long>, oldGeo: List<Long>?) =
    geo.isNotEmpty() && geo.toSet() != (oldGeo ?: emptyList()).toSet()

private fun <M : AdGroup> ModelChanges<M>.processGeo(geo: List<Long>, oldGeo: List<Long>?) {
    if (isGeoChanged(geo, oldGeo)) {
        process(geo, AdGroup.GEO)
        process(null, AdGroup.HYPER_GEO_ID)
    }
}

private fun <M : AdGroup> ModelChanges<M>.processCommonFields(item: AdGroupUpdateItem, oldAdGroup: AdGroup?) {
    processNotNull(item.name, AdGroup.NAME)
    processGeo(item.regionIds, oldAdGroup?.geo)
    processJaxbElement(item.negativeKeywords, AdGroup.MINUS_KEYWORDS, emptyList()) { it.items }
    processJaxbElement(item.negativeKeywordSharedSetIds, AdGroup.LIBRARY_MINUS_KEYWORDS_IDS, emptyList()) { it.items }
    processNotNull(item.trackingParams, AdGroup.TRACKING_PARAMS)
}

private fun <M : CoreDynamicAdGroup> ModelChanges<M>.processDynamicFields(item: DynamicAdGroup, oldAdGroup: AdGroup?) {
    process(mergeAutotargetingCategoriesFromRequestWithDb(item.autotargetingCategories,
        (oldAdGroup as? CoreDynamicAdGroup)?.relevanceMatchCategories), CoreDynamicAdGroup.RELEVANCE_MATCH_CATEGORIES)
}

@Component
class AdGroupsUpdateRequestConverter {
    fun convert(item: AdGroupUpdateItem, oldAdGroup: AdGroup?): AdGroupsContainer =
        when {
            isMixedGroup(item) -> AdGroupsValidationSignalContainer(possibleOnlyOneField())
            item.mobileAppAdGroup != null -> convertMobileAppAdGroup(item, oldAdGroup)
            item.dynamicTextAdGroup != null -> convertDynamicTextAdGroup(item, oldAdGroup)
            item.dynamicTextFeedAdGroup != null -> convertDynamicTextFeedAdGroup(item, oldAdGroup)
            item.smartAdGroup != null -> convertSmartAdGroup(item, oldAdGroup)
            item.textAdGroupFeedParams != null -> convertTextAdGroup(item, oldAdGroup)
            else -> UpdateAdGroupsSimpleContainer(buildModelChanges(item.id) {
                processCommonFields(item, oldAdGroup)
            }, oldAdGroup)
        }

    private fun convertMobileAppAdGroup(item: AdGroupUpdateItem, oldAdGroup: AdGroup?): AdGroupsContainer {
        val mobileAppAdGroup = item.mobileAppAdGroup
        return UpdateAdGroupsSimpleContainer(buildModelChanges<MobileContentAdGroup>(item.id) {
            processCommonFields(item, oldAdGroup)
            processNotNull(convertTargetDeviceType(mobileAppAdGroup.targetDeviceType.takeIf { it.isNotEmpty() }),
                MobileContentAdGroup.DEVICE_TYPE_TARGETING)
            processNotNull(convertTargetCarrier(mobileAppAdGroup.targetCarrier),
                MobileContentAdGroup.NETWORK_TARGETING)
            processNotNull(mobileAppAdGroup.targetOperatingSystemVersion,
                MobileContentAdGroup.MINIMAL_OPERATING_SYSTEM_VERSION)
        }, oldAdGroup)
    }

    private fun convertDynamicTextAdGroup(item: AdGroupUpdateItem, oldAdGroup: AdGroup?): AdGroupsContainer =
        UpdateAdGroupsSimpleContainer(buildModelChanges<DynamicTextAdGroup>(item.id) {
            processCommonFields(item, oldAdGroup)
            processDynamicFields(item.dynamicTextAdGroup, oldAdGroup)
            processNotNull(item.dynamicTextAdGroup.domainUrl, DynamicTextAdGroup.DOMAIN_URL)
        }, oldAdGroup)

    private fun convertDynamicTextFeedAdGroup(item: AdGroupUpdateItem, oldAdGroup: AdGroup?): AdGroupsContainer =
        UpdateAdGroupsSimpleContainer(buildModelChanges<DynamicFeedAdGroup>(item.id) {
            processCommonFields(item, oldAdGroup)
            processDynamicFields(item.dynamicTextFeedAdGroup, oldAdGroup)
            processNotNull(item.dynamicTextFeedAdGroup.feedId, DynamicFeedAdGroup.FEED_ID)
        }, oldAdGroup)

    private fun convertSmartAdGroup(item: AdGroupUpdateItem, oldAdGroup: AdGroup?): AdGroupsContainer {
        val smartAdGroup = item.smartAdGroup
        return if (smartAdGroup.logoExtensionHash == null) {
            UpdateAdGroupsSimpleContainer(buildModelChanges<PerformanceAdGroup>(item.id) {
                processCommonFields(item, oldAdGroup)
                processJaxbElement(smartAdGroup.adTitleSource, PerformanceAdGroup.FIELD_TO_USE_AS_NAME)
                processJaxbElement(smartAdGroup.adBodySource, PerformanceAdGroup.FIELD_TO_USE_AS_BODY)
                processNotNull(smartAdGroup.feedId, PerformanceAdGroup.FEED_ID)
            }, oldAdGroup)
        } else {
            val performanceAdGroup = oldAdGroup as? PerformanceAdGroup
            val performanceMainBanner = oldAdGroup?.banners?.asSequence()
                ?.filterIsInstance<PerformanceBannerMain>()
                ?.firstOrNull()

            val isGeoChanged = isGeoChanged(item.regionIds, oldAdGroup?.geo)

            UpdateAdGroupsComplexPerformanceContainer(ComplexPerformanceAdGroup().apply {
                adGroup = PerformanceAdGroup().apply {
                    id = item.id
                    type = AdGroupType.PERFORMANCE
                    name = item.name ?: oldAdGroup?.name
                    minusKeywords = item.negativeKeywords
                        .extractValueOrDefault(oldAdGroup?.minusKeywords) { it?.items ?: emptyList() }
                    libraryMinusKeywordsIds = item.negativeKeywordSharedSetIds
                        .extractValueOrDefault(oldAdGroup?.libraryMinusKeywordsIds) { it?.items ?: emptyList() }
                    geo = if (isGeoChanged) item.regionIds else oldAdGroup?.geo
                    hyperGeoId = if (isGeoChanged) null else oldAdGroup?.hyperGeoId
                    trackingParams = item.trackingParams ?: oldAdGroup?.trackingParams
                    fieldToUseAsName = smartAdGroup.adTitleSource
                        .extractValueOrDefault(performanceAdGroup?.fieldToUseAsName)
                    fieldToUseAsBody = smartAdGroup.adBodySource
                        .extractValueOrDefault(performanceAdGroup?.fieldToUseAsBody)
                    feedId = smartAdGroup.feedId ?: performanceAdGroup?.feedId
                }
                banners = listOf(PerformanceBannerMain().apply {
                    id = performanceMainBanner?.id
                    logoImageHash = smartAdGroup.logoExtensionHash
                        .extractValueOrDefault(performanceMainBanner?.logoImageHash)
                })
            }, oldAdGroup)
        }
    }

    private fun convertTextAdGroup(item: AdGroupUpdateItem, oldAdGroup: AdGroup?): AdGroupsContainer {
        val textAdGroupFeedParams = item.textAdGroupFeedParams
        return UpdateAdGroupsSimpleContainer(buildModelChanges<TextAdGroup>(item.id) {
            processCommonFields(item, oldAdGroup)
            process(textAdGroupFeedParams.feedId, TextAdGroup.OLD_FEED_ID)
            if (textAdGroupFeedParams.feedCategoryIds != null) {
                process(textAdGroupFeedParams.feedCategoryIds.takeUnless { it.isNil }?.value?.items ?: emptyList(),
                    TextAdGroup.FEED_FILTER_CATEGORIES)
            }
        }, oldAdGroup)
    }
}
