package ru.yandex.direct.api.v5.entity.adgroups.converter;

import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import com.google.common.collect.ImmutableMap;
import com.yandex.direct.api.v5.adgroups.AdGroupAddItem;
import com.yandex.direct.api.v5.adgroups.AdGroupUpdateItem;
import com.yandex.direct.api.v5.adgroups.TargetCarrierEnum;
import com.yandex.direct.api.v5.adgroups.TargetDeviceTypeEnum;
import com.yandex.direct.api.v5.general.ArrayOfLong;
import com.yandex.direct.api.v5.general.ArrayOfString;

import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroupDeviceTypeTargeting;
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroupNetworkTargeting;

import static java.util.Collections.emptyList;
import static java.util.Collections.singleton;
import static java.util.stream.Collectors.toCollection;

final class ConverterUtils {
    private static final Map<TargetDeviceTypeEnum, MobileContentAdGroupDeviceTypeTargeting>
            DEVICE_TYPE_TARGETING_MAP =
            ImmutableMap.<TargetDeviceTypeEnum, MobileContentAdGroupDeviceTypeTargeting>builder()
                    .put(TargetDeviceTypeEnum.DEVICE_TYPE_MOBILE, MobileContentAdGroupDeviceTypeTargeting.PHONE)
                    .put(TargetDeviceTypeEnum.DEVICE_TYPE_TABLET, MobileContentAdGroupDeviceTypeTargeting.TABLET)
                    .build();

    private static final Map<TargetCarrierEnum, Set<MobileContentAdGroupNetworkTargeting>> CARRIER_TARGETING_MAP =
            ImmutableMap.<TargetCarrierEnum, Set<MobileContentAdGroupNetworkTargeting>>builder()
                    .put(TargetCarrierEnum.WI_FI_AND_CELLULAR,
                            EnumSet.of(MobileContentAdGroupNetworkTargeting.WI_FI,
                                    MobileContentAdGroupNetworkTargeting.CELLULAR))
                    .put(TargetCarrierEnum.WI_FI_ONLY,
                            singleton(MobileContentAdGroupNetworkTargeting.WI_FI))
                    .build();

    static List<String> convertMinusKeywords(ArrayOfString negativeKeywords) {
        return Optional.ofNullable(negativeKeywords)
                .map(ArrayOfString::getItems)
                .orElse(emptyList());
    }

    static List<Long> convertLibraryMinusKeywords(ArrayOfLong negativeKeywordIds) {
        return Optional.ofNullable(negativeKeywordIds)
                .map(ArrayOfLong::getItems)
                .orElse(emptyList());
    }

    static Set<MobileContentAdGroupDeviceTypeTargeting> convertTargetDeviceType(
            List<TargetDeviceTypeEnum> targetDeviceTypes) {
        if (targetDeviceTypes == null) {
            return null;
        }

        return targetDeviceTypes.stream()
                .map(DEVICE_TYPE_TARGETING_MAP::get)
                .collect(toCollection(() -> EnumSet.noneOf(MobileContentAdGroupDeviceTypeTargeting.class)));
    }

    static Set<MobileContentAdGroupNetworkTargeting> convertTargetCarrier(TargetCarrierEnum targetCarrier) {
        if (targetCarrier == null) {
            return null;
        }
        return CARRIER_TARGETING_MAP.get(targetCarrier);
    }

    static boolean isMixedGroup(AdGroupAddItem item) {
        return isMixed(item.getMobileAppAdGroup(),
                item.getDynamicTextAdGroup(),
                item.getCpmVideoAdGroup(),
                item.getCpmBannerKeywordsAdGroup(),
                item.getCpmBannerUserProfileAdGroup(),
                item.getSmartAdGroup(),
                item.getContentPromotionAdGroup(),
                item.getTextAdGroupFeedParams());
    }

    static boolean isMixedGroup(AdGroupUpdateItem item) {
        return isMixed(item.getMobileAppAdGroup(),
                item.getDynamicTextAdGroup(),
                item.getSmartAdGroup(),
                item.getTextAdGroupFeedParams());
    }

    private static boolean isMixed(Object... groups) {
        int count = 0;
        for (Object group : groups) {
            if (group != null) {
                count++;
            }
            if (count > 1) {
                return true;
            }
        }
        return false;
    }
}
