package ru.yandex.direct.api.v5.entity.adgroups.converter;

import java.util.Collection;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.yandex.direct.api.v5.adgroups.AdGroupAppIconStatusSelectionEnum;
import com.yandex.direct.api.v5.adgroups.AdGroupStatusSelectionEnum;
import com.yandex.direct.api.v5.adgroups.PromotedContentTypeEnum;
import com.yandex.direct.api.v5.general.AdGroupTypesEnum;
import com.yandex.direct.api.v5.general.ServingStatusEnum;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.ServingStatus;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupAppIconStatus;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupStatus;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdgroupType;

@ParametersAreNonnullByDefault
@Service("adGroupGetRequestConverter")
public class GetRequestConverter {

    private static final Map<AdGroupTypesEnum, AdGroupType> EXTERNAL_TO_INTERNAL_ADGROUP_TYPE =
            ImmutableMap.<AdGroupTypesEnum, AdGroupType>builder()
                    .put(AdGroupTypesEnum.TEXT_AD_GROUP, AdGroupType.BASE)
                    .put(AdGroupTypesEnum.MOBILE_APP_AD_GROUP, AdGroupType.MOBILE_CONTENT)
                    .put(AdGroupTypesEnum.DYNAMIC_TEXT_AD_GROUP, AdGroupType.DYNAMIC)
                    .put(AdGroupTypesEnum.CPM_BANNER_AD_GROUP, AdGroupType.CPM_BANNER)
                    .put(AdGroupTypesEnum.CPM_VIDEO_AD_GROUP, AdGroupType.CPM_VIDEO)
                    .put(AdGroupTypesEnum.SMART_AD_GROUP, AdGroupType.PERFORMANCE)
                    .put(AdGroupTypesEnum.CONTENT_PROMOTION_AD_GROUP, AdGroupType.CONTENT_PROMOTION)
                    .build();

    private static final Map<AdGroupStatusSelectionEnum, AdGroupStatus> EXTERNAL_TO_INTERNAL_ADGROUP_STATUS =
            ImmutableMap.<AdGroupStatusSelectionEnum, AdGroupStatus>builder()
                    .put(AdGroupStatusSelectionEnum.DRAFT, AdGroupStatus.DRAFT)
                    .put(AdGroupStatusSelectionEnum.MODERATION, AdGroupStatus.MODERATION)
                    .put(AdGroupStatusSelectionEnum.PREACCEPTED, AdGroupStatus.PREACCEPTED)
                    .put(AdGroupStatusSelectionEnum.ACCEPTED, AdGroupStatus.ACCEPTED)
                    .put(AdGroupStatusSelectionEnum.REJECTED, AdGroupStatus.REJECTED)
                    .build();

    private static final Map<AdGroupAppIconStatusSelectionEnum, AdGroupAppIconStatus>
            EXTERNAL_TO_INTERNAL_ADGROUP_APP_ICON_STATUS =
            ImmutableMap.<AdGroupAppIconStatusSelectionEnum, AdGroupAppIconStatus>builder()
                    .put(AdGroupAppIconStatusSelectionEnum.MODERATION, AdGroupAppIconStatus.MODERATION)
                    .put(AdGroupAppIconStatusSelectionEnum.ACCEPTED, AdGroupAppIconStatus.ACCEPTED)
                    .put(AdGroupAppIconStatusSelectionEnum.REJECTED, AdGroupAppIconStatus.REJECTED)
                    .build();

    private static final Map<ServingStatusEnum, ServingStatus> EXTERNAL_TO_INTERNAL_ADGROUP_SERVING_STATUS =
            ImmutableMap.<ServingStatusEnum, ServingStatus>builder()
                    .put(ServingStatusEnum.ELIGIBLE, ServingStatus.ELIGIBLE)
                    .put(ServingStatusEnum.RARELY_SERVED, ServingStatus.RARELY_SERVED)
                    .build();

    public static Set<AdGroupType> convertTypes(List<AdGroupTypesEnum> externalTypes) {
        Set<AdGroupType> result = EnumSet.noneOf(AdGroupType.class);

        for (AdGroupTypesEnum value : externalTypes) {
            AdGroupType internalValue = EXTERNAL_TO_INTERNAL_ADGROUP_TYPE.get(value);

            if (internalValue == null) {
                throw new IllegalArgumentException("Not supported value: " + value);
            }

            result.add(internalValue);
        }

        return result;
    }

    public static Set<AdGroupStatus> convertStatuses(Collection<AdGroupStatusSelectionEnum> externalStatuses) {
        Set<AdGroupStatus> result = EnumSet.noneOf(AdGroupStatus.class);

        for (AdGroupStatusSelectionEnum value : externalStatuses) {
            AdGroupStatus internalValue = EXTERNAL_TO_INTERNAL_ADGROUP_STATUS.get(value);

            if (internalValue == null) {
                throw new IllegalArgumentException("Not supported value: " + value);
            }

            result.add(internalValue);
        }

        return result;
    }

    public static Set<AdGroupAppIconStatus> convertAppIconStatuses(
            List<AdGroupAppIconStatusSelectionEnum> externalStatuses) {
        Set<AdGroupAppIconStatus> result = EnumSet.noneOf(AdGroupAppIconStatus.class);

        for (AdGroupAppIconStatusSelectionEnum value : externalStatuses) {
            AdGroupAppIconStatus internalValue = EXTERNAL_TO_INTERNAL_ADGROUP_APP_ICON_STATUS.get(value);

            if (internalValue == null) {
                throw new IllegalArgumentException("Not supported value: " + value);
            }

            result.add(internalValue);
        }

        return result;
    }

    public static Set<ServingStatus> convertServingStatuses(List<ServingStatusEnum> externalStatuses) {
        Set<ServingStatus> result = EnumSet.noneOf(ServingStatus.class);

        for (ServingStatusEnum value : externalStatuses) {
            ServingStatus internalValue = EXTERNAL_TO_INTERNAL_ADGROUP_SERVING_STATUS.get(value);

            if (internalValue == null) {
                throw new IllegalArgumentException("Not supported value: " + value);
            }

            result.add(internalValue);
        }

        return result;
    }

    public static ContentPromotionAdgroupType convertContentPromotionType(
            PromotedContentTypeEnum externalContentPromotionType) {
        //null должен отсекаться json-валидацией
        switch (externalContentPromotionType) {
            case VIDEO:
                return ContentPromotionAdgroupType.VIDEO;
            case COLLECTION:
                return ContentPromotionAdgroupType.COLLECTION;
            case SERVICE:
                return ContentPromotionAdgroupType.SERVICE;
            case EDA:
                return ContentPromotionAdgroupType.EDA;
            default:
                throw new IllegalArgumentException("Not supported content promotion type: " +
                        externalContentPromotionType);
        }
    }
}
