package ru.yandex.direct.api.v5.entity.adgroups.delegate;

import java.util.List;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.adgroups.DeleteRequest;
import com.yandex.direct.api.v5.adgroups.DeleteResponse;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.OperationOnListDelegate;
import ru.yandex.direct.api.v5.entity.adgroups.AdGroupTypeValidationService;
import ru.yandex.direct.api.v5.entity.adgroups.Constants;
import ru.yandex.direct.api.v5.entity.adgroups.converter.DeleteAdGroupsRequestConverter;
import ru.yandex.direct.api.v5.result.ApiMassResult;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.api.v5.validation.DefectTypes;
import ru.yandex.direct.api.v5.validation.constraints.Constraints;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.MappingPathConverter;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.api.v5.entity.adgroups.Constants.MAX_IDS_PER_DELETE;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.eachNotNull;

@ParametersAreNonnullByDefault
@Service
public class DeleteAdGroupsDelegate
        extends OperationOnListDelegate<DeleteRequest, DeleteResponse, Long, Long> {
    private static final DefectType DELETE_MAX_IDS_DETAILED =
            DefectTypes.maxElementsInSelection(Constants.MAX_IDS_PER_DELETE);

    private final DeleteAdGroupsRequestConverter requestConverter;
    private final ResultConverter resultConverter;
    private final AdGroupService service;
    private final AdGroupTypeValidationService typeValidationService;

    @Autowired
    public DeleteAdGroupsDelegate(ApiAuthenticationSource auth,
                                  DeleteAdGroupsRequestConverter requestConverter,
                                  ResultConverter resultConverter,
                                  AdGroupService service,
                                  AdGroupTypeValidationService typeValidationService,
                                  PpcPropertiesSupport ppcPropertiesSupport,
                                  FeatureService featureService) {
        super(MappingPathConverter.builder(DeleteAdGroupsDelegate.class, "emptyConverter").build(),
                auth,
                ppcPropertiesSupport,
                featureService);
        this.requestConverter = requestConverter;
        this.resultConverter = resultConverter;
        this.service = service;
        this.typeValidationService = typeValidationService;
    }

    @Override
    public ValidationResult<DeleteRequest, DefectType> validateRequest(DeleteRequest request) {

        ItemValidationBuilder<DeleteRequest, DefectType> vb = ItemValidationBuilder.of(request);
        vb.item(request.getSelectionCriteria(), "SelectionCriteria")
                .item(request.getSelectionCriteria().getIds(), "Ids")
                .check(Constraints.maxListSize(MAX_IDS_PER_DELETE), DELETE_MAX_IDS_DETAILED)
                .check(eachNotNull());

        return vb.getResult();
    }

    @Override
    public List<Long> convertRequest(DeleteRequest externalRequest) {
        return requestConverter.convert(externalRequest);
    }

    @Nonnull
    @Override
    public ValidationResult<List<Long>, DefectType> validateInternalRequest(List<Long> internalRequest) {
        Set<AdGroupType> allowedAdGroupTypes = getAllowedAdGroupTypes();
        return typeValidationService.validate(internalRequest, allowedAdGroupTypes);
    }

    @Override
    public ApiMassResult<Long> processList(List<Long> adGroupIds) {
        ClientId clientId = auth.getSubclient().getClientId();
        long operatorUid = auth.getOperator().getUid();
        return resultConverter.toApiMassResult(service.deleteAdGroups(operatorUid, clientId, adGroupIds));
    }

    @Override
    public DeleteResponse convertResponse(ApiResult<List<ApiResult<Long>>> result) {
        return new DeleteResponse()
                .withDeleteResults(resultConverter.toActionResults(result, apiPathConverter));
    }
}
