package ru.yandex.direct.api.v5.entity.adgroups.delegate;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.adgroups.AdGroupGetItem;
import com.yandex.direct.api.v5.adgroups.GetRequest;
import com.yandex.direct.api.v5.adgroups.GetResponse;
import com.yandex.direct.api.v5.adgroups.PromotedContentTypeEnum;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.common.validation.GetRequestGeneralValidator;
import ru.yandex.direct.api.v5.entity.GenericGetRequest;
import ru.yandex.direct.api.v5.entity.GetApiServiceDelegate;
import ru.yandex.direct.api.v5.entity.adgroups.Constants;
import ru.yandex.direct.api.v5.entity.adgroups.converter.GetRequestConverter;
import ru.yandex.direct.api.v5.entity.adgroups.converter.GetResponseConverter;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.entity.adgroup.container.AdGroupsSelectionCriteria;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdgroupType;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.MappingPathConverter;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.yandex.direct.api.v5.adgroups.AdGroupsSelectionCriteria.PropInfo.CAMPAIGN_IDS;
import static com.yandex.direct.api.v5.adgroups.AdGroupsSelectionCriteria.PropInfo.IDS;
import static com.yandex.direct.api.v5.adgroups.AdGroupsSelectionCriteria.PropInfo.NEGATIVE_KEYWORD_SHARED_SET_IDS;
import static com.yandex.direct.api.v5.adgroups.AdGroupsSelectionCriteria.PropInfo.PROMOTED_CONTENT_TYPES;
import static com.yandex.direct.api.v5.adgroups.GetRequest.PropInfo.SELECTION_CRITERIA;
import static java.util.Arrays.asList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.api.v5.common.constants.GetRequestCommonConstants.DEFAULT_MAX_IDS_COUNT;
import static ru.yandex.direct.api.v5.validation.DefectTypes.invalidUseOfField;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxIdsInSelection;
import static ru.yandex.direct.api.v5.validation.DefectTypes.missedParamsInSelection;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.maxListSize;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.notNull;
import static ru.yandex.direct.core.entity.adgroup.container.AccessibleAdGroupTypes.API5_ALLOWED_CONTENT_PROMOTION_AD_GROUP_TYPES;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.MAX_LIBRARY_PACKS_COUNT;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

@Component
@ParametersAreNonnullByDefault
public class GetAdGroupsDelegate extends GetApiServiceDelegate<GetRequest, GetResponse, AdGroupAnyFieldEnum,
        AdGroupsSelectionCriteria, AdGroup> {

    private static final Logger logger = LoggerFactory.getLogger(GetAdGroupsDelegate.class);

    private final AdGroupService adGroupService;
    private final GetResponseConverter getResponseConverter;

    @Autowired
    public GetAdGroupsDelegate(ApiAuthenticationSource auth,
                               AdGroupService adGroupService,
                               GetResponseConverter getResponseConverter) {
        super(MappingPathConverter.builder(GetAdGroupsDelegate.class, "emptyConverter").build(), auth);

        this.adGroupService = adGroupService;
        this.getResponseConverter = getResponseConverter;
    }

    @Override
    public ValidationResult<GetRequest, DefectType> validateRequest(GetRequest request) {
        logger.debug("validate request");

        ItemValidationBuilder<GetRequest, DefectType> vb = ItemValidationBuilder.of(request);

        vb.item(request.getSelectionCriteria(), SELECTION_CRITERIA.schemaName.toString())
                .check(notNull())
                .checkBy(this::validateSelectionCriteria, When.isValid());

        vb.checkBy(GetRequestGeneralValidator::validateRequestWithDefectTypes, When.notNull());

        return vb.getResult();
    }

    @Override
    public List<Long> returnedCampaignIds(ApiResult<List<AdGroup>> result) {
        return mapList(result.getResult(), AdGroup::getCampaignId);
    }

    private ValidationResult<com.yandex.direct.api.v5.adgroups.AdGroupsSelectionCriteria, DefectType>
    validateSelectionCriteria(
            com.yandex.direct.api.v5.adgroups.AdGroupsSelectionCriteria selectionCriteria) {
        ItemValidationBuilder<com.yandex.direct.api.v5.adgroups.AdGroupsSelectionCriteria, DefectType> vb =
                ItemValidationBuilder.of(selectionCriteria);

        vb.check(Constraint.fromPredicate(sc -> !sc.getIds().isEmpty() || !sc.getCampaignIds().isEmpty(),
                missedParamsInSelection(String.join(
                        ", ", asList(IDS.schemaName.toString(), CAMPAIGN_IDS.schemaName.toString())))));
        /*
            NB: если SELECTION_CRITERIA.IDS / SELECTION_CRITERIA.CAMPAIGN_IDS не был передан или был передан
            пустой, то AdGroupsSelectionCriteria.getIds() / AdGroupsSelectionCriteria.getCampaignIds() всегда
            возвращает пустой список, в следствии чего невозможно сделать проверки на минимальное количество
            элементов в этом списке (нет возможности проверить был ли список передан в запросе и провалидировать
            его размер)

            vb.item(selectionCriteria.getIds(), IDS).check(minListSize(DEFAULT_MIN_IDS_COUNT))

            vb.item(selectionCriteria.getCampaignIds(), CAMPAIGN_IDS).check(minListSize(MIN_CAMPAIGN_IDS_COUNT))
        */

        vb.item(selectionCriteria.getIds(), IDS.schemaName.toString())
                .check(maxListSize(DEFAULT_MAX_IDS_COUNT), maxIdsInSelection());

        vb.item(selectionCriteria.getCampaignIds(), CAMPAIGN_IDS.schemaName.toString())
                .check(maxListSize(Constants.MAX_CAMPAIGN_IDS_COUNT), maxIdsInSelection());

        vb.item(selectionCriteria.getNegativeKeywordSharedSetIds(), NEGATIVE_KEYWORD_SHARED_SET_IDS.schemaName
                .toString())
                .check(maxListSize(MAX_LIBRARY_PACKS_COUNT), maxIdsInSelection());

        vb.item(selectionCriteria.getPromotedContentTypes(), PROMOTED_CONTENT_TYPES.schemaName.toString())
                .check(supportedContentPromotion());

        return vb.getResult();
    }

    private Constraint<List<PromotedContentTypeEnum>, DefectType> supportedContentPromotion() {
        boolean isServicesApp = getAuth().isServicesApplication();
        return fromPredicate(
                types -> isServicesApp || !types.contains(PromotedContentTypeEnum.SERVICE),
                invalidUseOfField()
        );
    }

    @Override
    public AdGroupsSelectionCriteria extractSelectionCriteria(GetRequest externalRequest) {
        logger.debug("extract selection criteria");

        com.yandex.direct.api.v5.adgroups.AdGroupsSelectionCriteria selectionCriteria =
                externalRequest.getSelectionCriteria();

        return new AdGroupsSelectionCriteria()
                .withAdGroupIds(new HashSet<>(selectionCriteria.getIds()))
                .withCampaignIds(new HashSet<>(selectionCriteria.getCampaignIds()))
                .withAdGroupTypes(GetRequestConverter.convertTypes(selectionCriteria.getTypes()))
                .withAdGroupStatuses(GetRequestConverter.convertStatuses(selectionCriteria.getStatuses()))
                .withAdGroupAppIconStatuses(
                        GetRequestConverter.convertAppIconStatuses(selectionCriteria.getAppIconStatuses()))
                .withAdGroupServingStatuses(
                        GetRequestConverter.convertServingStatuses(selectionCriteria.getServingStatuses()))
                .withNegativeKeywordSharedSetIds(new HashSet<>(selectionCriteria.getNegativeKeywordSharedSetIds()))
                .withContentPromotionAdgroupTypes(listToSet(selectionCriteria.getPromotedContentTypes(),
                        GetRequestConverter::convertContentPromotionType));
    }

    @Override
    public Set<AdGroupAnyFieldEnum> extractFieldNames(GetRequest externalRequest) {
        logger.debug("extract field names");

        return Stream.of(
                externalRequest.getFieldNames().stream().map(AdGroupAnyFieldEnum::fromAdGroupFieldEnum),
                externalRequest.getMobileAppAdGroupFieldNames().stream()
                        .map(AdGroupAnyFieldEnum::fromMobileAppAdGroupFieldEnum),
                externalRequest.getDynamicTextAdGroupFieldNames().stream()
                        .map(AdGroupAnyFieldEnum::fromDynamicTextAdGroupFieldEnum),
                externalRequest.getDynamicTextFeedAdGroupFieldNames().stream()
                        .map(AdGroupAnyFieldEnum::fromDynamicTextFeedAdGroupFieldEnum),
                externalRequest.getSmartAdGroupFieldNames().stream()
                        .map(AdGroupAnyFieldEnum::fromSmartAdGroupFieldEnum),
                externalRequest.getContentPromotionAdGroupFieldNames().stream()
                        .map(AdGroupAnyFieldEnum::fromContentPromotionAdGroupFieldEnum),
                externalRequest.getTextAdGroupFeedParamsFieldNames().stream()
                        .map(AdGroupAnyFieldEnum::fromTextAdGroupFeedParamsFieldEnum)
        ).flatMap(Function.identity()).collect(toSet());
    }

    @Override
    public List<AdGroup> get(GenericGetRequest<AdGroupAnyFieldEnum, AdGroupsSelectionCriteria> getRequest) {
        logger.debug("seek adgroups in repository");

        Long operatorUid = auth.getOperator().getUid();
        ClientId clientId = auth.getChiefSubclient().getClientId();
        AdGroupsSelectionCriteria selectionCriteria = getRequest.getSelectionCriteria();

        Set<AdGroupType> allowedAdGroupTypes = getAllowedAdGroupTypes();
        if (selectionCriteria.getAdGroupTypes().isEmpty()) {
            selectionCriteria.withAdGroupTypes(allowedAdGroupTypes);
        } else {
            Set<AdGroupType> filteredAdGroupTypes = selectionCriteria.getAdGroupTypes()
                    .stream()
                    .filter(allowedAdGroupTypes::contains)
                    .collect(toSet());
            selectionCriteria.withAdGroupTypes(filteredAdGroupTypes);
        }

        List<AdGroup> adGroups = new ArrayList<>(adGroupService.getAdGroupsBySelectionCriteria(
                operatorUid, clientId, selectionCriteria, getRequest.getLimitOffset(), true));
        adGroupService.enrichPerformanceAdGroups(clientId, adGroups);

        // если не было фильтрации по ContentPromotionAdgroupType,
        // то отфильтровываем неподдерживаемые типы продвижения контента
        if (selectionCriteria.getContentPromotionAdgroupTypes().isEmpty()) {
            adGroups.removeIf(this::badContentPromotionGroups);
        }
        return adGroups;
    }

    /**
     * Метод для определения "плохих" групп типа content_promotion.
     * "Плохая" == не поддерживается в API или поддерживается, но нет доступа у текущего оператора
     */
    private boolean badContentPromotionGroups(AdGroup group) {
        if (group.getType() != AdGroupType.CONTENT_PROMOTION) {
            // не content promotion -- всё хорошо
            return false;
        }
        ContentPromotionAdgroupType type = ((ContentPromotionAdGroup) group).getContentPromotionType();
        boolean allowedContentType = API5_ALLOWED_CONTENT_PROMOTION_AD_GROUP_TYPES.contains(type);
        if (!allowedContentType) {
            // не поддерживается в API
            return true;
        }
        boolean isServicesApp = getAuth().isServicesApplication();
        boolean noAccessToTypeService = type == ContentPromotionAdgroupType.SERVICE && !isServicesApp;
        if (noAccessToTypeService) {
            // тип продвижения "Услуги" недоступен текущему приложению
            return true;
        }
        return false;
    }

    @Override
    public GetResponse convertGetResponse(List<AdGroup> result, Set<AdGroupAnyFieldEnum> requestedFields,
                                          @Nullable Long limitedBy) {
        logger.debug("convert response");

        GetResponse response = new GetResponse().withLimitedBy(limitedBy);

        if (!result.isEmpty()) {
            List<AdGroupGetItem> getItems = mapList(result, getResponseConverter::convertToExternalItem);

            getResponseConverter.filterProperties(getItems, requestedFields);

            response.withAdGroups(getItems);
        }

        return response;
    }
}
