package ru.yandex.direct.api.v5.entity.ads;

import java.util.Objects;

import ru.yandex.direct.api.v5.common.ConverterUtils;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.api.v5.validation.DefectTypes;
import ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefectTranslations;
import ru.yandex.direct.core.validation.CommonDefectTranslations;
import ru.yandex.direct.i18n.types.Identity;
import ru.yandex.direct.queryrec.model.Language;
import ru.yandex.direct.validation.defect.params.StringDefectParams;
import ru.yandex.direct.validation.result.Path;

import static ru.yandex.direct.api.v5.entity.ads.Constants.AGE_VALUES;
import static ru.yandex.direct.api.v5.entity.ads.Constants.BABY_FOOD_VALUES;
import static ru.yandex.direct.api.v5.validation.DefectTypes.badStatus;
import static ru.yandex.direct.api.v5.validation.DefectTypes.duplicatedArrayItem;
import static ru.yandex.direct.api.v5.validation.DefectTypes.emptyValue;
import static ru.yandex.direct.api.v5.validation.DefectTypes.inconsistentLanguageAndGeo;
import static ru.yandex.direct.api.v5.validation.DefectTypes.inconsistentState;
import static ru.yandex.direct.api.v5.validation.DefectTypes.invalidChars;
import static ru.yandex.direct.api.v5.validation.DefectTypes.invalidFormat;
import static ru.yandex.direct.api.v5.validation.DefectTypes.invalidId;
import static ru.yandex.direct.api.v5.validation.DefectTypes.invalidUseOfField;
import static ru.yandex.direct.api.v5.validation.DefectTypes.invalidValue;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxCollectionSize;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxElementsExceeded;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxElementsPerRequest;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxStringSize;
import static ru.yandex.direct.api.v5.validation.DefectTypes.mixedOperations;
import static ru.yandex.direct.api.v5.validation.DefectTypes.noRights;
import static ru.yandex.direct.api.v5.validation.DefectTypes.notFound;
import static ru.yandex.direct.api.v5.validation.DefectTypes.notSupported;
import static ru.yandex.direct.api.v5.validation.DefectTypes.requiredAtLeastOneOfFields;
import static ru.yandex.direct.api.v5.validation.DefectTypes.requiredButEmpty;
import static ru.yandex.direct.api.v5.validation.DefectTypes.warningAlreadySuspended;
import static ru.yandex.direct.api.v5.validation.DefectTypes.warningDuplicated;
import static ru.yandex.direct.api.v5.validation.DefectTypes.warningNoEffect;
import static ru.yandex.direct.api.v5.validation.DefectTypes.warningNotSuspended;
import static ru.yandex.direct.api.v5.validation.DefectTypes.warningObjectsAlreadyAssigned;
import static ru.yandex.direct.api.v5.validation.DefectTypes.warningObjectsAreNotAssigned;
import static ru.yandex.direct.core.entity.region.RegionConstants.LANGUAGE_TO_ALLOWED_REGIONS_CONTAINER;

public class AdsDefectTypes {

    private AdsDefectTypes() {
    }

    public static DefectType maxHrefLength(int maxLength) {
        return invalidValue().withDetailedMessage(translations().maxHrefLengthDetailed(new Identity(maxLength)));
    }

    public static DefectType maxLengthWord(int maxLength) {
        return invalidUseOfField()
                .withDetailedMessage((path, value) ->
                        translations().maxWordLengthDetailed(path, new Identity(maxLength), Objects.toString(value)));
    }

    public static DefectType illegalComma() {
        return invalidUseOfField()
                .withDetailedMessage((path, value) -> translations().improperUseOfPunctuationMarks(path));
    }

    public static DefectType insufficientRights() {
        return noRights().withDetailedMessage(translations().insufficientRights());
    }

    public static DefectType invalidHref() {
        return invalidValue().withDetailedMessage(translations().invalidHrefDetailed());
    }

    public static DefectType invalidCharsInDisplayHref(String illegalSubstring) {
        return invalidChars().withDetailedMessage(translations().invalidCharsDisplayHrefDetailed(illegalSubstring));
    }

    public static DefectType duplicatedSpecCharsInDisplayHref() {
        return invalidFormat().withDetailedMessage(translations().duplicateSpecCharsInDisplayHrefDetailed());
    }

    public static DefectType displayHrefLengthExceeded(StringDefectParams t) {
        return maxStringSize(t.getMaxLength())
                .withDetailedMessage((path, value) -> commonTranslations()
                        .lengthOfFieldValueMustNotExceed(path, String.valueOf(t.getMaxLength())));
    }

    public static DefectType invalidTemplateQuery() {
        return invalidValue().withDetailedMessage(translations().invalidTemplateQuery());
    }

    public static DefectType cannotHaveTemplate() {
        return invalidValue().withDetailedMessage(translations().cannotHaveTemplate());
    }

    public static DefectType invalidSitelinkSetIdUsage() {
        return inconsistentState()
                .withDetailedMessage(translations().inconsistentStateSitelinkSetIdWithNoHrefDetailed());
    }

    public static DefectType invalidDisplayHrefUsage() {
        return inconsistentState().withDetailedMessage(translations().inconsistentStateDisplayHrefWithNoHrefDetailed());
    }

    public static DefectType emptyHref() {
        return emptyValue().withDetailedMessage(translations().emptyHrefDetailed());
    }

    public static DefectType emptyDisplayHref() {
        return invalidFormat()
                .withDetailedMessage((path, value) -> commonTranslations().emptyDetailed(path));
    }

    public static DefectType trackingSystemDomainNotSupported() {
        return invalidUseOfField().withDetailedMessage(translations().trackingSystemDomainNotSupported());
    }

    public static DefectType trackingSystemDomainSupportsHttpsOnly() {
        return invalidUseOfField().withDetailedMessage(translations().trackingSystemDomainSupportsHttpsOnly());
    }

    public static DefectType unsupportedBannerType() {
        return notSupported().withDetailedMessage(translations().unsupportedBannerType());
    }

    public static DefectType unsupportedButtonAction() {
        return notSupported().withDetailedMessage(translations().unsupportedButtonAction());
    }

    public static DefectType inconsistentBannerType() {
        return inconsistentState().withDetailedMessage(translations().inconsistentBannerType());
    }

    public static DefectType requiredButEmptyHrefOrVcardId() {
        return inconsistentState().withDetailedMessage(translations().requiredOneOfHrefOrVcardIdDetailed());
    }

    public static DefectType requiredButEmptyHrefOrTurbolandingId() {
        return inconsistentState().withDetailedMessage(translations().requiredOneOfHrefOrTurbolandingIdDetailed());
    }

    public static DefectType requiredButEmptyTurbolandingId() {
        return inconsistentState().withDetailedMessage(translations().requiredTurbolandingIdDetailed());
    }

    public static DefectType cannotUnarchiveBannerWhenParamsIncompatibleWithPayForConversionStrategy() {
        return inconsistentState().withDetailedMessage(translations().cannotUnarchiveBannerWhenParamsIsIncompatibleWithPayForConversionStrategy());
    }

    public static DefectType cannotUnarchiveBannerWhenSitelinksParamsIncompatibleWithPayForConversionStrategy() {
        return inconsistentState().withDetailedMessage(translations().cannotUnarchiveBannerWhenSitelinksParamsIsIncompatibleWithPayForConversionStrategy());
    }

    public static DefectType inconsistentBannerTypeAndAdgroupType() {
        return inconsistentState()
                .withDetailedMessage(translations().inconsistentBannerTypeAndGroupTypeDetailed());
    }

    public static DefectType inconsistentStateBannerTypeAndImageType() {
        return inconsistentState()
                .withDetailedMessage(translations().inconsistentStateBannerTypeAndImageSizeDetailed());
    }

    public static DefectType inconsistentStateBannerTypeAndTurbolandingType() {
        return inconsistentState()
                .withDetailedMessage(translations().inconsistentStateBannerTypeAndTurbolandingTypeDetailed());
    }

    public static DefectType cannotUpdateImageSize() {
        return inconsistentState().withDetailedMessage(translations().cannotUpdateImageSize());
    }

    public static DefectType actionInArchivedCampaign() {
        return badStatus().withDetailedMessage(translations().cannotUpdateArchivedCampaignDetailed());
    }

    public static DefectType addToArchivedAdGroupNotAllowed() {
        return badStatus().withDetailedMessage(translations().addToArchivedCampaignNotAllowedDetailed());
    }

    public static DefectType cannotUpdateArchivedAd() {
        return badStatus().withDetailedMessage(translations().cannotUpdateArchivedAd());
    }

    public static DefectType maxBannersInAdgroup(Integer maxBannersInAdGroup) {
        return maxElementsExceeded(maxBannersInAdGroup)
                .withDetailedMessage(translations().maxBannersInGroupDetailed(new Identity(maxBannersInAdGroup)));
    }

    public static DefectType maxBannersInInternalCampaign(Integer maxBannersInInternalCampaign) {
        return maxElementsExceeded(maxBannersInInternalCampaign)
                .withDetailedMessage(translations().maxBannersInInternalCampaignDetailed(new Identity(maxBannersInInternalCampaign)));
    }

    public static DefectType maxBannersInUniversalAppCampaign(Integer maxBannersInUniversalAppCampaign) {
        return maxElementsExceeded(maxBannersInUniversalAppCampaign)
                .withDetailedMessage(translations().maxBannersInUniversalAppCampaignDetailed(new Identity(maxBannersInUniversalAppCampaign)));
    }

    public static DefectType inconsistentLanguageWithGeo(Language language) {
        return inconsistentLanguageAndGeo()
                .withDetailedMessage(LANGUAGE_TO_ALLOWED_REGIONS_CONTAINER.get(language).getBannerDefectMessage());
    }

    public static DefectType imageNotFound() {
        return notFound().withDetailedMessage(translations().imageNotFoundDetailed());
    }

    public static DefectType sitelinksSetNotFound() {
        return notFound().withDetailedMessage(translations().sitelinksSetNotFoundDetailed());
    }

    public static DefectType vcardNotFound() {
        return notFound().withDetailedMessage(translations().vcardNotFoundDetailed());
    }

    public static DefectType creativeNotFound() {
        return notFound().withDetailedMessage(translations().creativeNotFound());
    }

    public static DefectType videoExtNotFound() {
        return notFound().withDetailedMessage(translations().videoExtNotFoundDetailed());
    }

    public static DefectType cpcVideoNotFound() {
        return notFound().withDetailedMessage(translations().cpcVideoNotFoundDetailed());
    }

    public static DefectType turboPageNotFound() {
        return notFound().withDetailedMessage(translations().turboPageNotFoundDetailed());
    }


    public static DefectType adNotFound() {
        return notFound().withDetailedMessage(translations().adNotFoundDetailed());
    }

    public static DefectType adGroupNotFound() {
        return notFound().withDetailedMessage(translations().adGroupNotFoundDetailed());
    }

    public static DefectType adExtensionNotFound(long id) {
        return notFound().withDetailedMessage(translations().adExtensionNotFoundDetailed(new Identity(id)));
    }

    public static DefectType notSupportedAdGroupType() {
        return notSupported().withDetailedMessage(commonTranslations().adGroupTypeNotSupportedDetailed());
    }

    public static DefectType maxBannersPerAddRequest(int max) {
        return maxElementsPerRequest(max)
                .withDetailedMessage(translations().maxBannersPerAddRequestDetailed(new Identity(max)));
    }

    public static DefectType maxBannersPerUpdateRequest(int max) {
        return maxElementsPerRequest(max)
                .withDetailedMessage(translations().maxBannersPerUpdateRequestDetailed(new Identity(max)));
    }

    public static DefectType maxBannersPerResumeRequest(int max) {
        return maxElementsPerRequest(max)
                .withDetailedMessage(translations().maxBannersPerResumeRequestDetailed(new Identity(max)));
    }

    public static DefectType maxBannersPerSuspendRequest(int max) {
        return maxElementsPerRequest(max)
                .withDetailedMessage(translations().maxBannersPerSuspendRequestDetailed(new Identity(max)));
    }

    public static DefectType maxBannersPerArchiveRequest(int max) {
        return maxElementsPerRequest(max)
                .withDetailedMessage(translations().maxBannersPerArchiveRequestDetailed(new Identity(max)));
    }

    public static DefectType maxBannersPerUnarchiveRequest(int max) {
        return maxElementsPerRequest(max)
                .withDetailedMessage(translations().maxBannersPerUnarchiveRequestDetailed(new Identity(max)));
    }

    public static DefectType invalidIdInArrayElement() {
        return invalidId()
                .withDetailedMessage((path, value) -> {
                    Path converted = ConverterUtils.convertArrayIndices(path);
                    return translations().invalidIdInArrayElement(converted);
                });
    }

    public static DefectType invalidIdInField() {
        return invalidId()
                .withDetailedMessage((path, value) -> {
                    Path converted = ConverterUtils.convertArrayIndices(path);
                    return commonTranslations().invalidValueShouldBePositiveDetailed(converted);
                });
    }

    public static DefectType vcardOfAnotherCampaign() {
        return inconsistentState().withDetailedMessage(translations().vcardOfAnotherCampaignDetailed());
    }

    /**
     * Для Ads.Add
     */
    public static DefectType maxAdExtensionsExceeded(int max) {
        return maxCollectionSize(max)
                .withDetailedMessage(translations().adExtensionsLimitExceededDetailed(new Identity(max)));
    }

    /**
     * Для Ads.Update
     */
    public static DefectType maxAdExtensionsOnBannerExceeded(int max) {
        return maxCollectionSize(max)
                .withDetailedMessage(translations().adExtensionsPerBannerLimitExceededDetailed(new Identity(max)));
    }

    public static DefectType duplicatedAdExtensionId(Long id) {
        return duplicatedArrayItem()
                .withDetailedMessage((path, value) ->
                        commonTranslations()
                                .duplicatedValueDetailedWithList(new Identity(id), path.getFieldName()));
    }

    public static DefectType invalidCalloutsModification() {
        return mixedOperations().withDetailedMessage(translations().invalidCalloutsModificationDetailed());
    }

    public static DefectType duplicatedValue(Long value) {
        return duplicatedArrayItem()
                .withDetailedMessage((path, v) ->
                        commonTranslations().duplicatedValueDetailedWithList(new Identity(value), path.toString()));
    }

    public static DefectType emptyValueInField() {
        return emptyValue().withDetailedMessage((path, value) -> translations().emptyValueInField(path));
    }

    public static DefectType warningAdExtensionAlreadyAssigned(Long id) {
        return warningObjectsAlreadyAssigned().withDetailedMessage(
                translations().warningAdExtensionAlreadyAssigned(new Identity(id)));
    }

    public static DefectType warningAdExtensionIsNotAssigned(Long id) {
        return warningObjectsAreNotAssigned().withDetailedMessage(
                translations().warningAdExtensionIsNotAssigned(new Identity(id)));
    }

    public static DefectType maxNumberOfNarrowCharacters(int maxLength) {
        return invalidUseOfField()
                .withDetailedMessage((path, value) ->
                        translations().maxNumberOfNarrowCharactersInFieldDetailed(path, new Identity(maxLength)));
    }

    public static DefectType suspendInArchivedCampaign(long campaignId) {
        return badStatus().withDetailedMessage(translations()
                .suspendBannerInArchivedCampaign(new Identity(campaignId)));
    }

    public static DefectType resumeInArchivedCampaign(long campaignId) {
        return badStatus().withDetailedMessage(translations()
                .resumeBannerInArchivedCampaign(new Identity(campaignId)));
    }

    public static DefectType cannotSuspendArchivedBanner() {
        return badStatus().withDetailedMessage(translations().cannotSuspendArchivedBanner());
    }

    public static DefectType cannotResumeArchivedBanner() {
        return badStatus().withDetailedMessage(translations().cannotResumeArchivedBanner());
    }

    public static DefectType cannotSuspendDraftBanner() {
        return badStatus().withDetailedMessage(translations().cannotSuspendDraftBanner());
    }

    public static DefectType cannotResumeDraftBanner() {
        return badStatus().withDetailedMessage(translations().cannotResumeDraftBanner());
    }

    public static DefectType bannerAlreadySuspended() {
        return warningAlreadySuspended().withDetailedMessage(translations().alreadySuspended());
    }

    public static DefectType bannerNotSuspended() {
        return warningNotSuspended().withDetailedMessage(translations().isNotSuspended());
    }

    public static DefectType cannotArchiveBannerShownInBs() {
        return badStatus().withDetailedMessage(translations().cannotArchiveBannerShownInBs());
    }

    public static DefectType cannotArchiveDraftBanner() {
        return badStatus().withDetailedMessage(translations().cannotArchiveDraftBanner());
    }

    public static DefectType cannotUnarchiveInArchivedCampaign(Long id) {
        return badStatus().withDetailedMessage(translations().unarchiveInArchivedCampaign(new Identity(id)));
    }

    public static DefectType cannotArchiveInArchivedCampaign(Long id) {
        return badStatus().withDetailedMessage(translations().archiveInArchivedCampaign(new Identity(id)));
    }

    public static DefectType cannotChangeBannerFlagsFromBabyFoodToOtherTypes() {
        return inconsistentState().withDetailedMessage(
                translations().onlyTheFollowingAgeLabelValuesAreAvailable(BABY_FOOD_VALUES));
    }

    public static DefectType cannotChangeBannerFlagsFromAgeToOtherTypes() {
        return inconsistentState().withDetailedMessage(
                translations().onlyTheFollowingAgeLabelValuesAreAvailable(AGE_VALUES));
    }

    public static DefectType clientCannotSetBannerFlags() {
        return inconsistentState().withDetailedMessage(
                translations().clientCannotSetBannerFlags());
    }

    public static DefectType adMustContainOneOfTheTypeObjects(String allowedAdTypes) {
        return requiredAtLeastOneOfFields().withDetailedMessage(
                translations().adMustContainOneOfTheTypeObjects(allowedAdTypes));
    }

    public static DefectType unableToDelete() {
        return badStatus().withDetailedMessage(translations().unableToDeleteDetailed());
    }

    public static DefectType maxIdsToDelete() {
        return DefectTypes.maxElementsInSelection(Constants.MAX_IDS_PER_DELETE)
                .withDetailedMessage(translations()
                        .maxBannersPerDeleteRequestDetailed(new Identity(Constants.MAX_IDS_PER_DELETE)));
    }

    public static DefectType adMoreThanOnceInRequest() {
        return warningDuplicated().withDetailedMessage(translations().adMoreThanOnceInRequest());
    }

    public static DefectType warningAdAlreadySuspended() {
        return new DefectType(10200,
                translations().adAlreadySuspendedShort());
    }

    public static DefectType warningAdNotSuspended() {
        return new DefectType(10201,
                translations().adNotSuspendedShort());
    }

    public static DefectType warningAdAlreadyArchived() {
        return new DefectType(10202,
                translations().alreadyArchived());
    }

    public static DefectType warningAdNotArchived() {
        return new DefectType(10203,
                translations().isNotArchived());
    }

    public static DefectType warningTrackingUrlDoesntContainLogid() {
        return new DefectType(10250, translations().trackingUrlDoesntContainLogid());
    }

    public static DefectType invalidPixelFormat() {
        return invalidFormat().withDetailedMessage(translations().invalidPixelFormat());
    }

    public static DefectType noRightsToPixel(String url) {
        return invalidFormat().withDetailedMessage(translations().noRightsToPixel(url));
    }

    public static DefectType equalPixelProviders(String providerNamesJoined) {
        return invalidFormat().withDetailedMessage(translations().equalPixelProviders(providerNamesJoined));
    }

    public static DefectType noRightsToAudiencePixel(String url) {
        return invalidFormat().withDetailedMessage(translations().noRightsToAudiencePixel(url));
    }

    public static DefectType invalidPixelWithMeasurer(String url, String measurerSystemName) {
        return invalidFormat().withDetailedMessage(translations().invalidPixelWithMeasurer(url, measurerSystemName));
    }

    public static DefectType maxYaAudiencePixelNumberOnBannerExceeded() {
        return inconsistentState()
                .withDetailedMessage(translations().maxYaAudiencePixelsOnBannerExceeded());
    }

    public static DefectType maxNonYaAudiencePixelNumberOnBannerExceeded() {
        return inconsistentState()
                .withDetailedMessage(translations().maxNonYaAudiencePixelsOnBannerExceeded());
    }

    public static BannerDefectTranslations translations() {
        return BannerDefectTranslations.INSTANCE;
    }

    private static CommonDefectTranslations commonTranslations() {
        return CommonDefectTranslations.INSTANCE;
    }

    public static DefectType maxIdsToModerate() {
        return maxElementsPerRequest(Constants.MAX_IDS_PER_MODERATE)
                .withDetailedMessage(translations()
                        .maxBannersPerModerateRequestDetailed(new Identity(Constants.MAX_IDS_PER_MODERATE)));
    }

    public static DefectType moderateBannerInAdGroupWithoutShowConditions() {
        return badStatus().withDetailedMessage(translations().moderateBannerInAdGroupWithoutShowConditions());
    }

    public static DefectType moderateArchivedBanner() {
        return badStatus().withDetailedMessage(translations().moderateArchivedBanner());
    }

    public static DefectType moderateNonDraftBanner() {
        return badStatus().withDetailedMessage(translations().moderateNonDraftBanner());
    }

    public static DefectType remoderateDraftBanner() {
        return badStatus().withDetailedMessage(translations().remoderateDraftBanner());
    }

    public static DefectType moderateBannerInArchivedCampaign(Long id) {
        return badStatus().withDetailedMessage(translations().moderateBannerInArchivedCampaign(new Identity(id)));
    }

    public static DefectType remoderateBannerInDraftCampaign(Long id) {
        return badStatus().withDetailedMessage(translations().remoderateBannerInDraftCampaign(new Identity(id)));
    }

    public static DefectType remoderateBannerIsNotAllowed() {
        return badStatus().withDetailedMessage(translations().remoderateBannerIsNotAllowed());
    }

    public static DefectType remoderateBannerWithNotAllowedModReasons() {
        return badStatus().withDetailedMessage(translations().remoderateBannerWithNotAllowedModReasons());
    }

    public static DefectType remoderateBannerWithNoModReasons() {
        return badStatus().withDetailedMessage(translations().remoderateBannerWithNoModReasons());
    }

    public static DefectType updatingCpmOutdoorBannerIsForbidden() {
        return badStatus().withDetailedMessage(translations().updatingCpmOutdoorBannerIsForbidden());
    }

    public static DefectType contentTypeNotMatchesAdGroupContentType() {
        return invalidValue().withDetailedMessage(translations().contentTypeNotMatchesAdGroupContentType());
    }

    public static DefectType contentPromotionVideoIsInaccessible() {
        return invalidValue().withDetailedMessage(translations().contentPromotionVideoIsInaccessible());
    }

    public static DefectType contentPromotionCollectionIsInaccessible() {
        return invalidValue().withDetailedMessage(translations().contentPromotionCollectionIsInaccessible());
    }

    public static DefectType bannerPriceMissing() {
        return requiredButEmpty().withDetailedMessage(translations().bannerPriceMissing());
    }

    public static DefectType bannerPriceCurrencyMissing() {
        return requiredButEmpty().withDetailedMessage(translations().bannerPriceCurrencyMissing());
    }

    public static DefectType bannerPriceGreaterThanOld() {
        return invalidValue().withDetailedMessage(translations().bannerPriceGreaterThanOld());
    }

    public static DefectType inconsistentLeadformHrefAndButtonText() {
        return inconsistentState().withDetailedMessage(translations().inconsistentLeadformHrefAndButtonText());
    }

    public static DefectType bannerInThisAdgroupAlreadyExists() {
        return inconsistentState().withDetailedMessage(translations().bannerInThisAdgroupAlreadyExists());
    }

    public static DefectType bannersWithoutCreativeNotEnabled() {
        return invalidValue().withDetailedMessage(translations().bannersWithoutCreativeNotEnabled());
    }

    public static DefectType bannersWithCreativeDeprecated() {
        return warningNoEffect().withDetailedMessage(translations().bannersWithCreativeDeprecated());
    }

    public static DefectType logoIsOnlyForBannersWithoutCreative() {
        return invalidValue().withDetailedMessage(translations().logoIsOnlyForBannersWithoutCreative());
    }

    public static DefectType cannotAddBannerToNoCreativeAdGroup() {
        return inconsistentState().withDetailedMessage(translations().cannotAddBannerToNoCreativeAdGroup());
    }

    public static DefectType incorrectImpressionUrlTrackingSystem() {
        return invalidUseOfField().withDetailedMessage(translations().incorrectImpressionUrlTrackingSystem());
    }

    public static DefectType incorrectImpressionUrlTrackerId() {
        return invalidUseOfField().withDetailedMessage(translations().incorrectImpressionUrlTrackerId());
    }
}
