package ru.yandex.direct.api.v5.entity.ads.converter;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.kscs.util.jaxb.SinglePropertyInfo;
import com.yandex.direct.api.v5.ads.AdAddItem;
import com.yandex.direct.api.v5.ads.AddRequest;
import com.yandex.direct.api.v5.ads.ButtonExtensionAddItem;
import com.yandex.direct.api.v5.ads.ContentPromotionCollectionAdAdd;
import com.yandex.direct.api.v5.ads.ContentPromotionEdaAdAdd;
import com.yandex.direct.api.v5.ads.ContentPromotionServiceAdAdd;
import com.yandex.direct.api.v5.ads.ContentPromotionVideoAdAdd;
import com.yandex.direct.api.v5.ads.CpcVideoAdBuilderAdAdd;
import com.yandex.direct.api.v5.ads.CpmBannerAdBuilderAdAdd;
import com.yandex.direct.api.v5.ads.CpmVideoAdBuilderAdAdd;
import com.yandex.direct.api.v5.ads.DynamicTextAdAdd;
import com.yandex.direct.api.v5.ads.MobileAppAdAdd;
import com.yandex.direct.api.v5.ads.MobileAppAdBuilderAdAdd;
import com.yandex.direct.api.v5.ads.MobileAppCpcVideoAdBuilderAdAdd;
import com.yandex.direct.api.v5.ads.MobileAppImageAdAdd;
import com.yandex.direct.api.v5.ads.SmartAdBuilderAdAdd;
import com.yandex.direct.api.v5.ads.TextAdAdd;
import com.yandex.direct.api.v5.ads.TextAdBuilderAdAdd;
import com.yandex.direct.api.v5.ads.TextImageAdAdd;
import com.yandex.direct.api.v5.ads.VideoExtensionAddItem;
import com.yandex.direct.api.v5.general.ArrayOfString;
import com.yandex.direct.api.v5.general.YesNoEnum;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.entity.ads.validation.AdsAddRequestValidator;
import ru.yandex.direct.core.entity.banner.model.BannerFlags;
import ru.yandex.direct.core.entity.banner.model.BannerWithAdGroupId;
import ru.yandex.direct.core.entity.banner.model.BannerWithButton;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.ButtonAction;
import ru.yandex.direct.core.entity.banner.model.ContentPromotionBanner;
import ru.yandex.direct.core.entity.banner.model.CpcVideoBanner;
import ru.yandex.direct.core.entity.banner.model.CpmBanner;
import ru.yandex.direct.core.entity.banner.model.DynamicBanner;
import ru.yandex.direct.core.entity.banner.model.ImageBanner;
import ru.yandex.direct.core.entity.banner.model.MobileAppBanner;
import ru.yandex.direct.core.entity.banner.model.PerformanceBanner;
import ru.yandex.direct.core.entity.banner.model.PerformanceBannerMain;
import ru.yandex.direct.core.entity.banner.model.TextBanner;

import static com.yandex.direct.api.v5.ads.TextImageAdAdd.PropInfo.BUTTON_EXTENSION;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.api.v5.entity.ads.converter.BannerFlagsConverter.toCoreBannerFlagsAdd;
import static ru.yandex.direct.api.v5.entity.ads.validation.AdsApiValidationSignals.bannerWithAmbiguousType;
import static ru.yandex.direct.api.v5.entity.ads.validation.AdsApiValidationSignals.bannerWithBannerTypeNotSpecified;
import static ru.yandex.direct.api.v5.entity.ads.validation.AdsApiValidationSignals.bannerWithVideoExtensionIdNotSpecified;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Конвертер, формирующий список моделей {@link BannerWithAdGroupId} на основе запроса на добавление.
 *
 * @see AddRequest
 */
@Component
@ParametersAreNonnullByDefault
public class AdsAddRequestConverter {

    public List<BannerWithAdGroupId> convert(AddRequest addRequest) {
        return mapList(addRequest.getAds(), this::convert);
    }

    /**
     * Если {@link AdAddItem} не содержит элементов или содержит более одного,
     * результатом будет баннер специального типа.
     *
     * @see AdsAddRequestValidator
     * @see ru.yandex.direct.api.v5.entity.ads.validation.AdsApiValidationSignals
     */
    @Nullable
    BannerWithAdGroupId convert(AdAddItem item) {
        List<Object> allTypeFields = getAllFields(item);
        if (allAreNull(allTypeFields)) {
            return bannerWithBannerTypeNotSpecified();
        }
        if (moreThanOneIsNotNull(allTypeFields)) {
            return bannerWithAmbiguousType();
        }
        try {
            return tryConvert(item);
        } catch (NoVideoExtIdSpecified ignored) {
            return bannerWithVideoExtensionIdNotSpecified();
        }
    }

    private BannerWithAdGroupId tryConvert(AdAddItem item) {
        BannerWithSystemFields banner;

        if (item.getTextAd() != null) {
            banner = convertTextAd(item);

        } else if (item.getDynamicTextAd() != null) {
            banner = convertDynamicAd(item);

        } else if (item.getTextImageAd() != null) {
            banner = convertImageAd(item);

        } else if (item.getTextAdBuilderAd() != null) {
            banner = convertImageCreativeAd(item);

        } else if (item.getMobileAppAd() != null) {
            banner = convertMobileAppAd(item);

        } else if (item.getMobileAppImageAd() != null) {
            banner = convertMobileImageAd(item);

        } else if (item.getMobileAppCpcVideoAdBuilderAd() != null) {
            banner = convertMobileAppCpcVideoAd(item);

        } else if (item.getMobileAppAdBuilderAd() != null) {
            banner = convertMobileImageCreativeAd(item);

        } else if (item.getCpmBannerAdBuilderAd() != null) {
            banner = convertCpmBannerAdBuilderAd(item);

        } else if (item.getCpcVideoAdBuilderAd() != null) {
            banner = convertCpcVideoAd(item);

        } else if (item.getCpmVideoAdBuilderAd() != null) {
            banner = convertCpmVideoAd(item);
        } else if (item.getSmartAdBuilderAd() != null) {
            banner = convertSmartAd(item);
        } else if (item.getContentPromotionVideoAd() != null) {
            banner = convertContentPromotionVideoAd(item);

        } else if (item.getContentPromotionCollectionAd() != null) {
            banner = convertContentPromotionCollectionAd(item);

        } else if (item.getContentPromotionServiceAd() != null) {
            banner = convertContentPromotionServiceAd(item);

        } else if (item.getContentPromotionEdaAd() != null) {
            banner = convertContentPromotionEdaAd(item);

        } else {
            throw new UnsupportedOperationException("Unexpected ad type");
        }

        return banner.withCampaignId(null).withAdGroupId(item.getAdGroupId());
    }

    private static TextBanner convertTextAd(AdAddItem adAddItem) {
        TextAdAdd textAdAdd = adAddItem.getTextAd();

        TextBanner newTextBanner = new TextBanner();
        convertButton(newTextBanner, textAdAdd, TextAdAdd.PropInfo.BUTTON_EXTENSION);

        return newTextBanner
                .withVcardId(textAdAdd.getVCardId())
                .withSitelinksSetId(textAdAdd.getSitelinkSetId())
                .withLogoImageHash(textAdAdd.getLogoExtensionHash())
                .withImageHash(textAdAdd.getAdImageHash())
                .withCreativeId(convertVideoExtCreativeId(textAdAdd.getVideoExtension()))
                .withShowTitleAndBody(convertVideoExtShowTitleAndBody(textAdAdd.getVideoExtension()))
                .withCalloutIds(textAdAdd.getAdExtensionIds())
                .withIsMobile(textAdAdd.getMobile() == YesNoEnum.YES)
                .withTitle(textAdAdd.getTitle())
                .withTitleExtension(textAdAdd.getTitle2())
                .withBody(textAdAdd.getText())
                .withHref(textAdAdd.getHref())
                .withDisplayHref(textAdAdd.getDisplayUrlPath())
                .withDisplayHrefPrefix(textAdAdd.getDutPrefix())
                .withDisplayHrefSuffix(textAdAdd.getDutSuffix())
                .withBannerPrice(BannerPriceConverter.toCore(textAdAdd.getPriceExtension()))
                .withTurboLandingId(textAdAdd.getTurboPageId())
                .withPermalinkId(textAdAdd.getBusinessId())
                .withPhoneId(textAdAdd.getTrackingPhoneId())
                .withPreferVCardOverPermalink(textAdAdd.getPreferVCardOverBusiness() == YesNoEnum.YES)
                .withLeadformHref(textAdAdd.getLfHref())
                .withLeadformButtonText(textAdAdd.getLfButtonText());
    }

    @Nullable
    private static Long convertVideoExtCreativeId(@Nullable VideoExtensionAddItem videoExt) {
        if (videoExt == null) {
            return null;
        }
        Long creativeId = videoExt.getCreativeId();
        if (creativeId == null) {
            throw new NoVideoExtIdSpecified();
        }
        return creativeId;
    }

    @Nullable
    private static Boolean convertVideoExtShowTitleAndBody(@Nullable VideoExtensionAddItem videoExt) {
        if (videoExt == null) {
            return null;
        }
        YesNoEnum flag = videoExt.getShowTitleAndBody();
        if (flag == null) {
            return null;
        }
        return flag == YesNoEnum.YES;
    }

    private static DynamicBanner convertDynamicAd(AdAddItem adAddItem) {
        DynamicTextAdAdd dynamicAdAdd = adAddItem.getDynamicTextAd();
        return new DynamicBanner()
                .withVcardId(dynamicAdAdd.getVCardId())
                .withSitelinksSetId(dynamicAdAdd.getSitelinkSetId())
                .withImageHash(dynamicAdAdd.getAdImageHash())
                .withCalloutIds(dynamicAdAdd.getAdExtensionIds())
                .withBody(dynamicAdAdd.getText());
    }

    private static ButtonAction convertButtonAction(String action) {
        if (action == null) {
            return null;
        }

        return ButtonAction.fromSource(action);
    }

    private static <T> void convertButton(BannerWithButton bannerWithButton, T item, SinglePropertyInfo<T,
            ButtonExtensionAddItem> property) {
        ButtonExtensionAddItem buttonExtension = property.get(item);

        if (buttonExtension == null) {
            return;
        }

        bannerWithButton.withButtonAction(convertButtonAction(buttonExtension.getAction().value()));
        bannerWithButton.withButtonCaption(buttonExtension.getCustomActionText());
        bannerWithButton.withButtonHref(buttonExtension.getHref());
    }

    private static ImageBanner convertImageAd(AdAddItem adAddItem) {
        TextImageAdAdd textImageAdAdd = adAddItem.getTextImageAd();

        ImageBanner newImageBanner = new ImageBanner();
        convertButton(newImageBanner, textImageAdAdd, BUTTON_EXTENSION);

        return newImageBanner
                .withTitle(textImageAdAdd.getTitle())
                .withTitleExtension(textImageAdAdd.getTitle2())
                .withBody(textImageAdAdd.getText())
                .withLogoImageHash(textImageAdAdd.getLogoExtensionHash())
                .withIsMobileImage(false)
                .withHref(textImageAdAdd.getHref())
                .withImageHash(textImageAdAdd.getAdImageHash())
                .withTurboLandingId(textImageAdAdd.getTurboPageId());
    }

    private static ImageBanner convertImageCreativeAd(AdAddItem adAddItem) {
        TextAdBuilderAdAdd creativeAdAdd = adAddItem.getTextAdBuilderAd();

        ImageBanner newImageBanner = new ImageBanner();
        convertButton(newImageBanner, creativeAdAdd, TextAdBuilderAdAdd.PropInfo.BUTTON_EXTENSION);

        return newImageBanner
                .withTitle(creativeAdAdd.getTitle())
                .withTitleExtension(creativeAdAdd.getTitle2())
                .withBody(creativeAdAdd.getText())
                .withLogoImageHash(creativeAdAdd.getLogoExtensionHash())
                .withIsMobileImage(false)
                .withHref(creativeAdAdd.getHref())
                .withTurboLandingId(creativeAdAdd.getTurboPageId())
                .withCreativeId(
                        creativeAdAdd.getCreative().getCreativeId());   // creative is non-null (wsdl constraint)
    }

    private static MobileAppBanner convertMobileAppAd(AdAddItem adAddItem) {
        MobileAppAdAdd mobileAdAdd = adAddItem.getMobileAppAd();
        return new MobileAppBanner()
                .withBody(mobileAdAdd.getText())
                .withTitle(mobileAdAdd.getTitle())
                .withHref(mobileAdAdd.getTrackingUrl())
                .withPrimaryAction(PrimaryActionConverter.toCorePrimaryAction(mobileAdAdd.getAction()))
                .withReflectedAttributes(ReflectedAttrsConverter.toCoreReflectedAttrs(mobileAdAdd.getFeatures()))
                .withImpressionUrl(mobileAdAdd.getImpressionUrl())
                .withImageHash(mobileAdAdd.getAdImageHash())
                .withFlags(new BannerFlags().with(BannerFlags.AGE, toCoreBannerFlagsAdd(mobileAdAdd.getAgeLabel())))
                .withCreativeId(convertVideoExtCreativeId(mobileAdAdd.getVideoExtension()));
    }

    private static ImageBanner convertMobileImageAd(AdAddItem adAddItem) {
        MobileAppImageAdAdd mobileImageAdAdd = adAddItem.getMobileAppImageAd();
        return new ImageBanner()
                .withIsMobileImage(true)
                .withHref(mobileImageAdAdd.getTrackingUrl())
                .withImageHash(mobileImageAdAdd.getAdImageHash());
    }

    private static CpcVideoBanner convertMobileAppCpcVideoAd(AdAddItem adAddItem) {
        MobileAppCpcVideoAdBuilderAdAdd mobileAppCpcVideoAdBuilderAdAdd = adAddItem.getMobileAppCpcVideoAdBuilderAd();

        return new CpcVideoBanner()
                .withIsMobileVideo(true)
                .withHref(mobileAppCpcVideoAdBuilderAdAdd.getTrackingUrl())
                .withCreativeId(mobileAppCpcVideoAdBuilderAdAdd.getCreative().getCreativeId());
    }

    private static ImageBanner convertMobileImageCreativeAd(AdAddItem adAddItem) {
        MobileAppAdBuilderAdAdd mobileCreativeAdAdd = adAddItem.getMobileAppAdBuilderAd();
        return new ImageBanner()
                .withIsMobileImage(true)
                .withHref(mobileCreativeAdAdd.getTrackingUrl())
                .withCreativeId(
                        mobileCreativeAdAdd.getCreative().getCreativeId()); // creative is non-null (wsdl constraint)
    }

    private static CpmBanner convertCpmBannerAdBuilderAd(AdAddItem adAddItem) {
        CpmBannerAdBuilderAdAdd cpmBannerAdBuilderAdAdd = adAddItem.getCpmBannerAdBuilderAd();
        return new CpmBanner()
                .withHref(cpmBannerAdBuilderAdAdd.getHref())
                .withCreativeId(
                        cpmBannerAdBuilderAdAdd.getCreative()
                                .getCreativeId())    // creative is non-null (wsdl constraint)
                .withPixels(Optional.ofNullable(cpmBannerAdBuilderAdAdd.getTrackingPixels())
                        .map(ArrayOfString::getItems)
                        .orElse(emptyList())
                )
                .withTnsId(cpmBannerAdBuilderAdAdd.getTnsId())
                .withTurboLandingId(cpmBannerAdBuilderAdAdd.getTurboPageId());
    }

    private static CpmBanner convertCpmVideoAd(AdAddItem adAddItem) {
        CpmVideoAdBuilderAdAdd cpmVideoAdBuilderAdAdd = adAddItem.getCpmVideoAdBuilderAd();

        CpmBanner newCpmBanner = new CpmBanner();
        convertButton(newCpmBanner, cpmVideoAdBuilderAdAdd, CpmVideoAdBuilderAdAdd.PropInfo.BUTTON_EXTENSION);

        return newCpmBanner
                .withTitle(cpmVideoAdBuilderAdAdd.getTitle())
                .withTitleExtension(cpmVideoAdBuilderAdAdd.getTitle2())
                .withBody(cpmVideoAdBuilderAdAdd.getText())
                .withHref(cpmVideoAdBuilderAdAdd.getHref())
                .withLogoImageHash(cpmVideoAdBuilderAdAdd.getLogoExtensionHash())
                .withCreativeId(
                        cpmVideoAdBuilderAdAdd.getCreative()
                                .getCreativeId())    // creative is non-null (wsdl constraint)
                .withPixels(Optional.ofNullable(cpmVideoAdBuilderAdAdd.getTrackingPixels())
                        .map(ArrayOfString::getItems)
                        .orElse(emptyList())
                )
                .withTurboLandingId(cpmVideoAdBuilderAdAdd.getTurboPageId());
    }

    private static CpcVideoBanner convertCpcVideoAd(AdAddItem adAddItem) {
        CpcVideoAdBuilderAdAdd cpcVideoAdBuilderAdAdd = adAddItem.getCpcVideoAdBuilderAd();

        return new CpcVideoBanner()
                .withHref(cpcVideoAdBuilderAdAdd.getHref())
                .withCreativeId(cpcVideoAdBuilderAdAdd.getCreative().getCreativeId())
                .withTurboLandingId(cpcVideoAdBuilderAdAdd.getTurboPageId());
    }

    private static BannerWithSystemFields convertSmartAd(AdAddItem adAddItem) {
        SmartAdBuilderAdAdd smartAdBuilderAdAdd = adAddItem.getSmartAdBuilderAd();
        if (smartAdBuilderAdAdd.getCreative() == null || smartAdBuilderAdAdd.getCreative().getCreativeId() == null) {
            return new PerformanceBannerMain()
                    .withLogoImageHash(smartAdBuilderAdAdd.getLogoExtensionHash());
        } else {
            return new PerformanceBanner()
                    .withCreativeId(smartAdBuilderAdAdd.getCreative().getCreativeId());
        }
    }

    private static List<Object> getAllFields(AdAddItem adAddItem) {
        List<Object> ads = new ArrayList<>();
        ads.add(adAddItem.getTextAd());
        ads.add(adAddItem.getDynamicTextAd());
        ads.add(adAddItem.getTextImageAd());
        ads.add(adAddItem.getTextAdBuilderAd());
        ads.add(adAddItem.getMobileAppAd());
        ads.add(adAddItem.getMobileAppImageAd());
        ads.add(adAddItem.getMobileAppCpcVideoAdBuilderAd());
        ads.add(adAddItem.getMobileAppAdBuilderAd());
        ads.add(adAddItem.getCpmBannerAdBuilderAd());
        ads.add(adAddItem.getCpcVideoAdBuilderAd());
        ads.add(adAddItem.getCpmVideoAdBuilderAd());
        ads.add(adAddItem.getSmartAdBuilderAd());
        ads.add(adAddItem.getContentPromotionVideoAd());
        ads.add(adAddItem.getContentPromotionCollectionAd());
        ads.add(adAddItem.getContentPromotionServiceAd());
        ads.add(adAddItem.getContentPromotionEdaAd());
        return ads;
    }

    private static ContentPromotionBanner convertContentPromotionVideoAd(AdAddItem adAddItem) {
        ContentPromotionVideoAdAdd contentPromotionVideoAdAdd = adAddItem.getContentPromotionVideoAd();

        return convertContentPromotionAdCommon(contentPromotionVideoAdAdd.getPromotedContentId(),
                contentPromotionVideoAdAdd.getVisitHref(),
                contentPromotionVideoAdAdd.getTitle(),
                contentPromotionVideoAdAdd.getText());
    }

    private static ContentPromotionBanner convertContentPromotionCollectionAd(AdAddItem adAddItem) {
        ContentPromotionCollectionAdAdd contentPromotionCollectionAdAdd =
                adAddItem.getContentPromotionCollectionAd();

        return convertContentPromotionAdCommon(contentPromotionCollectionAdAdd.getPromotedContentId(),
                contentPromotionCollectionAdAdd.getVisitHref(), null, null);
    }

    private static ContentPromotionBanner convertContentPromotionServiceAd(AdAddItem adAddItem) {
        ContentPromotionServiceAdAdd contentPromotionServiceAdAdd = adAddItem.getContentPromotionServiceAd();

        return convertContentPromotionAdCommon(contentPromotionServiceAdAdd.getPromotedContentId(),
                null, contentPromotionServiceAdAdd.getTitle(), null);
    }

    private static ContentPromotionBanner convertContentPromotionEdaAd(AdAddItem adAddItem) {
        ContentPromotionEdaAdAdd contentPromotionEdaAdAdd = adAddItem.getContentPromotionEdaAd();

        return convertContentPromotionAdCommon(
                contentPromotionEdaAdAdd.getPromotedContentId(),
                null, contentPromotionEdaAdAdd.getTitle(), contentPromotionEdaAdAdd.getText());
    }

    private static ContentPromotionBanner convertContentPromotionAdCommon(Long contentId,
                                                                          @Nullable String visitUrl,
                                                                          @Nullable String title,
                                                                          @Nullable String body) {
        return new ContentPromotionBanner()
                .withContentPromotionId(contentId)
                .withVisitUrl(visitUrl)
                .withTitle(title)
                .withBody(body);
    }

    private static boolean allAreNull(List<Object> elements) {
        return StreamEx.of(elements).allMatch(Objects::isNull);
    }

    private static boolean moreThanOneIsNotNull(List<Object> elements) {
        return StreamEx.of(elements).nonNull().count() > 1;
    }

    private static class NoVideoExtIdSpecified extends IllegalArgumentException {
    }

}
