package ru.yandex.direct.api.v5.entity.ads.converter;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import com.yandex.direct.api.v5.adextensiontypes.AdExtensionSetting;
import com.yandex.direct.api.v5.adextensiontypes.AdExtensionSettingItem;
import com.yandex.direct.api.v5.general.OperationEnum;

import ru.yandex.direct.api.v5.entity.ads.validation.AdsApiValidationSignals;
import ru.yandex.direct.api.v5.entity.ads.validation.AdsApiValidationSignals.Callouts.DefectsContainer;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.api.v5.entity.ads.validation.AdsApiValidationSignals.Callouts.Defect.ALREADY_LINKED;
import static ru.yandex.direct.api.v5.entity.ads.validation.AdsApiValidationSignals.Callouts.Defect.DUPLICATE;
import static ru.yandex.direct.api.v5.entity.ads.validation.AdsApiValidationSignals.Callouts.Defect.INVALID_ID;
import static ru.yandex.direct.api.v5.entity.ads.validation.AdsApiValidationSignals.Callouts.Defect.NOT_LINKED;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.validId;
import static ru.yandex.direct.core.entity.banner.type.callouts.BannerWithCalloutsConstants.MAX_CALLOUTS_COUNT_ON_BANNER;

@ParametersAreNonnullByDefault
class AdsUpdateRequestCalloutsConverter {

    /**
     * Преобразовать исходный список id расширений с помощью установок, переданных пользователем в API.
     * <p>
     * Действия над набором расширений:
     * <ul>
     * <li>{@code ADD} — привязать расширение к объявлению;</li>
     * <li>{@code REMOVE} — отвязать расширение от объявления;</li>
     * <li>{@code SET} — заменить набор расширений и привязать расширение к объявлению в составе нового набора.</li>
     * </ul>
     * <p>
     * Тип {@code SET} несовместим с {@code ADD} и {@code REMOVE}: для всех элементов
     * в массиве AdExtensions должен быть указан тип {@code SET}.
     *
     * @param originalCalloutIds список id расширений исходной модели
     * @param settings           DTO-инструкция к изменениям над списком, переданная в API
     * @return Результирующий список id расширений для отправки в сервис для обновления модели.
     * @see ru.yandex.direct.api.v5.entity.ads.validation.AdsUpdateRequestValidator
     */
    @Nullable
    static List<Long> toCalloutIds(List<Long> originalCalloutIds, @Nullable AdExtensionSetting settings) {
        if (settings == null || settings.getAdExtensions() == null || settings.getAdExtensions().isEmpty()) {
            return null;
        }
        if (settings.getAdExtensions().size() > MAX_CALLOUTS_COUNT_ON_BANNER) {
            return AdsApiValidationSignals.Callouts.LIST_TOO_LONG;
        }
        if (!operationsAreValid(settings.getAdExtensions())) {
            return AdsApiValidationSignals.Callouts.INCOMPATIBLE_OPERATIONS;
        }

        return isReplacing(settings) ? replaceCollection(settings) : modifyCollection(originalCalloutIds, settings);
    }

    private static boolean operationsAreValid(Collection<AdExtensionSettingItem> settings) {
        Predicate<AdExtensionSettingItem> isOperationSet = AdsUpdateRequestCalloutsConverter::isOperationSet;
        boolean all = settings.stream().allMatch(isOperationSet);
        boolean none = settings.stream().noneMatch(isOperationSet);
        return all || none;
    }

    private static boolean isReplacing(AdExtensionSetting settings) {
        return isOperationSet(settings.getAdExtensions().iterator().next());
    }

    private static boolean isOperationSet(AdExtensionSettingItem setting) {
        return setting.getOperation() == OperationEnum.SET;
    }

    private static List<Long> replaceCollection(AdExtensionSetting settings) {
        final DefectsContainer result = new DefectsContainer(new ArrayList<>());
        final Set<Long> seen = new HashSet<>();

        for (int i = 0; i < settings.getAdExtensions().size(); i++) {
            checkState(settings.getAdExtensions().get(i).getOperation() == OperationEnum.SET);
            Long id = settings.getAdExtensions().get(i).getAdExtensionId();

            /*
            Дефекты хранятся не по индексам исходного списка (хотя они запоминаются), а по id,
            и перетираются, если на тот же id повторно добавлен дефект.
            Таким образом, чем выше if в коде, тем выше приоритет дефекта.
             */

            if (!seen.add(id)) {
                result.addDefect(i, id, DUPLICATE);
            } else if (validId().apply(id) != null) {
                result.addDefect(i, id, INVALID_ID);
            } else {
                result.add(id);
            }
        }

        return result.hasDefects() ? result : new ArrayList<>(result);
    }

    private static List<Long> modifyCollection(List<Long> originalCalloutIds, AdExtensionSetting settings) {
        final DefectsContainer result = new DefectsContainer(originalCalloutIds);
        final Set<Long> originalSet = ImmutableSet.copyOf(originalCalloutIds);
        final Set<Long> seen = new HashSet<>();

        for (int i = 0; i < settings.getAdExtensions().size(); i++) {
            OperationEnum operation = settings.getAdExtensions().get(i).getOperation();
            Long id = settings.getAdExtensions().get(i).getAdExtensionId();

            /*
            Дефекты хранятся не по индексам исходного списка (хотя они запоминаются), а по id,
            и перетираются, если на тот же id повторно добавлен дефект.
            Таким образом, чем выше if в коде, тем выше приоритет дефекта.
             */

            if (!seen.add(id)) {
                result.addDefect(i, id, DUPLICATE);
                continue;
            } else if (validId().apply(id) != null) {
                result.addDefect(i, id, INVALID_ID);
                continue;
            }

            if (operation == OperationEnum.ADD) {
                if (originalSet.contains(id)) {
                    result.addDefect(i, id, ALREADY_LINKED);
                } else {
                    result.add(id);
                }

            } else if (operation == OperationEnum.REMOVE) {
                if (originalSet.contains(id)) {
                    result.remove(id);
                } else {
                    result.addDefect(i, id, NOT_LINKED);
                }

            } else {
                throw new IllegalStateException("This ad extension setting is not expected here: " + operation);
            }
        }

        return result.hasDefects() ? result : new ArrayList<>(result);

    }

}
