package ru.yandex.direct.api.v5.entity.ads.converter;

import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.xml.bind.JAXBElement;

import com.google.common.base.Preconditions;
import com.yandex.direct.api.v5.adextensiontypes.AdExtensionSetting;
import com.yandex.direct.api.v5.adextensiontypes.AdExtensionSettingItem;
import com.yandex.direct.api.v5.ads.AdBuilderAdUpdateItem;
import com.yandex.direct.api.v5.ads.AdUpdateItem;
import com.yandex.direct.api.v5.ads.ButtonExtensionUpdateItem;
import com.yandex.direct.api.v5.ads.ContentPromotionCollectionAdUpdate;
import com.yandex.direct.api.v5.ads.ContentPromotionEdaAdUpdate;
import com.yandex.direct.api.v5.ads.ContentPromotionServiceAdUpdate;
import com.yandex.direct.api.v5.ads.ContentPromotionVideoAdUpdate;
import com.yandex.direct.api.v5.ads.CpcVideoAdBuilderAdUpdate;
import com.yandex.direct.api.v5.ads.CpmBannerAdBuilderAdUpdate;
import com.yandex.direct.api.v5.ads.CpmVideoAdBuilderAdUpdate;
import com.yandex.direct.api.v5.ads.DynamicTextAdUpdate;
import com.yandex.direct.api.v5.ads.MobileAppAdBuilderAdUpdate;
import com.yandex.direct.api.v5.ads.MobileAppAdUpdate;
import com.yandex.direct.api.v5.ads.MobileAppCpcVideoAdBuilderAdUpdate;
import com.yandex.direct.api.v5.ads.MobileAppImageAdUpdate;
import com.yandex.direct.api.v5.ads.SmartAdBuilderAdUpdate;
import com.yandex.direct.api.v5.ads.TextAdBuilderAdUpdate;
import com.yandex.direct.api.v5.ads.TextAdUpdate;
import com.yandex.direct.api.v5.ads.TextAdUpdateBase;
import com.yandex.direct.api.v5.ads.TextImageAdUpdate;
import com.yandex.direct.api.v5.ads.UpdateRequest;
import com.yandex.direct.api.v5.general.ArrayOfString;
import com.yandex.direct.api.v5.general.OperationEnum;
import com.yandex.direct.api.v5.general.YesNoEnum;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.entity.ads.AdsUpdateRequestItem;
import ru.yandex.direct.core.entity.banner.model.BannerFlags;
import ru.yandex.direct.core.entity.banner.model.BannerPrice;
import ru.yandex.direct.core.entity.banner.model.BannerWithCallouts;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.ButtonAction;
import ru.yandex.direct.core.entity.banner.model.ContentPromotionBanner;
import ru.yandex.direct.core.entity.banner.model.CpcVideoBanner;
import ru.yandex.direct.core.entity.banner.model.CpmBanner;
import ru.yandex.direct.core.entity.banner.model.DynamicBanner;
import ru.yandex.direct.core.entity.banner.model.ImageBanner;
import ru.yandex.direct.core.entity.banner.model.MobileAppBanner;
import ru.yandex.direct.core.entity.banner.model.PerformanceBanner;
import ru.yandex.direct.core.entity.banner.model.PerformanceBannerMain;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.banner.service.BannerService;
import ru.yandex.direct.model.ModelChanges;

import static java.util.Collections.emptyList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.api.v5.converter.ModelChangesHelper.processJaxbElement;
import static ru.yandex.direct.api.v5.entity.ads.converter.AdsUpdateRequestCalloutsConverter.toCalloutIds;
import static ru.yandex.direct.api.v5.entity.ads.validation.AdsApiValidationSignals.bannerChangesWithAmbiguousType;
import static ru.yandex.direct.api.v5.entity.ads.validation.AdsApiValidationSignals.bannerChangesWithBannerTypeNotSpecified;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Конвертер, формирующий список моделей {@link BannerWithSystemFields} на основе запроса на обновление.
 *
 * @see UpdateRequest
 */
@Component
@ParametersAreNonnullByDefault
public class AdsUpdateRequestConverter {

    private final BannerService bannerService;

    @Autowired
    public AdsUpdateRequestConverter(BannerService bannerService) {
        this.bannerService = bannerService;
    }

    /**
     * Предполагается, что запрос уже провалидирован.
     *
     * @param updateRequest        API-запрос на обновление баннеров
     * @param canChangeBannerFlags имеет особые привилегии в проставлении флагов
     * @return Список изменений моделей для передачи в ядро, в сервис обновления баннеров.
     * @see ru.yandex.direct.api.v5.entity.ads.validation.AdsUpdateRequestValidator
     */
    public List<AdsUpdateRequestItem<BannerWithSystemFields>> convert(UpdateRequest updateRequest, boolean canChangeBannerFlags) {
        // all banner ids
        List<Long> bannerIds = StreamEx.of(updateRequest.getAds())
                .map(AdUpdateItem::getId)
                .toList();

        // callouts
        List<Long> bannersWithCalloutsIds = StreamEx.of(updateRequest.getAds())
                .filter(this::isModifyingCalloutIds)
                .map(AdUpdateItem::getId)
                .toList();
        // age labels
        List<Long> bannersWithAgeLabelModified = StreamEx.of(updateRequest.getAds())
                .filter(this::isModifyingAgeLabel)
                .map(AdUpdateItem::getId)
                .toList();
        // smart banners
        // TODO : убрать префетчинг смартов после переезда на баннеры performance_main
        List<Long> smartBannersIds = StreamEx.of(updateRequest.getAds())
                .filter(t -> t.getSmartAdBuilderAd() != null)
                .map(AdUpdateItem::getId)
                .toList();

        Set<Long> bannerIdsToPreFetch = StreamEx.of(bannersWithCalloutsIds)
                .append(bannersWithAgeLabelModified)
                .append(smartBannersIds)
                .toCollection(LinkedHashSet::new);

        var unmodifiedBanners = bannerService.getBannersByIds(bannerIdsToPreFetch);
        Map<Long, BannerPrice> bannerPricesByBannerId = bannerService.getNewBannerPricesByBannerId(bannerIds);
        var prefetchedBannersMap = StreamEx.of(unmodifiedBanners).toMap(BannerWithSystemFields::getId, identity());

        Map<Long, List<Long>> bannersCalloutIds = StreamEx.of(unmodifiedBanners)
                .select(BannerWithCallouts.class)
                .mapToEntry(BannerWithCallouts::getId, BannerWithCallouts::getCalloutIds)
                .nonNullValues()
                .toMap();

        var bannerFlags = StreamEx.of(unmodifiedBanners)
                .filter(b -> b.getFlags() != null)
                .collect(toMap(BannerWithSystemFields::getId, BannerWithSystemFields::getFlags));
        return mapList(updateRequest.getAds(),
                externalItem -> new AdsUpdateRequestItem<>(externalItem, convert(externalItem,
                        bannersCalloutIds.getOrDefault(externalItem.getId(), emptyList()),
                        bannerFlags.getOrDefault(externalItem.getId(), null),
                        bannerPricesByBannerId, canChangeBannerFlags,
                        prefetchedBannersMap.get(externalItem.getId()))));
    }

    /**
     * @param item элемент списка изменений – изменение одного баннера
     * @return {@code true}, только если элемент содержит изменения списка расширений баннера (callouts),
     * и эти изменения не являются полной заменой списка, а состоят из добавлений и удалений отдельных id.
     * В остальных случаях – {@code false}.
     */
    private boolean isModifyingCalloutIds(AdUpdateItem item) {
        return Optional.ofNullable(ObjectUtils.firstNonNull(item.getTextAd(), item.getDynamicTextAd()))
                .map(TextAdUpdateBase::getCalloutSetting)
                .filter(elem -> !elem.isNil())
                .map(JAXBElement::getValue)
                .map(settings -> !isReplacing(settings))
                .orElse(false);
    }

    private boolean isModifyingAgeLabel(AdUpdateItem item) {
        return Optional.ofNullable(item.getTextAd()).map(upd -> upd.getAgeLabel() != null).orElse(false)
                || Optional.ofNullable(item.getMobileAppAd()).map(upd -> upd.getAgeLabel() != null).orElse(false);
    }

    /**
     * @param item                   элемент коллекции из запроса на обновление
     * @param calloutIds             коллекция id расширений, соответствующих этому элементу
     * @param oldFlags               значение баннер-флагов до обновления
     * @param bannerPricesByBannerId набор цен на товар на баннеры по ID баннера
     * @param canChangeBannerFlags   имеет особые привилегии в проставлении флагов
     * @return Изменения баннера {@link ModelChanges}
     */
    private ModelChanges<BannerWithSystemFields> convert(
            AdUpdateItem item, List<Long> calloutIds, @Nullable BannerFlags oldFlags,
            Map<Long, BannerPrice> bannerPricesByBannerId, boolean canChangeBannerFlags,
            @Nullable BannerWithSystemFields smartBanner) {
        Long id = item.getId();
        ModelChanges<? extends BannerWithSystemFields> result;

        if (ObjectUtils.firstNonNull(getAllFields(item)) == null) {
            return bannerChangesWithBannerTypeNotSpecified();
        }
        if (!onlyOneIsNonNull(getAllFields(item))) {
            return bannerChangesWithAmbiguousType();
        }

        if (item.getTextAd() != null) {
            result = convertTextAd(id, item.getTextAd(), calloutIds, oldFlags, bannerPricesByBannerId.get(id), canChangeBannerFlags);

        } else if (item.getDynamicTextAd() != null) {
            result = convertDynamicAd(id, item.getDynamicTextAd(), calloutIds);

        } else if (item.getTextImageAd() != null) {
            result = convertTextImageAd(id, item.getTextImageAd());

        } else if (item.getTextAdBuilderAd() != null) {
            result = convertImageCreativeAd(id, item.getTextAdBuilderAd());

        } else if (item.getMobileAppAd() != null) {
            result = convertMobileAppAd(id, item.getMobileAppAd(), oldFlags, canChangeBannerFlags);

        } else if (item.getMobileAppImageAd() != null) {
            result = convertMobileImageAd(id, item.getMobileAppImageAd());

        } else if (item.getMobileAppCpcVideoAdBuilderAd() != null) {
            result = convertMobileAppCpcVideoAd(id, item.getMobileAppCpcVideoAdBuilderAd());

        } else if (item.getMobileAppAdBuilderAd() != null) {
            result = convertMobileImageCreativeAd(id, item.getMobileAppAdBuilderAd());

        } else if (item.getCpmBannerAdBuilderAd() != null) {
            result = convertCpmBannerAdBuilderAd(id, item.getCpmBannerAdBuilderAd());

        } else if (item.getCpcVideoAdBuilderAd() != null) {
            result = convertCpcVideoAd(id, item.getCpcVideoAdBuilderAd());

        } else if (item.getCpmVideoAdBuilderAd() != null) {
            result = convertCpmVideoAdBuilderAd(id, item.getCpmVideoAdBuilderAd());

        } else if (item.getSmartAdBuilderAd() != null) {
            Preconditions.checkNotNull(smartBanner);
            result = convertSmartAd(id, item.getSmartAdBuilderAd(), smartBanner);
        } else if (item.getContentPromotionVideoAd() != null) {
            result = convertContentPromotionVideoAd(id, item.getContentPromotionVideoAd());

        } else if (item.getContentPromotionCollectionAd() != null) {
            result = convertContentPromotionCollectionAd(id, item.getContentPromotionCollectionAd());

        } else if (item.getContentPromotionServiceAd() != null) {
            result = convertContentPromotionServiceAd(id, item.getContentPromotionServiceAd());

        } else if (item.getContentPromotionEdaAd() != null) {
            result = convertContentPromotionEdaAd(id, item.getContentPromotionEdaAd());

        } else {
            throw new UnsupportedOperationException("Unexpected ad type");
        }

        return result.castModelUp(BannerWithSystemFields.class);
    }

    private static ModelChanges<TextBanner> convertTextAd(Long id, TextAdUpdate update, List<Long> calloutIds,
                                                          @Nullable BannerFlags oldFlags,
                                                          @Nullable BannerPrice currentPrice,
                                                          boolean canChangeBannerFlags) {
        ModelChanges<TextBanner> mc = new ModelChanges<>(id, TextBanner.class)
                .processNotNull(update.getText(), TextBanner.BODY)
                .processNotNull(update.getTitle(), TextBanner.TITLE)
                .processNotNull(update.getAgeLabel(), TextBanner.FLAGS,
                        ageLabel -> BannerFlagsConverter.toCoreBannerFlagsUpdate(ageLabel, oldFlags, canChangeBannerFlags))
                .processNotNull(update.getPreferVCardOverBusiness(), TextBanner.PREFER_V_CARD_OVER_PERMALINK,
                        preferVCard -> preferVCard.getValue() == YesNoEnum.YES);

        if (update.getVideoExtension() != null) {
            processJaxbElement(mc, update.getVideoExtension().getCreativeId(), TextBanner.CREATIVE_ID);
            mc.processNotNull(
                    ifNotNull(update.getVideoExtension().getShowTitleAndBody(), flag -> flag == YesNoEnum.YES),
                    TextBanner.SHOW_TITLE_AND_BODY
            );
        }

        if (update.getButtonExtension() != null) {
            processJaxbElement(mc, update.getButtonExtension(), TextBanner.BUTTON_CAPTION,
                    e -> ifNotNull(e.getCustomActionText(), JAXBElement::getValue));

            processJaxbElement(mc, update.getButtonExtension(), TextBanner.BUTTON_HREF,
                    ButtonExtensionUpdateItem::getHref);
            processJaxbElement(mc, update.getButtonExtension(), TextBanner.BUTTON_ACTION,
                    e -> convertButtonAction(e.getAction().value()));
        }

        processJaxbElement(mc, update.getAdImageHash(), TextBanner.IMAGE_HASH);
        processJaxbElement(mc, update.getLogoExtensionHash(), TextBanner.LOGO_IMAGE_HASH);
        processJaxbElement(mc, update.getTitle2(), TextBanner.TITLE_EXTENSION);
        processJaxbElement(mc, update.getHref(), TextBanner.HREF);
        processJaxbElement(mc, update.getDisplayUrlPath(), TextBanner.DISPLAY_HREF);
        processJaxbElement(mc, update.getDutPrefix(), TextBanner.DISPLAY_HREF_PREFIX);
        processJaxbElement(mc, update.getDutSuffix(), TextBanner.DISPLAY_HREF_SUFFIX);
        processJaxbElement(mc, update.getSitelinkSetId(), TextBanner.SITELINKS_SET_ID);
        processJaxbElement(mc, update.getVCardId(), TextBanner.VCARD_ID);
        processJaxbElement(mc, update.getCalloutSetting(), TextBanner.CALLOUT_IDS,
                setting -> toCalloutIds(calloutIds, setting));
        processJaxbElement(mc, update.getPriceExtension(), TextBanner.BANNER_PRICE,
                price -> BannerPriceConverter.toCore(price, currentPrice));
        processJaxbElement(mc, update.getTurboPageId(), TextBanner.TURBO_LANDING_ID);
        processJaxbElement(mc, update.getBusinessId(), TextBanner.PERMALINK_ID);
        processJaxbElement(mc, update.getTrackingPhoneId(), TextBanner.PHONE_ID);
        processJaxbElement(mc, update.getLfHref(), TextBanner.LEADFORM_HREF);
        processJaxbElement(mc, update.getLfButtonText(), TextBanner.LEADFORM_BUTTON_TEXT);

        if (mc.isPropChanged(TextBanner.PERMALINK_ID) && mc.getChangedProp(TextBanner.PERMALINK_ID) == null) {
            mc.process(null, TextBanner.PHONE_ID);
        }
        return mc;
    }

    private static ModelChanges<DynamicBanner> convertDynamicAd(Long id, DynamicTextAdUpdate update,
                                                                List<Long> calloutIds) {
        ModelChanges<DynamicBanner> mc = new ModelChanges<>(id, DynamicBanner.class)
                .processNotNull(update.getText(), DynamicBanner.BODY);
        processJaxbElement(mc, update.getAdImageHash(), DynamicBanner.IMAGE_HASH);
        processJaxbElement(mc, update.getSitelinkSetId(), DynamicBanner.SITELINKS_SET_ID);
        processJaxbElement(mc, update.getVCardId(), DynamicBanner.VCARD_ID);
        processJaxbElement(mc, update.getCalloutSetting(), DynamicBanner.CALLOUT_IDS,
                setting -> toCalloutIds(calloutIds, setting));

        return mc;
    }

    private static ButtonAction convertButtonAction(String action) {
        if (action == null) {
            return null;
        }

        return ButtonAction.fromSource(action);
    }

    private static ModelChanges<ImageBanner> convertTextImageAd(Long id, TextImageAdUpdate update) {
        ModelChanges<ImageBanner> mc = new ModelChanges<>(id, ImageBanner.class)
                .processNotNull(update.getAdImageHash(), ImageBanner.IMAGE_HASH)
                .process(false, ImageBanner.IS_MOBILE_IMAGE);
        processJaxbElement(mc, update.getHref(), ImageBanner.HREF);
        processJaxbElement(mc, update.getLogoExtensionHash(), ImageBanner.LOGO_IMAGE_HASH);
        processJaxbElement(mc, update.getTurboPageId(), ImageBanner.TURBO_LANDING_ID);
        processJaxbElement(mc, update.getTitle2(), ImageBanner.TITLE_EXTENSION);
        processJaxbElement(mc, update.getTitle(), ImageBanner.TITLE);
        processJaxbElement(mc, update.getText(), ImageBanner.BODY);

        if (update.getButtonExtension() != null) {
            processJaxbElement(mc, update.getButtonExtension(), ImageBanner.BUTTON_CAPTION,
                    e -> ifNotNull(e.getCustomActionText(), JAXBElement::getValue));
            processJaxbElement(mc, update.getButtonExtension(), ImageBanner.BUTTON_HREF,
                    ButtonExtensionUpdateItem::getHref);
            processJaxbElement(mc, update.getButtonExtension(), ImageBanner.BUTTON_ACTION,
                    e -> convertButtonAction(e.getAction().value()));
        }

        return mc;
    }

    private ModelChanges<ImageBanner> convertImageCreativeAd(Long id, TextAdBuilderAdUpdate update) {
        Long creativeId = getCreativeId(update.getCreative());
        ModelChanges<ImageBanner> mc = new ModelChanges<>(id, ImageBanner.class)
                .processNotNull(creativeId, ImageBanner.CREATIVE_ID)
                .process(false, ImageBanner.IS_MOBILE_IMAGE);
        processJaxbElement(mc, update.getHref(), ImageBanner.HREF);
        processJaxbElement(mc, update.getTurboPageId(), ImageBanner.TURBO_LANDING_ID);
        processJaxbElement(mc, update.getTitle2(), ImageBanner.TITLE_EXTENSION);
        processJaxbElement(mc, update.getTitle(), ImageBanner.TITLE);
        processJaxbElement(mc, update.getText(), ImageBanner.BODY);
        processJaxbElement(mc, update.getLogoExtensionHash(), ImageBanner.LOGO_IMAGE_HASH);

        if (update.getButtonExtension() != null) {
            processJaxbElement(mc, update.getButtonExtension(), ImageBanner.BUTTON_CAPTION,
                    e -> ifNotNull(e.getCustomActionText(), JAXBElement::getValue));
            processJaxbElement(mc, update.getButtonExtension(), ImageBanner.BUTTON_HREF,
                    ButtonExtensionUpdateItem::getHref);
            processJaxbElement(mc, update.getButtonExtension(), ImageBanner.BUTTON_ACTION,
                    e -> convertButtonAction(e.getAction().value()));
        }

        return mc;
    }

    private ModelChanges<MobileAppBanner> convertMobileAppAd(Long id, MobileAppAdUpdate update,
                                                             @Nullable BannerFlags oldFlags,
                                                             boolean canChangeBannerFlags) {
        ModelChanges<MobileAppBanner> changes = new ModelChanges<>(id, MobileAppBanner.class)
                .processNotNull(update.getTitle(), MobileAppBanner.TITLE)
                .processNotNull(update.getText(), MobileAppBanner.BODY)
                .processNotNull(update.getAction(), MobileAppBanner.PRIMARY_ACTION,
                        PrimaryActionConverter::toCorePrimaryAction)
                .processNotNull(update.getFeatures(), MobileAppBanner.REFLECTED_ATTRIBUTES,
                        ReflectedAttrsConverter::toCoreReflectedAttrs)
                .processNotNull(update.getAgeLabel(), MobileAppBanner.FLAGS,
                        ageLabel -> BannerFlagsConverter.toCoreBannerFlagsUpdate(ageLabel, oldFlags, canChangeBannerFlags));
        processJaxbElement(changes, update.getTrackingUrl(), MobileAppBanner.HREF);
        processJaxbElement(changes, update.getImpressionUrl(), MobileAppBanner.IMPRESSION_URL);
        processJaxbElement(changes, update.getAdImageHash(), MobileAppBanner.IMAGE_HASH);

        if (update.getVideoExtension() != null) {
            processJaxbElement(changes, update.getVideoExtension().getCreativeId(), MobileAppBanner.CREATIVE_ID);
        }

        return changes;
    }

    private ModelChanges<ImageBanner> convertMobileImageAd(Long id, MobileAppImageAdUpdate update) {
        ModelChanges<ImageBanner> mc = new ModelChanges<>(id, ImageBanner.class)
                .processNotNull(update.getAdImageHash(), ImageBanner.IMAGE_HASH);
        processJaxbElement(mc, update.getTrackingUrl(), ImageBanner.HREF);
        return mc.process(true, ImageBanner.IS_MOBILE_IMAGE);
    }

    private ModelChanges<CpcVideoBanner> convertMobileAppCpcVideoAd(Long id,
                                                                    MobileAppCpcVideoAdBuilderAdUpdate update) {
        Long creativeId = getCreativeId(update.getCreative());
        ModelChanges<CpcVideoBanner> mc = new ModelChanges<>(id, CpcVideoBanner.class)
                .processNotNull(creativeId, CpcVideoBanner.CREATIVE_ID);
        processJaxbElement(mc, update.getTrackingUrl(), CpcVideoBanner.HREF);
        return mc.process(true, CpcVideoBanner.IS_MOBILE_VIDEO);
    }

    private ModelChanges<ImageBanner> convertMobileImageCreativeAd(Long id, MobileAppAdBuilderAdUpdate update) {
        Long creativeId = getCreativeId(update.getCreative());
        ModelChanges<ImageBanner> mc = new ModelChanges<>(id, ImageBanner.class)
                .processNotNull(creativeId, ImageBanner.CREATIVE_ID)
                .process(true, ImageBanner.IS_MOBILE_IMAGE);
        processJaxbElement(mc, update.getTrackingUrl(), ImageBanner.HREF);
        return mc;
    }

    private ModelChanges<CpmBanner> convertCpmBannerAdBuilderAd(Long id, CpmBannerAdBuilderAdUpdate update) {
        Long creativeId = getCreativeId(update.getCreative());
        ModelChanges<CpmBanner> mc = new ModelChanges<>(id, CpmBanner.class)
                .processNotNull(creativeId, CpmBanner.CREATIVE_ID);

        processJaxbElement(mc, update.getTrackingPixels(), CpmBanner.PIXELS,
                ArrayOfString::getItems, emptyList());
        processJaxbElement(mc, update.getHref(), CpmBanner.HREF);
        processJaxbElement(mc, update.getTurboPageId(), CpmBanner.TURBO_LANDING_ID);
        processJaxbElement(mc, update.getTnsId(), CpmBanner.TNS_ID);
        return mc;
    }

    private ModelChanges<CpmBanner> convertCpmVideoAdBuilderAd(Long id, CpmVideoAdBuilderAdUpdate update) {
        Long creativeId = getCreativeId(update.getCreative());
        ModelChanges<CpmBanner> mc = new ModelChanges<>(id, CpmBanner.class)
                .processNotNull(creativeId, CpmBanner.CREATIVE_ID);

        processJaxbElement(mc, update.getLogoExtensionHash(), CpmBanner.LOGO_IMAGE_HASH);
        processJaxbElement(mc, update.getTitle2(), CpmBanner.TITLE_EXTENSION);
        processJaxbElement(mc, update.getTitle(), CpmBanner.TITLE);
        processJaxbElement(mc, update.getText(), CpmBanner.BODY);

        if (update.getButtonExtension() != null) {
            processJaxbElement(mc, update.getButtonExtension(), CpmBanner.BUTTON_CAPTION,
                    e -> ifNotNull(e.getCustomActionText(), JAXBElement::getValue));
            processJaxbElement(mc, update.getButtonExtension(), CpmBanner.BUTTON_HREF,
                    ButtonExtensionUpdateItem::getHref);
            processJaxbElement(mc, update.getButtonExtension(), CpmBanner.BUTTON_ACTION,
                    e -> convertButtonAction(e.getAction().value()));
        }

        processJaxbElement(mc, update.getTrackingPixels(), CpmBanner.PIXELS,
                ArrayOfString::getItems, emptyList());
        processJaxbElement(mc, update.getHref(), CpmBanner.HREF);
        processJaxbElement(mc, update.getTurboPageId(), CpmBanner.TURBO_LANDING_ID);
        processJaxbElement(mc, update.getTnsId(), CpmBanner.TNS_ID);
        return mc;
    }

    private ModelChanges<? extends BannerWithSystemFields> convertSmartAd(Long id,
                                                                          SmartAdBuilderAdUpdate update,
                                                                          BannerWithSystemFields smartBanner) {
        if (smartBanner instanceof PerformanceBanner) {
            Long creativeId = getCreativeId(update.getCreative());
            return new ModelChanges<>(id, PerformanceBanner.class)
                    .processNotNull(creativeId, PerformanceBanner.CREATIVE_ID);
        } else if (smartBanner instanceof PerformanceBannerMain) {
            ModelChanges<PerformanceBannerMain> mc = new ModelChanges<>(id, PerformanceBannerMain.class);
            processJaxbElement(mc, update.getLogoExtensionHash(), PerformanceBannerMain.LOGO_IMAGE_HASH);
            return mc;
        } else {
            throw new IllegalStateException("Unsupported type");
        }
    }

    private ModelChanges<ContentPromotionBanner> convertContentPromotionVideoAd(Long id,
                                                                                ContentPromotionVideoAdUpdate update) {
        ModelChanges<ContentPromotionBanner> mc = new ModelChanges<>(id, ContentPromotionBanner.class)
                .processNotNull(update.getPromotedContentId(), ContentPromotionBanner.CONTENT_PROMOTION_ID)
                .processNotNull(update.getText(), ContentPromotionBanner.BODY)
                .processNotNull(update.getTitle(), ContentPromotionBanner.TITLE);
        processJaxbElement(mc, update.getVisitHref(), ContentPromotionBanner.VISIT_URL);
        return mc;
    }

    private ModelChanges<ContentPromotionBanner> convertContentPromotionCollectionAd(Long id,
                                                                                     ContentPromotionCollectionAdUpdate update) {
        ModelChanges<ContentPromotionBanner> mc = new ModelChanges<>(id, ContentPromotionBanner.class)
                .processNotNull(update.getPromotedContentId(), ContentPromotionBanner.CONTENT_PROMOTION_ID);
        processJaxbElement(mc, update.getVisitHref(), ContentPromotionBanner.VISIT_URL);
        return mc;
    }

    private ModelChanges<ContentPromotionBanner> convertContentPromotionServiceAd(
            Long id,
            ContentPromotionServiceAdUpdate update) {
        ModelChanges<ContentPromotionBanner> mc = new ModelChanges<>(id, ContentPromotionBanner.class)
                .processNotNull(update.getPromotedContentId(), ContentPromotionBanner.CONTENT_PROMOTION_ID)
                .processNotNull(update.getTitle(), ContentPromotionBanner.TITLE);
        return mc;
    }

    private ModelChanges<ContentPromotionBanner> convertContentPromotionEdaAd(
            Long id,
            ContentPromotionEdaAdUpdate update) {
        ModelChanges<ContentPromotionBanner> mc = new ModelChanges<>(id, ContentPromotionBanner.class)
                .processNotNull(update.getPromotedContentId(), ContentPromotionBanner.CONTENT_PROMOTION_ID)
                .processNotNull(update.getTitle(), ContentPromotionBanner.TITLE);
        processJaxbElement(mc, update.getText(), ContentPromotionBanner.BODY);
        return mc;
    }

    private static boolean isReplacing(AdExtensionSetting settings) {
        return isOperationSet(settings.getAdExtensions().iterator().next());
    }

    private static boolean isOperationSet(AdExtensionSettingItem setting) {
        return setting.getOperation() == OperationEnum.SET;
    }

    private ModelChanges<CpcVideoBanner> convertCpcVideoAd(Long id, CpcVideoAdBuilderAdUpdate update) {
        Long creativeId = getCreativeId(update.getCreative());
        ModelChanges<CpcVideoBanner> mc = new ModelChanges<>(id, CpcVideoBanner.class)
                .processNotNull(creativeId, CpcVideoBanner.CREATIVE_ID);
        processJaxbElement(mc, update.getHref(), CpcVideoBanner.HREF);
        processJaxbElement(mc, update.getTurboPageId(), CpcVideoBanner.TURBO_LANDING_ID);
        return mc;
    }

    private static Object[] getAllFields(AdUpdateItem adUpdateItem) {
        return new Object[]{
                adUpdateItem.getTextAd(),
                adUpdateItem.getDynamicTextAd(),
                adUpdateItem.getTextImageAd(),
                adUpdateItem.getTextAdBuilderAd(),
                adUpdateItem.getMobileAppAd(),
                adUpdateItem.getMobileAppImageAd(),
                adUpdateItem.getMobileAppCpcVideoAdBuilderAd(),
                adUpdateItem.getMobileAppAdBuilderAd(),
                adUpdateItem.getCpmBannerAdBuilderAd(),
                adUpdateItem.getCpcVideoAdBuilderAd(),
                adUpdateItem.getCpmVideoAdBuilderAd(),
                adUpdateItem.getSmartAdBuilderAd(),
                adUpdateItem.getContentPromotionVideoAd(),
                adUpdateItem.getContentPromotionCollectionAd(),
                adUpdateItem.getContentPromotionServiceAd(),
                adUpdateItem.getContentPromotionEdaAd(),
        };
    }

    private static boolean onlyOneIsNonNull(Object... elements) {
        return StreamEx.of(elements).nonNull().count() == 1;
    }

    @Nullable
    private static Long getCreativeId(@Nullable AdBuilderAdUpdateItem creative) {
        return creative == null ? null : creative.getCreativeId();
    }
}
