package ru.yandex.direct.api.v5.entity.ads.converter;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;
import com.yandex.direct.api.v5.ads.ObjectFactory;
import com.yandex.direct.api.v5.ads.PriceCurrencyEnum;
import com.yandex.direct.api.v5.ads.PriceExtensionAddItem;
import com.yandex.direct.api.v5.ads.PriceExtensionGetItem;
import com.yandex.direct.api.v5.ads.PriceExtensionUpdateItem;
import com.yandex.direct.api.v5.ads.PriceQualifierEnum;

import ru.yandex.direct.core.entity.banner.model.BannerPrice;
import ru.yandex.direct.core.entity.banner.model.BannerPricesCurrency;
import ru.yandex.direct.core.entity.banner.model.BannerPricesPrefix;

import static ru.yandex.direct.api.v5.common.ConverterUtils.convertToDbPrice;
import static ru.yandex.direct.api.v5.common.ConverterUtils.convertToMicros;

/**
 * Выполняет преобразование цены из API в {@link BannerPrice} и обратно.
 */
public class BannerPriceConverter {
    private static final ObjectFactory FACTORY = new ObjectFactory();

    private static final BiMap<PriceQualifierEnum, BannerPricesPrefix> newPrefixMap;

    static {
        Map<PriceQualifierEnum, BannerPricesPrefix> apiToNewCoreMapping = new HashMap<>();
        apiToNewCoreMapping.put(PriceQualifierEnum.FROM, BannerPricesPrefix.FROM);
        apiToNewCoreMapping.put(PriceQualifierEnum.UP_TO, BannerPricesPrefix.TO);
        apiToNewCoreMapping.put(PriceQualifierEnum.NONE, null);
        newPrefixMap = HashBiMap.create(apiToNewCoreMapping);
    }

    @Nullable
    static BannerPrice toCore(@Nullable PriceExtensionAddItem item) {
        return item == null
                ? null
                : new BannerPrice()
                .withPrice(convertToDbPrice(item.getPrice()))
                .withPriceOld(convertToDbPrice(item.getOldPrice()))
                .withCurrency(toCore(item.getPriceCurrency()))
                .withPrefix(toCore(item.getPriceQualifier()));
    }

    /**
     * Формирует объект {@link BannerPrice} для Update. Выполняет merge текущего значения с новым.
     *
     * @param item         Значение, пришедшее в Update
     * @param currentPrice Текущее значение из базы
     * @return Заполненный объект для Update
     */
    @Nullable
    static BannerPrice toCore(@Nullable PriceExtensionUpdateItem item, @Nullable BannerPrice currentPrice) {
        if (item == null) {
            return null;
        }

        BannerPrice result = new BannerPrice();
        Optional<BannerPrice> currentPriceO = Optional.ofNullable(currentPrice);

        if (item.getPrice() == null) {
            result.withPrice(currentPriceO.map(BannerPrice::getPrice).orElse(null));
        } else {
            result.withPrice(convertToDbPrice(item.getPrice()));
        }

        if (item.getOldPrice() == null) {
            result.withPriceOld(currentPriceO.map(BannerPrice::getPriceOld).orElse(null));
        } else if (item.getOldPrice().isNil()) {
            result.withPriceOld(null);
        } else {
            result.withPriceOld(convertToDbPrice(item.getOldPrice().getValue()));
        }

        if (item.getPriceCurrency() == null) {
            result.withCurrency(currentPriceO.map(BannerPrice::getCurrency).orElse(null));
        } else {
            result.withCurrency(toCore(item.getPriceCurrency()));
        }

        if (item.getPriceQualifier() == null) {
            result.withPrefix(currentPriceO.map(BannerPrice::getPrefix).orElse(null));
        } else {
            result.withPrefix(toCore(item.getPriceQualifier()));
        }

        return result;
    }

    @Nullable
    static PriceExtensionGetItem fromCore(@Nullable BannerPrice price) {
        return price == null
                ? null
                : new PriceExtensionGetItem()
                .withPrice(convertToMicros(price.getPrice()))
                .withOldPrice(FACTORY.createPriceExtensionGetItemOldPrice(convertToMicros(price.getPriceOld())))
                .withPriceCurrency(fromCore(price.getCurrency()))
                .withPriceQualifier(fromCore(price.getPrefix()));
    }

    @Nullable
    static BannerPricesCurrency toCore(@Nullable PriceCurrencyEnum currency) {
        return currency == null ? null : BannerPricesCurrency.valueOf(currency.value());
    }

    @Nullable
    static PriceCurrencyEnum fromCore(@Nullable BannerPricesCurrency currency) {
        return currency == null ? null : PriceCurrencyEnum.valueOf(currency.name());
    }

    @Nullable
    static BannerPricesPrefix toCore(@Nonnull PriceQualifierEnum qualifier) {
        return newPrefixMap.get(qualifier);
    }

    @Nullable
    static PriceQualifierEnum fromCore(@Nullable BannerPricesPrefix prefix) {
        return newPrefixMap.inverse().get(prefix);
    }
}

