package ru.yandex.direct.api.v5.entity.ads.converter;

import java.util.Collection;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMultimap;
import com.google.common.collect.Multimap;
import com.yandex.direct.api.v5.ads.AdStateSelectionEnum;
import com.yandex.direct.api.v5.ads.AdStatusSelectionEnum;
import com.yandex.direct.api.v5.ads.AdTypeEnum;
import com.yandex.direct.api.v5.general.ExtensionStatusSelectionEnum;
import com.yandex.direct.api.v5.general.YesNoEnum;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.YesNo;
import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdgroupType;
import ru.yandex.direct.core.entity.banner.model.ExtensionStatus;
import ru.yandex.direct.core.entity.banner.model.State;
import ru.yandex.direct.core.entity.banner.model.Status;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;

@ParametersAreNonnullByDefault
@Service("adGetRequestConverter")
public class GetRequestConverter {

    private static final Multimap<AdTypeEnum, BannersBannerType> EXTERNAL_TO_INTERNAL_TYPE =
            ImmutableMultimap.<AdTypeEnum, BannersBannerType>builder()
                    .put(AdTypeEnum.TEXT_AD, BannersBannerType.text)
                    .put(AdTypeEnum.MOBILE_APP_AD, BannersBannerType.mobile_content)
                    .put(AdTypeEnum.CPC_VIDEO_AD, BannersBannerType.cpc_video)
                    .put(AdTypeEnum.DYNAMIC_TEXT_AD, BannersBannerType.dynamic)
                    .put(AdTypeEnum.IMAGE_AD, BannersBannerType.image_ad)
                    .put(AdTypeEnum.CPM_BANNER_AD, BannersBannerType.cpm_banner)
                    .put(AdTypeEnum.CPM_VIDEO_AD, BannersBannerType.cpm_banner)
                    .put(AdTypeEnum.SMART_AD, BannersBannerType.performance)
                    .put(AdTypeEnum.SMART_AD, BannersBannerType.performance_main)
                    .put(AdTypeEnum.CONTENT_PROMOTION_VIDEO_AD, BannersBannerType.content_promotion)
                    .put(AdTypeEnum.CONTENT_PROMOTION_COLLECTION_AD, BannersBannerType.content_promotion)
                    .put(AdTypeEnum.CONTENT_PROMOTION_SERVICE_AD, BannersBannerType.content_promotion)
                    .put(AdTypeEnum.CONTENT_PROMOTION_EDA_AD, BannersBannerType.content_promotion)
                    .build();

    private static final Map<AdStateSelectionEnum, State> EXTERNAL_TO_INTERNAL_STATES =
            ImmutableMap.<AdStateSelectionEnum, State>builder()
                    .put(AdStateSelectionEnum.ARCHIVED, State.ARCHIVED)
                    .put(AdStateSelectionEnum.OFF, State.OFF)
                    .put(AdStateSelectionEnum.OFF_BY_MONITORING, State.OFF_BY_MONITORING)
                    .put(AdStateSelectionEnum.ON, State.ON)
                    .put(AdStateSelectionEnum.SUSPENDED, State.SUSPENDED)
                    .build();

    private static final Map<AdStatusSelectionEnum, Status> EXTERNAL_TO_INTERNAL_STATUSES =
            ImmutableMap.<AdStatusSelectionEnum, Status>builder()
                    .put(AdStatusSelectionEnum.ACCEPTED, Status.ACCEPTED)
                    .put(AdStatusSelectionEnum.DRAFT, Status.DRAFT)
                    .put(AdStatusSelectionEnum.MODERATION, Status.MODERATION)
                    .put(AdStatusSelectionEnum.PREACCEPTED, Status.PREACCEPTED)
                    .put(AdStatusSelectionEnum.REJECTED, Status.REJECTED)
                    .build();

    private static final Map<ExtensionStatusSelectionEnum, ExtensionStatus> EXTERNAL_TO_INTERNAL_EXTENSION_STATUSES =
            ImmutableMap.<ExtensionStatusSelectionEnum, ExtensionStatus>builder()
                    .put(ExtensionStatusSelectionEnum.ACCEPTED, ExtensionStatus.ACCEPTED)
                    .put(ExtensionStatusSelectionEnum.DRAFT, ExtensionStatus.DRAFT)
                    .put(ExtensionStatusSelectionEnum.MODERATION, ExtensionStatus.MODERATION)
                    .put(ExtensionStatusSelectionEnum.REJECTED, ExtensionStatus.REJECTED)
                    .build();

    private static final Map<AdTypeEnum, ContentPromotionAdgroupType> EXTERNAL_TO_CONTENT_PROMOTION_TYPE =
            ImmutableMap.<AdTypeEnum, ContentPromotionAdgroupType>builder()
                    .put(AdTypeEnum.CONTENT_PROMOTION_VIDEO_AD, ContentPromotionAdgroupType.VIDEO)
                    .put(AdTypeEnum.CONTENT_PROMOTION_COLLECTION_AD, ContentPromotionAdgroupType.COLLECTION)
                    .put(AdTypeEnum.CONTENT_PROMOTION_SERVICE_AD, ContentPromotionAdgroupType.SERVICE)
                    .put(AdTypeEnum.CONTENT_PROMOTION_EDA_AD, ContentPromotionAdgroupType.EDA)
                    .build();


    private static <K, V extends Enum<V>> Set<V> convert(List<K> values, Class<V> clazz, Map<K, V> map) {
        Set<V> result = EnumSet.noneOf(clazz);

        for (K extValue : values) {
            V intValue = map.get(extValue);

            if (intValue == null) {
                throw new IllegalArgumentException("Not supported value: " + extValue);
            }

            result.add(intValue);
        }

        return result;
    }

    private static <K, V extends Enum<V>> Set<V> convert(List<K> values, Class<V> clazz, Multimap<K, V> map) {
        Set<V> result = EnumSet.noneOf(clazz);

        for (K extValue : values) {
            Collection<V> intValue = map.get(extValue);

            if (intValue == null) {
                throw new IllegalArgumentException("Not supported value: " + extValue);
            }

            result.addAll(intValue);
        }

        return result;
    }

    public static Set<BannersBannerType> convertTypes(List<AdTypeEnum> externalTypes) {
        return convert(externalTypes, BannersBannerType.class, EXTERNAL_TO_INTERNAL_TYPE);
    }

    public static Set<ContentPromotionAdgroupType> convertToContentPromotionTypes(List<AdTypeEnum> externalTypes) {
        return StreamEx.of(externalTypes).map(EXTERNAL_TO_CONTENT_PROMOTION_TYPE::get).nonNull().toSet();
    }

    public static Set<State> convertStates(List<AdStateSelectionEnum> externalStates) {
        return convert(externalStates, State.class, EXTERNAL_TO_INTERNAL_STATES);
    }

    public static Set<Status> convertStatuses(List<AdStatusSelectionEnum> externalStatuses) {
        return convert(externalStatuses, Status.class, EXTERNAL_TO_INTERNAL_STATUSES);
    }

    public static Set<ExtensionStatus> convertExtensionStatuses(
            List<ExtensionStatusSelectionEnum> externalExtStatuses) {
        return convert(externalExtStatuses, ExtensionStatus.class, EXTERNAL_TO_INTERNAL_EXTENSION_STATUSES);
    }

    public static YesNo convertMobile(@Nullable YesNoEnum extenalValue) {
        if (extenalValue == null) {
            return null;
        }
        return YesNoEnum.YES.equals(extenalValue) ? YesNo.YES : YesNo.NO;
    }

    public static Set<BannersBannerType> getSupportedTypes() {
        return new HashSet<>(EXTERNAL_TO_INTERNAL_TYPE.values());
    }

}
