package ru.yandex.direct.api.v5.entity.ads.converter;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.xml.bind.JAXBElement;

import com.google.common.collect.ImmutableMap;
import com.yandex.direct.api.v5.ads.ObjectFactory;
import com.yandex.direct.api.v5.ads.VideoExtensionGetItem;
import com.yandex.direct.api.v5.general.ExtensionModeration;
import com.yandex.direct.api.v5.general.StatusEnum;
import com.yandex.direct.api.v5.general.YesNoEnum;
import one.util.streamex.StreamEx;

import ru.yandex.direct.api.v5.entity.ads.StatusClarificationTranslations;
import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.banner.model.BannerCreativeStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerDisplayHrefStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerStatusSitelinksModerate;
import ru.yandex.direct.core.entity.banner.model.BannerVcardStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerWithBannerImage;
import ru.yandex.direct.core.entity.banner.model.BannerWithCreativeModeration;
import ru.yandex.direct.core.entity.banner.model.BannerWithDisplayHref;
import ru.yandex.direct.core.entity.banner.model.BannerWithSitelinks;
import ru.yandex.direct.core.entity.banner.model.BannerWithVcard;
import ru.yandex.direct.core.entity.banner.model.StatusBannerImageModerate;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiag;

import static ru.yandex.direct.utils.CommonUtils.nvl;

@ParametersAreNonnullByDefault
class ModerationStatusBuilder {
    private static final ObjectFactory FACTORY = new ObjectFactory();

    private static final Map<StatusBannerImageModerate, StatusEnum>
            INTERNAL_TO_EXTERNAL_BANNERIMAGE_MODERATION_STATUS =
            ImmutableMap.<StatusBannerImageModerate, StatusEnum>builder()
                    .put(StatusBannerImageModerate.NEW, StatusEnum.DRAFT)
                    .put(StatusBannerImageModerate.SENDING, StatusEnum.MODERATION)
                    .put(StatusBannerImageModerate.SENT, StatusEnum.MODERATION)
                    .put(StatusBannerImageModerate.READY, StatusEnum.MODERATION)
                    .put(StatusBannerImageModerate.YES, StatusEnum.ACCEPTED)
                    .put(StatusBannerImageModerate.NO, StatusEnum.REJECTED)
                    .build();

    private static final Map<BannerDisplayHrefStatusModerate, StatusEnum>
            INTERNAL_TO_EXTERNAL_DISPLAY_HREF_MODERATION_STATUS =
            ImmutableMap.<BannerDisplayHrefStatusModerate, StatusEnum>builder()
                    .put(BannerDisplayHrefStatusModerate.SENDING, StatusEnum.MODERATION)
                    .put(BannerDisplayHrefStatusModerate.SENT, StatusEnum.MODERATION)
                    .put(BannerDisplayHrefStatusModerate.READY, StatusEnum.MODERATION)
                    .put(BannerDisplayHrefStatusModerate.YES, StatusEnum.ACCEPTED)
                    .put(BannerDisplayHrefStatusModerate.NO, StatusEnum.REJECTED)
                    .build();

    private static final Map<BannerStatusSitelinksModerate, StatusEnum> INTERNAL_TO_EXTERNAL_SITELINKS_MODERATION_STATUS =
            ImmutableMap.<BannerStatusSitelinksModerate, StatusEnum>builder()
                    .put(BannerStatusSitelinksModerate.NEW, StatusEnum.DRAFT)
                    .put(BannerStatusSitelinksModerate.SENDING, StatusEnum.MODERATION)
                    .put(BannerStatusSitelinksModerate.SENT, StatusEnum.MODERATION)
                    .put(BannerStatusSitelinksModerate.READY, StatusEnum.MODERATION)
                    .put(BannerStatusSitelinksModerate.YES, StatusEnum.ACCEPTED)
                    .put(BannerStatusSitelinksModerate.NO, StatusEnum.REJECTED)
                    .build();

    private static final Map<BannerVcardStatusModerate, StatusEnum> INTERNAL_TO_EXTERNAL_VCARD_MODERATION_STATUS =
            ImmutableMap.<BannerVcardStatusModerate, StatusEnum>builder()
                    .put(BannerVcardStatusModerate.NEW, StatusEnum.DRAFT)
                    .put(BannerVcardStatusModerate.SENDING, StatusEnum.MODERATION)
                    .put(BannerVcardStatusModerate.SENT, StatusEnum.MODERATION)
                    .put(BannerVcardStatusModerate.READY, StatusEnum.MODERATION)
                    .put(BannerVcardStatusModerate.YES, StatusEnum.ACCEPTED)
                    .put(BannerVcardStatusModerate.NO, StatusEnum.REJECTED)
                    .build();

    private static final Map<BannerCreativeStatusModerate, StatusEnum>
            INTERNAL_TO_EXTERNAL_VIDEO_MODERATION_STATUS =
            ImmutableMap.<BannerCreativeStatusModerate, StatusEnum>builder()
                    .put(BannerCreativeStatusModerate.NEW, StatusEnum.DRAFT)
                    .put(BannerCreativeStatusModerate.SENDING, StatusEnum.MODERATION)
                    .put(BannerCreativeStatusModerate.SENT, StatusEnum.MODERATION)
                    .put(BannerCreativeStatusModerate.READY, StatusEnum.MODERATION)
                    .put(BannerCreativeStatusModerate.YES, StatusEnum.ACCEPTED)
                    .put(BannerCreativeStatusModerate.NO, StatusEnum.REJECTED)
                    .build();

    private static String calcRelatedEntityModerationReasons(List<ModerationDiag> reasons) {
        return StreamEx.of(reasons).map(ModerationDiag::getDiagText).joining("\n");
    }

    static JAXBElement<VideoExtensionGetItem> convertVideoExtension(
            @Nullable BannerWithCreativeModeration bannerCreative,
            @Nullable Creative creative) {
        VideoExtensionGetItem videoExtensionGetItem = null;

        if (bannerCreative != null && creative != null) {
            StatusEnum externalStatus =
                    INTERNAL_TO_EXTERNAL_VIDEO_MODERATION_STATUS.get(bannerCreative.getCreativeStatusModerate());

            if (externalStatus == null) {
                throw new IllegalArgumentException("can\'t calc video addition status: " + externalStatus);
            }
            YesNoEnum showTitleAndBodyExt = nvl(bannerCreative.getShowTitleAndBody(), false) ? YesNoEnum.YES : YesNoEnum.NO;

            videoExtensionGetItem = FACTORY.createVideoExtensionGetItem()
                    .withCreativeId(creative.getId())
                    .withShowTitleAndBody(showTitleAndBodyExt)
                    .withPreviewUrl(creative.getLivePreviewUrl())
                    .withThumbnailUrl(creative.getPreviewUrl())
                    .withStatus(externalStatus);
        }

        return FACTORY.createTextAdGetVideoExtension(videoExtensionGetItem);
    }

    static JAXBElement<ExtensionModeration> buildVCardModeration(BannerWithVcard ad, List<ModerationDiag> reasons,
                                                                 TranslationService translationService) {
        StatusClarificationTranslations translations = StatusClarificationTranslations.INSTANCE;

        ExtensionModeration extensionModeration = null;

        if (ad.getVcardId() != null) {
            var statusPhoneFlagModerate = ad.getVcardStatusModerate();

            StatusEnum status = INTERNAL_TO_EXTERNAL_VCARD_MODERATION_STATUS.get(statusPhoneFlagModerate);

            if (status == null) {
                throw new IllegalArgumentException(
                        "can\'t calc vcard moderation status: " + statusPhoneFlagModerate);
            }

            final String statusClarification;
            if (status == StatusEnum.DRAFT) {
                statusClarification = translationService.translate(translations.relatedObjectIsDraft());
            } else if (status == StatusEnum.MODERATION) {
                statusClarification = translationService.translate(translations.contactInfoAwaitingModeration());
            } else if (status == StatusEnum.ACCEPTED) {
                statusClarification = translationService.translate(translations.contactInfoAcceptedAtModeration());
            } else if (status == StatusEnum.REJECTED) {
                List<String> clarifications = new ArrayList<>();
                clarifications.add(translationService.translate(translations.contactInfoRejectedAtModeration()));
                clarifications.add(calcRelatedEntityModerationReasons(reasons));
                statusClarification = StreamEx.of(clarifications).joining(" ");
            } else {
                statusClarification = "";
            }

            extensionModeration =
                    new ExtensionModeration().withStatus(status).withStatusClarification(statusClarification);
        }

        return FACTORY.createTextAdGetBaseVCardModeration(extensionModeration);
    }

    static JAXBElement<ExtensionModeration> buildSitelinksModeration(BannerWithSitelinks ad,
                                                                     List<ModerationDiag> reasons,
                                                                     TranslationService translationService) {
        StatusClarificationTranslations translations = StatusClarificationTranslations.INSTANCE;

        ExtensionModeration extensionModeration = null;

        if (ad.getSitelinksSetId() != null) {
            var statusSitelinksModerate = ad.getStatusSitelinksModerate();

            StatusEnum status = INTERNAL_TO_EXTERNAL_SITELINKS_MODERATION_STATUS.get(statusSitelinksModerate);

            if (status == null) {
                throw new IllegalArgumentException(
                        "can\'t calc sitelinks moderation status: " + statusSitelinksModerate);
            }

            final String statusClarification;
            if (status == StatusEnum.DRAFT) {
                statusClarification = translationService.translate(translations.relatedObjectIsDraft());
            } else if (status == StatusEnum.MODERATION) {
                statusClarification = translationService.translate(translations.sitelinksAwaitingModeration());
            } else if (status == StatusEnum.ACCEPTED) {
                statusClarification = translationService.translate(translations.sitelinksAcceptedAtModeration());
            } else if (status == StatusEnum.REJECTED) {
                List<String> clarifications = new ArrayList<>();
                clarifications.add(translationService.translate(translations.sitelinksRejectedAtModeration()));
                clarifications.add(calcRelatedEntityModerationReasons(reasons));
                statusClarification = StreamEx.of(clarifications).joining(" ");
            } else {
                statusClarification = "";
            }

            extensionModeration =
                    new ExtensionModeration().withStatus(status).withStatusClarification(statusClarification);
        }

        return FACTORY.createTextAdGetBaseSitelinksModeration(extensionModeration);
    }

    static JAXBElement<ExtensionModeration> buildDisplayUrlPathModeration(BannerWithDisplayHref ad,
                                                                          List<ModerationDiag> reasons,
                                                                          TranslationService translationService) {
        StatusClarificationTranslations translations = StatusClarificationTranslations.INSTANCE;

        ExtensionModeration extensionModeration = null;

        if (ad.getDisplayHref() != null) {
            var displayHrefStatusModerate = ad.getDisplayHrefStatusModerate();

            StatusEnum status = INTERNAL_TO_EXTERNAL_DISPLAY_HREF_MODERATION_STATUS.get(displayHrefStatusModerate);

            if (status == null) {
                throw new IllegalArgumentException(
                        "can\'t calc display url path moderation status: " + displayHrefStatusModerate);
            }

            final String statusClarification;
            if (status == StatusEnum.DRAFT) {
                statusClarification = translationService.translate(translations.relatedObjectIsDraft());
            } else if (status == StatusEnum.MODERATION) {
                statusClarification = translationService.translate(translations.displayLinkAwaitingModeration());
            } else if (status == StatusEnum.ACCEPTED) {
                statusClarification = translationService.translate(translations.displayLinkAcceptedAtModeration());
            } else if (status == StatusEnum.REJECTED) {
                List<String> clarifications = new ArrayList<>();
                clarifications.add(translationService.translate(translations.displayLinkRejectedAtModeration()));
                clarifications.add(calcRelatedEntityModerationReasons(reasons));
                statusClarification = StreamEx.of(clarifications).joining(" ");
            } else {
                statusClarification = "";
            }

            extensionModeration =
                    new ExtensionModeration().withStatus(status).withStatusClarification(statusClarification);
        }

        return FACTORY.createTextAdGetDisplayUrlPathModeration(extensionModeration);
    }

    static ExtensionModeration buildAdImageModeration(BannerWithBannerImage ad, List<ModerationDiag> reasons,
                                                      TranslationService translationService) {
        StatusClarificationTranslations translations = StatusClarificationTranslations.INSTANCE;

        ExtensionModeration extensionModeration = null;

        if (ad.getImageHash() != null) {
            var statusBannerImageModerate = ad.getImageStatusModerate();

            StatusEnum status =
                    INTERNAL_TO_EXTERNAL_BANNERIMAGE_MODERATION_STATUS.get(statusBannerImageModerate);

            if (status == null) {
                throw new IllegalArgumentException(
                        "can\'t calc ad image moderation status for image: " + statusBannerImageModerate);
            }

            final String statusClarification;
            if (status == StatusEnum.DRAFT) {
                statusClarification = translationService.translate(translations.relatedObjectIsDraft());
            } else if (status == StatusEnum.MODERATION) {
                statusClarification = translationService.translate(translations.imageAwaitingModeration());
            } else if (status == StatusEnum.ACCEPTED) {
                statusClarification = translationService.translate(translations.imageAcceptedAtModeration());
            } else if (status == StatusEnum.REJECTED) {
                List<String> clarifications = new ArrayList<>();
                clarifications.add(translationService.translate(translations.imageRejectedAtModeration()));
                clarifications.add(calcRelatedEntityModerationReasons(reasons));
                statusClarification = StreamEx.of(clarifications).joining(" ");
            } else {
                statusClarification = "";
            }

            extensionModeration =
                    new ExtensionModeration().withStatus(status).withStatusClarification(statusClarification);
        }

        return extensionModeration;
    }

}
