package ru.yandex.direct.api.v5.entity.ads.converter;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.general.StateEnum;
import com.yandex.direct.api.v5.general.StatusEnum;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.api.v5.entity.ads.StatusClarificationTranslations;
import ru.yandex.direct.api.v5.entity.ads.container.AdsGetContainer;
import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.banner.model.BannerCreativeStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerStatusPostModerate;
import ru.yandex.direct.core.entity.banner.model.BannerWithCreativeModeration;
import ru.yandex.direct.core.entity.banner.model.BannerWithImageModeration;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.ImageBanner;
import ru.yandex.direct.core.entity.banner.model.NewStatusImageModerate;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiag;

import static ru.yandex.direct.core.entity.banner.service.validation.type.BannerTypeValidationPredicates.isCpcVideoBanner;
import static ru.yandex.direct.core.entity.banner.service.validation.type.BannerTypeValidationPredicates.isCpmBanner;
import static ru.yandex.direct.core.entity.banner.service.validation.type.BannerTypeValidationPredicates.isImageBanner;

@ParametersAreNonnullByDefault
class StateAndStatusCalculator {

    private static final Logger logger = LoggerFactory.getLogger(StateAndStatusCalculator.class);

    private static StatusEnum calcStatus(BannerWithImageModeration ad) {
        var statusModerate = ad.getStatusModerate();
        var statusPostModerate = ad.getStatusPostModerate();
        var imageStatusModerate = ad.getImageStatusModerate();

        StatusEnum externalStatus = StatusEnum.UNKNOWN;

        // для ГО считается что для отправки в БК картинка должна быть промодерирована, хотя сам баннер может быть
        // на постмодерации

        if (statusModerate == BannerStatusModerate.NEW || imageStatusModerate == NewStatusImageModerate.NEW) {
            externalStatus = StatusEnum.DRAFT;
        } else if (statusModerate == BannerStatusModerate.YES && statusPostModerate == BannerStatusPostModerate.YES
                && imageStatusModerate == NewStatusImageModerate.YES) {
            externalStatus = StatusEnum.ACCEPTED;
        } else if (statusModerate == BannerStatusModerate.NO || imageStatusModerate == NewStatusImageModerate.NO) {
            externalStatus = StatusEnum.REJECTED;
        } else if ((statusModerate == BannerStatusModerate.SENDING || statusModerate == BannerStatusModerate.SENT)
                && statusPostModerate == BannerStatusPostModerate.YES && imageStatusModerate == NewStatusImageModerate.YES) {
            externalStatus = StatusEnum.PREACCEPTED;
        } else if (imageStatusModerate == NewStatusImageModerate.SENDING || imageStatusModerate == NewStatusImageModerate.SENT
                || imageStatusModerate == NewStatusImageModerate.READY) {
            externalStatus = StatusEnum.MODERATION;
        } else if ((statusModerate == BannerStatusModerate.SENDING || statusModerate == BannerStatusModerate.SENT
                || statusModerate == BannerStatusModerate.READY)
                && (statusPostModerate == BannerStatusPostModerate.NO || statusPostModerate == BannerStatusPostModerate.REJECTED)
                && imageStatusModerate != NewStatusImageModerate.NO) {
            externalStatus = StatusEnum.MODERATION;
        }

        if (externalStatus == StatusEnum.UNKNOWN) {
            logger.error("Can\'t calc status for ad with image {}", ad);
        }

        return externalStatus;
    }

    /**
     * Метод вычисления общего статуса баннера, основанного на креативе (image_ad, cpc_video, cpm_banner)
     */
    private static StatusEnum calcStatus(BannerWithCreativeModeration ad) {
        var statusModerate = ad.getStatusModerate();
        var creativeStatusModerate = ad.getCreativeStatusModerate();

        StatusEnum externalStatus = StatusEnum.UNKNOWN;

        // при вычислении статуса ACCEPTED значение в banners.statusPostModerate не имеет значения

        // для объявления с канвасными креативами статуса PREACCEPTED нет

        // при вычислении статуса MODERATION значение в banners.statusModerate = New, т.к. в этом случае ранее будет
        // вычислено значение DRAFT

        if (statusModerate == BannerStatusModerate.NEW || creativeStatusModerate == BannerCreativeStatusModerate.NEW) {
            externalStatus = StatusEnum.DRAFT;
        } else if (statusModerate == BannerStatusModerate.YES && creativeStatusModerate == BannerCreativeStatusModerate.YES) {
            externalStatus = StatusEnum.ACCEPTED;
        } else if (statusModerate == BannerStatusModerate.NO || creativeStatusModerate == BannerCreativeStatusModerate.NO) {
            externalStatus = StatusEnum.REJECTED;
        } else if (statusModerate != BannerStatusModerate.NO && (
                creativeStatusModerate == BannerCreativeStatusModerate.SENDING
                        || creativeStatusModerate == BannerCreativeStatusModerate.SENT
                        || creativeStatusModerate == BannerCreativeStatusModerate.READY)) {
            externalStatus = StatusEnum.MODERATION;
        } else if ((statusModerate == BannerStatusModerate.SENDING || statusModerate == BannerStatusModerate.SENT
                || statusModerate == BannerStatusModerate.READY) && creativeStatusModerate != BannerCreativeStatusModerate.NO) {
            externalStatus = StatusEnum.MODERATION;
        }

        if (externalStatus == StatusEnum.UNKNOWN) {
            logger.error("Can\'t calc status for ad based on creative {}", ad);
        }

        return externalStatus;
    }

    static StatusEnum calcStatus(BannerWithSystemFields ad) {
        /*
                считается что status_moderate = 'New' может быть только при создании банера,
            в дальнейшем при изменении баннера либо подчиненного объекта устанавливается
            status_moderate = 'Ready' у измененного объекта

                так как статусы вычисляются последовательно, то при вычислении более
            "поздних" статусов допустимо не рассматривать некоторые значения полей,
            при которых статус был бы определен ранее (например, при проверке не является
            ли PREACCEPTED конечным можно не рассматривать imagead_status_moderate = 'No',
            т.к. в этом случае ранее был бы вычислен статус REJECTED).
        */

        if (isImageBanner(ad) && ((ImageBanner) ad).getImageHash() != null) {
            return calcStatus((BannerWithImageModeration) ad);
        }

        if (isImageBanner(ad) && ((ImageBanner) ad).getCreativeId() != null
                || isCpcVideoBanner(ad) || isCpmBanner(ad)) {
            return calcStatus((BannerWithCreativeModeration) ad);
        }

        var statusModerate = ad.getStatusModerate();
        var statusPostModerate = ad.getStatusPostModerate();

        final StatusEnum externalStatus;
        if (statusModerate == BannerStatusModerate.NEW) {
            externalStatus = StatusEnum.DRAFT;
        } else if (statusModerate == BannerStatusModerate.YES && statusPostModerate == BannerStatusPostModerate.YES) {
            externalStatus = StatusEnum.ACCEPTED;
        } else if (statusModerate == BannerStatusModerate.NO) {
            externalStatus = StatusEnum.REJECTED;
        } else if ((statusModerate == BannerStatusModerate.SENDING || statusModerate == BannerStatusModerate.SENT)
                && statusPostModerate == BannerStatusPostModerate.YES) {
            externalStatus = StatusEnum.PREACCEPTED;
        } else if ((statusModerate == BannerStatusModerate.SENDING || statusModerate == BannerStatusModerate.SENT
                || statusModerate == BannerStatusModerate.READY)
                && (statusPostModerate == BannerStatusPostModerate.NO || statusPostModerate == BannerStatusPostModerate.REJECTED)) {
            externalStatus = StatusEnum.MODERATION;
        } else {
            externalStatus = StatusEnum.UNKNOWN;
            logger.warn("Can\'t calc status for ad {}", ad);
        }

        return externalStatus;
    }

    static StateEnum calcState(BannerWithSystemFields ad, Boolean isAdStoppedByMonitoring, Campaign campaign) {
        StateEnum externalState = StateEnum.UNKNOWN;

        if (ad.getStatusArchived() || campaign.getStatusArchived()) {
            externalState = StateEnum.ARCHIVED;
        } else if (!ad.getStatusArchived() && !ad.getStatusShow() && !campaign.getStatusArchived()) {
            externalState = StateEnum.SUSPENDED;
        } else if (!ad.getStatusArchived() && isAdStoppedByMonitoring && ad.getStatusShow()
                && !campaign.getStatusArchived()) {
            externalState = StateEnum.OFF_BY_MONITORING;
        } else if (ad.getStatusActive() && !ad.getStatusArchived() && !isAdStoppedByMonitoring && ad.getStatusShow()
                && campaign.getStatusActive() && !campaign.getStatusArchived() && campaign.getStatusShow()) {
            externalState = StateEnum.ON;
        } else if ((!ad.getStatusArchived() && ad.getStatusShow() && !isAdStoppedByMonitoring
                && !campaign.getStatusArchived() && !campaign.getStatusShow())
                || (!ad.getStatusActive() && !ad.getStatusArchived() && ad.getStatusShow() && !isAdStoppedByMonitoring
                && !campaign.getStatusArchived())
                || (!ad.getStatusArchived() && ad.getStatusShow() && !isAdStoppedByMonitoring
                && !campaign.getStatusArchived() && !campaign.getStatusActive())) {
            externalState = StateEnum.OFF;
        }

        if (externalState == StateEnum.UNKNOWN) {
            logger.error("Can\'t calc state for ad {} with campaign {}", ad, campaign);
        }

        return externalState;
    }

    static String calcStatusClarification(StatusEnum status, StateEnum state, Boolean isCampaignShow,
                                          AdsGetContainer adsGetContainer, TranslationService translationService) {
        StatusClarificationTranslations translations = StatusClarificationTranslations.INSTANCE;

        final String statusClarification;
        if (state == StateEnum.ARCHIVED) {
            statusClarification = translationService.translate(translations.adArchived());
        } else if (status == StatusEnum.DRAFT) {
            statusClarification = translationService.translate(translations.adDraft());
        } else if (state == StateEnum.SUSPENDED) {
            statusClarification = translationService.translate(translations.adStopped());
        } else if (state == StateEnum.OFF_BY_MONITORING) {
            statusClarification = translationService.translate(translations.adStoppedBySiteMonitoring());
        } else if (state == StateEnum.ON) {
            if (status == StatusEnum.ACCEPTED || status == StatusEnum.PREACCEPTED) {
                statusClarification = translationService.translate(translations.adRunning());
            } else {
                statusClarification = translationService.translate(translations.previousVersionOfAdRunning());
            }
        } else if (state == StateEnum.OFF && !isCampaignShow) {
            statusClarification = translationService.translate(translations.campaignStopped());
        } else if (status == StatusEnum.MODERATION) {
            statusClarification = translationService.translate(translations.adAwaitingModeration());
        } else if (status == StatusEnum.PREACCEPTED) {
            statusClarification = translationService.translate(translations.adPreacceptedAtModeration());
        } else if (status == StatusEnum.ACCEPTED) {
            statusClarification = translationService.translate(translations.adAccepted());
        } else {
            statusClarification = "";
        }

        final String result;
        if (status == StatusEnum.REJECTED) {
            List<String> clarifications = new ArrayList<>();

            if (!statusClarification.isEmpty()) {
                clarifications.add(statusClarification);
            }

            clarifications.add(translationService.translate(translations.adRejected()));

            // TODO: где и как выполняется i18n-ция текстов причин отклонения?

            List<ModerationDiag> bannerModerationReasons = adsGetContainer.getBannerModerationReasons();
            if (!bannerModerationReasons.isEmpty()) {
                clarifications.addAll(StreamEx.of(bannerModerationReasons).map(ModerationDiag::getDiagText).toList());
            }

            List<ModerationDiag> imageModerationReasons = adsGetContainer.getImageModerationReasons();
            if (!imageModerationReasons.isEmpty()) {
                clarifications.addAll(StreamEx.of(imageModerationReasons).map(ModerationDiag::getDiagText).toList());
            }

            result = StreamEx.of(clarifications).joining("\n");
        } else {
            result = statusClarification;
        }

        return result;
    }
}
