package ru.yandex.direct.api.v5.entity.ads.converter;

import java.util.Map;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.yandex.direct.api.v5.ads.AdSubtypeEnum;
import com.yandex.direct.api.v5.ads.AdTypeEnum;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdgroupType;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.CpcVideoBanner;
import ru.yandex.direct.core.entity.banner.model.ImageBanner;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;

import static ru.yandex.direct.core.entity.banner.repository.BannerRepositoryConstants.BANNER_CLASS_TO_TYPE;
import static ru.yandex.direct.core.entity.banner.service.validation.type.BannerTypeValidationPredicates.isCpcVideoBanner;
import static ru.yandex.direct.core.entity.banner.service.validation.type.BannerTypeValidationPredicates.isImageBanner;

@ParametersAreNonnullByDefault
class TypeAndSubtypesConverter {

    /**
     * Маппинг типов баннеров во внешние типы один к одному, если типу баннера соответствует больше одного внешнего,
     * надо мапить руками, см. {@link #EXTERNAL_AD_TYPE_BY_INTERNAL_AD_GROUP_TYPE}
     */
    private static final Map<BannersBannerType, AdTypeEnum> EXTERNAL_AD_TYPE_BY_INTERNAL_BANNER_TYPE =
            ImmutableMap.<BannersBannerType, AdTypeEnum>builder()
                    .put(BannersBannerType.dynamic, AdTypeEnum.DYNAMIC_TEXT_AD)
                    .put(BannersBannerType.image_ad, AdTypeEnum.IMAGE_AD)
                    .put(BannersBannerType.mobile_content, AdTypeEnum.MOBILE_APP_AD)
                    .put(BannersBannerType.text, AdTypeEnum.TEXT_AD)
                    .put(BannersBannerType.cpc_video, AdTypeEnum.CPC_VIDEO_AD)
                    .put(BannersBannerType.performance, AdTypeEnum.SMART_AD)
                    .put(BannersBannerType.performance_main, AdTypeEnum.SMART_AD)
                    .build();

    /**
     * Маппинг типов групп во внешние типы баннеров, применяется после {@link #EXTERNAL_AD_TYPE_BY_INTERNAL_BANNER_TYPE}
     */
    private static final Map<AdGroupType, AdTypeEnum> EXTERNAL_AD_TYPE_BY_INTERNAL_AD_GROUP_TYPE =
            ImmutableMap.<AdGroupType, AdTypeEnum>builder()
                    .put(AdGroupType.CPM_BANNER, AdTypeEnum.CPM_BANNER_AD)
                    .put(AdGroupType.CPM_VIDEO, AdTypeEnum.CPM_VIDEO_AD)
                    .build();

    /**
     * Маппинг типов групп во внешние типы баннеров,
     * применяется после {@link #EXTERNAL_AD_TYPE_BY_INTERNAL_AD_GROUP_TYPE}
     */
    private static final Map<ContentPromotionAdgroupType, AdTypeEnum>
            EXTERNAL_AD_TYPE_BY_INTERNAL_CONTENT_PROMOTION_AD_GROUP_TYPE =
            ImmutableMap.<ContentPromotionAdgroupType, AdTypeEnum>builder()
                    .put(ContentPromotionAdgroupType.VIDEO, AdTypeEnum.CONTENT_PROMOTION_VIDEO_AD)
                    .put(ContentPromotionAdgroupType.COLLECTION, AdTypeEnum.CONTENT_PROMOTION_COLLECTION_AD)
                    .put(ContentPromotionAdgroupType.SERVICE, AdTypeEnum.CONTENT_PROMOTION_SERVICE_AD)
                    .put(ContentPromotionAdgroupType.EDA, AdTypeEnum.CONTENT_PROMOTION_EDA_AD)
                    .build();

    static AdTypeEnum convertType(Banner banner,
                                  AdGroupType adGroupType,
                                  @Nullable ContentPromotionAdgroupType contentPromotionAdgroupType) {
        var bannersBannerType = BANNER_CLASS_TO_TYPE.get(banner.getClass());
        AdTypeEnum externalType =
                Optional.ofNullable(EXTERNAL_AD_TYPE_BY_INTERNAL_BANNER_TYPE.get(bannersBannerType))
                        .or(() -> Optional.ofNullable(EXTERNAL_AD_TYPE_BY_INTERNAL_AD_GROUP_TYPE.get(adGroupType)))
                        .orElseGet(() -> EXTERNAL_AD_TYPE_BY_INTERNAL_CONTENT_PROMOTION_AD_GROUP_TYPE
                                .get(contentPromotionAdgroupType));

        if (externalType == null) {
            StringBuilder sb = new StringBuilder("not supported ad type: ");
            sb.append(bannersBannerType.toString());
            sb.append(" ");
            sb.append(adGroupType.toString());

            if (contentPromotionAdgroupType != null) {
                sb.append(" ");
                sb.append(contentPromotionAdgroupType.toString());
            }

            throw new IllegalArgumentException(sb.toString());
        }

        return externalType;
    }

    static AdSubtypeEnum calcSubtype(BannerWithSystemFields ad) {
        AdSubtypeEnum subtype = AdSubtypeEnum.NONE;

        if (isImageBanner(ad)) {
            var adTyped = ((ImageBanner) ad);
            if (adTyped.getImageHash() != null && adTyped.getIsMobileImage() != null) {
                if (adTyped.getIsMobileImage()) {
                    subtype = AdSubtypeEnum.MOBILE_APP_IMAGE_AD;
                } else {
                    subtype = AdSubtypeEnum.TEXT_IMAGE_AD;
                }
            } else if (adTyped.getCreativeId() != null && adTyped.getIsMobileImage() != null) {
                if (adTyped.getIsMobileImage()) {
                    subtype = AdSubtypeEnum.MOBILE_APP_AD_BUILDER_AD;
                } else {
                    subtype = AdSubtypeEnum.TEXT_AD_BUILDER_AD;
                }
            }

            if (subtype == AdSubtypeEnum.NONE) {
                throw new IllegalArgumentException("can\'t calc subtype for ad: " + ad);
            }
        }
        if (isCpcVideoBanner(ad)) {
            var adTyped = ((CpcVideoBanner) ad);
            if (adTyped.getIsMobileVideo()) {
                subtype = AdSubtypeEnum.MOBILE_APP_CPC_VIDEO_AD_BUILDER_AD;
            }
        }

        return subtype;
    }
}
