package ru.yandex.direct.api.v5.entity.ads.delegate;

import java.util.List;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.ads.DeleteRequest;
import com.yandex.direct.api.v5.ads.DeleteResponse;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.api.v5.common.ApiPathConverter;
import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.OperationOnListDelegate;
import ru.yandex.direct.api.v5.entity.ads.validation.AdsAdGroupTypeValidator;
import ru.yandex.direct.api.v5.entity.ads.validation.AdsDeleteRequestValidator;
import ru.yandex.direct.api.v5.result.ApiMassResult;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.api.v5.validation.DefectTypes;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.banner.service.BannerService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.units.OperationSummary;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.api.v5.entity.OperationOnListUtils.notCountErrorsWithCodes;

@ParametersAreNonnullByDefault
@Service
public class DeleteAdsDelegate
        extends OperationOnListDelegate<DeleteRequest, DeleteResponse, Long, Long> {
    private static final int BAD_STATUS_CODE = DefectTypes.badStatus().getCode();

    private final ResultConverter resultConverter;
    private final AdsDeleteRequestValidator requestValidator;
    private final BannerService bannerService;
    private final AdsAdGroupTypeValidator adGroupTypeValidator;

    @Autowired
    public DeleteAdsDelegate(
            ApiAuthenticationSource auth,
            ResultConverter resultConverter,
            AdsDeleteRequestValidator requestValidator,
            BannerService bannerService,
            AdsAdGroupTypeValidator adGroupTypeValidator,
            PpcPropertiesSupport ppcPropertiesSupport,
            FeatureService featureService) {
        super(ApiPathConverter.forAds(), auth, ppcPropertiesSupport, featureService);
        this.resultConverter = resultConverter;
        this.requestValidator = requestValidator;
        this.bannerService = bannerService;
        this.adGroupTypeValidator = adGroupTypeValidator;
    }

    @Override
    public ValidationResult<DeleteRequest, DefectType> validateRequest(DeleteRequest request) {
        return requestValidator.validate(request);
    }

    @Override
    public List<Long> convertRequest(DeleteRequest externalRequest) {
        return externalRequest.getSelectionCriteria().getIds();
    }

    @Nonnull
    @Override
    public ValidationResult<List<Long>, DefectType> validateInternalRequest(List<Long> internalRequest) {
        ClientId clientId = auth.getChiefSubclient().getClientId();
        Set<AdGroupType> allowedAdGroupTypes = getAllowedAdGroupTypes();
        return adGroupTypeValidator.validateInternal(clientId, internalRequest, allowedAdGroupTypes);
    }

    @Override
    public ApiMassResult<Long> processList(List<Long> validItems) {
        ClientId clientId = auth.getSubclient().getClientId();
        long operatorUid = auth.getOperator().getUid();
        return resultConverter.toApiMassResult(bannerService.deleteBannersPartial(operatorUid, clientId, validItems));
    }

    @Override
    public DeleteResponse convertResponse(ApiResult<List<ApiResult<Long>>> result) {
        return new DeleteResponse()
                .withDeleteResults(resultConverter.toActionResults(result, apiPathConverter));
    }

    /**
     * Не вычитаем баллы в случае, когда невозможно удалить баннер из-за неверного статуса.
     */
    @Override
    public OperationSummary correctOperationSummary(List<Long> internalRequest,
                                                    ApiResult<List<ApiResult<Long>>> apiResult) {
        return notCountErrorsWithCodes(apiResult, Set.of(BAD_STATUS_CODE));
    }
}
