package ru.yandex.direct.api.v5.entity.ads.delegate;

import java.util.List;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.ads.ResumeRequest;
import com.yandex.direct.api.v5.ads.ResumeResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.common.ApiPathConverter;
import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.OperationOnListDelegate;
import ru.yandex.direct.api.v5.entity.ads.converter.AdsSuspendResumeRequestConverter;
import ru.yandex.direct.api.v5.entity.ads.validation.AdsSuspendResumeRequestValidator;
import ru.yandex.direct.api.v5.result.ApiMassResult;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.api.v5.validation.DefectTypes;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.service.BannerSuspendResumeService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.units.OperationSummary;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.api.v5.entity.OperationOnListUtils.notCountErrorsWithCodes;

@Component
@ParametersAreNonnullByDefault
public class ResumeAdsDelegate
        extends OperationOnListDelegate<ResumeRequest, ResumeResponse, ModelChanges<BannerWithSystemFields>, Long> {
    private static final Logger logger = LoggerFactory.getLogger(ResumeAdsDelegate.class);
    private static final int BAD_STATUS_CODE = DefectTypes.badStatus().getCode();

    private final BannerSuspendResumeService service;
    private final AdsSuspendResumeRequestValidator requestValidator;
    private final AdsSuspendResumeRequestConverter requestConverter;
    private final ResultConverter resultConverter;

    public ResumeAdsDelegate(BannerSuspendResumeService service,
                             ApiAuthenticationSource authentication,
                             AdsSuspendResumeRequestValidator requestValidator,
                             AdsSuspendResumeRequestConverter requestConverter,
                             ResultConverter resultConverter,
                             PpcPropertiesSupport ppcPropertiesSupport,
                             FeatureService featureService) {
        super(ApiPathConverter.forAds(), authentication, ppcPropertiesSupport, featureService);
        this.service = service;
        this.requestValidator = requestValidator;
        this.requestConverter = requestConverter;
        this.resultConverter = resultConverter;
    }

    @Override
    public ValidationResult<ResumeRequest, DefectType> validateRequest(ResumeRequest externalRequest) {
        logger.debug("validate request: {}", externalRequest);
        return requestValidator.validate(externalRequest);
    }

    @Override
    public List<ModelChanges<BannerWithSystemFields>> convertRequest(ResumeRequest externalRequest) {
        logger.debug("convert request: {}", externalRequest);
        return requestConverter.convertResume(externalRequest);
    }

    @Nonnull
    @Override
    public ValidationResult<List<ModelChanges<BannerWithSystemFields>>, DefectType> validateInternalRequest(
            List<ModelChanges<BannerWithSystemFields>> internalRequest) {
        ClientId clientId = auth.getChiefSubclient().getClientId();
        Set<AdGroupType> allowedAdGroupTypes = getAllowedAdGroupTypes();
        return requestValidator.validateInternalRequest(clientId, internalRequest, allowedAdGroupTypes);
    }

    @Override
    public ApiMassResult<Long> processList(List<ModelChanges<BannerWithSystemFields>> validItems) {
        MassResult<Long> result = service.suspendResumeBanners(getClientId(), getOperatorUid(), validItems,
                /* resume = */ true);
        return resultConverter.toApiMassResult(result);
    }

    @Override
    public ResumeResponse convertResponse(ApiResult<List<ApiResult<Long>>> result) {
        logger.debug("convert result: {}", result);
        return new ResumeResponse().withResumeResults(resultConverter.toActionResults(result, apiPathConverter));
    }

    /**
     * Не вычитаем баллы в случае, когда невозможно возобновить показы баннера из-за неверного статуса.
     */
    @Override
    public OperationSummary correctOperationSummary(List<ModelChanges<BannerWithSystemFields>> internalRequest,
                                                    ApiResult<List<ApiResult<Long>>> apiResult) {
        return notCountErrorsWithCodes(apiResult, Set.of(BAD_STATUS_CODE));
    }

    private ClientId getClientId() {
        return auth.getChiefSubclient().getClientId();
    }

    private Long getOperatorUid() {
        return auth.getOperator().getUid();
    }
}
