package ru.yandex.direct.api.v5.entity.ads.delegate;

import java.util.List;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import com.yandex.direct.api.v5.ads.UpdateRequest;
import com.yandex.direct.api.v5.ads.UpdateResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.common.ApiPathConverter;
import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.OperationOnListDelegate;
import ru.yandex.direct.api.v5.entity.ads.AdsUpdateRequestItem;
import ru.yandex.direct.api.v5.entity.ads.converter.AdsUpdateRequestConverter;
import ru.yandex.direct.api.v5.entity.ads.validation.AdsUpdateRequestValidator;
import ru.yandex.direct.api.v5.result.ApiMassResult;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.service.BannersUpdateOperationFactory;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.rbac.RbacRole.PLACER;
import static ru.yandex.direct.rbac.RbacRole.SUPER;
import static ru.yandex.direct.rbac.RbacRole.SUPPORT;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class UpdateAdsDelegate
        extends OperationOnListDelegate<UpdateRequest, UpdateResponse,
        AdsUpdateRequestItem<BannerWithSystemFields>, Long> {

    private static final Logger logger = LoggerFactory.getLogger(UpdateAdsDelegate.class);

    private static final Set<RbacRole> ROLES_ALLOWED_TO_CHANGE_BANNER_FLAGS = ImmutableSet.of(SUPER, PLACER, SUPPORT);

    private final BannersUpdateOperationFactory bannersUpdateOperationFactory;
    private final AdsUpdateRequestValidator requestValidator;
    private final AdsUpdateRequestConverter requestConverter;
    private final ResultConverter resultConverter;
    private final FeatureService featureService;

    public UpdateAdsDelegate(
            ApiAuthenticationSource auth,
            AdsUpdateRequestValidator requestValidator,
            AdsUpdateRequestConverter requestConverter,
            ResultConverter resultConverter,
            PpcPropertiesSupport ppcPropertiesSupport,
            FeatureService featureService,
            BannersUpdateOperationFactory bannersUpdateOperationFactory) {
        super(ApiPathConverter.forAds(), auth, ppcPropertiesSupport, featureService);
        this.requestValidator = requestValidator;
        this.requestConverter = requestConverter;
        this.resultConverter = resultConverter;
        this.bannersUpdateOperationFactory = bannersUpdateOperationFactory;
        this.featureService = featureService;
    }

    @Override
    public ValidationResult<UpdateRequest, DefectType> validateRequest(UpdateRequest externalRequest) {
        logger.debug("validate request: {}", externalRequest);
        ClientId clientId = auth.getChiefSubclient().getClientId();
        boolean smartNoCreatives = featureService.isEnabledForClientId(clientId, FeatureName.SMART_NO_CREATIVES);
        return requestValidator.validate(externalRequest, smartNoCreatives);
    }

    @Override
    public List<AdsUpdateRequestItem<BannerWithSystemFields>> convertRequest(UpdateRequest externalRequest) {
        logger.debug("convert request: {}", externalRequest);
        return requestConverter.convert(externalRequest, ROLES_ALLOWED_TO_CHANGE_BANNER_FLAGS.contains(auth.getOperator().getRole()));
    }

    @Override
    @Nonnull
    public ValidationResult<List<AdsUpdateRequestItem<BannerWithSystemFields>>, DefectType> validateInternalRequest(
            List<AdsUpdateRequestItem<BannerWithSystemFields>> request) {
        Set<AdGroupType> allowedAdGroupTypes = getAllowedAdGroupTypes();
        String allowedAdTypes = getAllowedAdTypes();
        return requestValidator.validateInternalRequest(request, allowedAdGroupTypes, allowedAdTypes);
    }

    @Override
    public ApiMassResult<Long> processList(List<AdsUpdateRequestItem<BannerWithSystemFields>> validItems) {
        logger.debug("process request: {}", validItems);

        MassResult<Long> result = bannersUpdateOperationFactory.createUpdateOperation(
                mapList(validItems, AdsUpdateRequestItem::getInternalItem), getOperatorUid(), getClientId(), true)
                .prepareAndApply();

        return resultConverter.toApiMassResult(result);
    }

    @Override
    public UpdateResponse convertResponse(ApiResult<List<ApiResult<Long>>> result) {
        logger.debug("convert result: {}", result);
        return new UpdateResponse().withUpdateResults(resultConverter.toActionResults(result, apiPathConverter));
    }

    private ClientId getClientId() {
        return auth.getChiefSubclient().getClientId();
    }

    private Long getOperatorUid() {
        return auth.getOperator().getUid();
    }
}
