package ru.yandex.direct.api.v5.entity.ads.validation;

import java.util.Map;
import java.util.Set;

import ru.yandex.direct.api.v5.entity.ads.AdsUpdateRequestItem;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.banner.model.BannerFlags;
import ru.yandex.direct.core.entity.banner.model.BannerWithAdGroupId;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.Constraint;

import static ru.yandex.direct.api.v5.entity.ads.AdsDefectTypes.clientCannotSetBannerFlags;
import static ru.yandex.direct.api.v5.entity.ads.AdsDefectTypes.notSupportedAdGroupType;

/**
 * Констреинты для методов сервиса Ads, проверяющие незапрещенность типов групп, которым принадлежат баннеры.
 */
public class AdsAdGroupTypeConstraints {
    /**
     * Констреинт для метода add.
     *
     * @param adGroupTypes        мап id группы -> тип группы
     * @param allowedAdGroupTypes
     * @return констреинт на тип группы
     */
    public static Constraint<BannerWithAdGroupId, DefectType> allowedAdGroupTypeForAdd(
            Map<Long, AdGroupType> adGroupTypes,
            Set<AdGroupType> allowedAdGroupTypes) {
        return Constraint.fromPredicate(item -> adGroupTypes.get(item.getAdGroupId()) == null
                        || allowedAdGroupTypes.contains(adGroupTypes.get(item.getAdGroupId())),
                notSupportedAdGroupType());
    }

    /**
     * Констреинт для методов delete, moderate.
     *
     * @param adGroupTypesByBannersIds мап id баннера -> тип группы этого баннера
     * @param allowedAdGroupTypes
     * @return констреинт на тип группы
     */
    public static Constraint<Long, DefectType> allowedAdGroupTypeForDeleteAndModerate(
            Map<Long, AdGroupType> adGroupTypesByBannersIds,
            Set<AdGroupType> allowedAdGroupTypes) {
        return Constraint.fromPredicate(id -> adGroupTypesByBannersIds.get(id) == null
                        || allowedAdGroupTypes.contains(adGroupTypesByBannersIds.get(id)),
                notSupportedAdGroupType());
    }

    /**
     * Констреинт для методов archive, unarchive, suspend, resume.
     *
     * @param adGroupTypesByBannersIds мап id баннера -> тип группы этого баннера
     * @param allowedAdGroupTypes
     * @return констреинт на тип группы
     */
    public static Constraint<ModelChanges<BannerWithSystemFields>, DefectType> allowedAdGroupType(
            Map<Long, AdGroupType> adGroupTypesByBannersIds,
            Set<AdGroupType> allowedAdGroupTypes) {
        return Constraint.fromPredicate(mc -> adGroupTypesByBannersIds.get(mc.getId()) == null
                        || allowedAdGroupTypes.contains(adGroupTypesByBannersIds.get(mc.getId())),
                notSupportedAdGroupType());
    }

    /**
     * Констреинт для метода update.
     *
     * @param adGroupTypesByBannersIds мап id баннера -> тип группы этого баннера
     * @param allowedAdGroupTypes
     * @return констреинт на тип группы
     */
    public static Constraint<AdsUpdateRequestItem<BannerWithSystemFields>, DefectType> allowedAdGroupTypeForUpdate(
            Map<Long, AdGroupType> adGroupTypesByBannersIds,
            Set<AdGroupType> allowedAdGroupTypes) {
        return Constraint.fromPredicate(ri -> adGroupTypesByBannersIds.get(ri.getBannerId()) == null
                        || allowedAdGroupTypes.contains(adGroupTypesByBannersIds.get(ri.getBannerId())),
                notSupportedAdGroupType());
    }

    /**
     * Констреинт для метода update.
     * Проверяет, что происходит корректное изменение возрастной метки у TextAd
     */
    public static Constraint<AdsUpdateRequestItem<BannerWithSystemFields>, DefectType> allowedChangeFlagsForUpdateTextAd() {
        return ri -> {

            var ageLabel = ri.getExternalItem().getTextAd().getAgeLabel();
            var flagsChange = ri.getInternalItem().isPropChanged(BannerWithSystemFields.FLAGS)
                    ? ri.getInternalItem().getChangedProp(BannerWithSystemFields.FLAGS)
                    : null;

            if (ageLabel != null &&
                    (flagsChange == null ||
                            flagsChange.get(BannerFlags.BABY_FOOD) == null &&
                            flagsChange.get(BannerFlags.AGE) == null)
            ) {
                return clientCannotSetBannerFlags();
            }
            return null;
        };
    }
}
