package ru.yandex.direct.api.v5.entity.ads.validation;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.api.v5.entity.ads.validation.AdsAdGroupTypeConstraints.allowedAdGroupType;
import static ru.yandex.direct.api.v5.entity.ads.validation.AdsAdGroupTypeConstraints.allowedAdGroupTypeForDeleteAndModerate;

/**
 * Валидатор типа групп для методов сервиса Ads. В случае, если валидируемый баннер находится в группе запрещенного
 * типа, то валидация вернет для этого баннера соответствующий дефект.
 */
@Component
@ParametersAreNonnullByDefault
public class AdsAdGroupTypeValidator {

    private final AdGroupService adGroupService;

    @Autowired
    public AdsAdGroupTypeValidator(AdGroupService adGroupService) {
        this.adGroupService = adGroupService;
    }

    public ValidationResult<List<Long>, DefectType> validateInternal(ClientId clientId, List<Long> internalRequest,
                                                                     Set<AdGroupType> allowedAdGroupTypes) {
        Set<Long> bannerIds = StreamEx.of(internalRequest)
                .filter(Objects::nonNull)
                .toSet();
        Map<Long, AdGroupType> adGroupIdsByBannerIds = adGroupService.getAdGroupTypesByBannerIds(clientId,
                bannerIds);
        return ListValidationBuilder.<Long, DefectType>of(internalRequest)
                .checkEach(allowedAdGroupTypeForDeleteAndModerate(adGroupIdsByBannerIds, allowedAdGroupTypes))
                .getResult();
    }

    public ValidationResult<List<ModelChanges<BannerWithSystemFields>>, DefectType> validateInternalRequest(
            ClientId clientId, List<ModelChanges<BannerWithSystemFields>> internalRequest,
            Set<AdGroupType> allowedAdGroupTypes) {
        Set<Long> bannerIds = StreamEx.of(internalRequest)
                .map(ModelChanges::getId)
                .filter(Objects::nonNull)
                .toSet();
        Map<Long, AdGroupType> adGroupIdsByBannerIds = adGroupService.getAdGroupTypesByBannerIds(clientId, bannerIds);

        return ListValidationBuilder.<ModelChanges<BannerWithSystemFields>, DefectType>of(internalRequest)
                .checkEach(allowedAdGroupType(adGroupIdsByBannerIds, allowedAdGroupTypes))
                .getResult();
    }
}
