package ru.yandex.direct.api.v5.entity.ads.validation;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.ads.AdAddItem;
import com.yandex.direct.api.v5.ads.AddRequest;
import com.yandex.direct.api.v5.ads.MobileAppAdAdd;
import com.yandex.direct.api.v5.ads.SmartAdBuilderAdAdd;
import com.yandex.direct.api.v5.ads.TextAdAdd;
import com.yandex.direct.api.v5.ads.VideoExtensionAddItem;
import com.yandex.direct.api.v5.general.YesNoEnum;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.banner.model.BannerWithAdGroupId;
import ru.yandex.direct.core.entity.banner.model.CpcVideoBanner;
import ru.yandex.direct.core.entity.banner.model.ImageBanner;
import ru.yandex.direct.core.entity.banner.model.PerformanceBannerMain;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;

import static org.apache.commons.lang3.StringUtils.capitalize;
import static ru.yandex.direct.api.v5.entity.ads.AdsDefectTypes.bannersWithCreativeDeprecated;
import static ru.yandex.direct.api.v5.entity.ads.AdsDefectTypes.cannotAddBannerToNoCreativeAdGroup;
import static ru.yandex.direct.api.v5.entity.ads.AdsDefectTypes.inconsistentBannerTypeAndAdgroupType;
import static ru.yandex.direct.api.v5.entity.ads.AdsDefectTypes.logoIsOnlyForBannersWithoutCreative;
import static ru.yandex.direct.api.v5.entity.ads.AdsDefectTypes.maxBannersPerAddRequest;
import static ru.yandex.direct.api.v5.entity.ads.Constants.MAX_ELEMENTS_PER_ADD;
import static ru.yandex.direct.api.v5.entity.ads.validation.AdsAdGroupTypeConstraints.allowedAdGroupTypeForAdd;
import static ru.yandex.direct.api.v5.validation.DefectTypes.invalidUseOfField;
import static ru.yandex.direct.api.v5.validation.DefectTypes.invalidValue;
import static ru.yandex.direct.api.v5.validation.DefectTypes.possibleOnlyOneField;
import static ru.yandex.direct.api.v5.validation.DefectTypes.requiredAtLeastOneOfFields;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.isNull;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.maxListSize;
import static ru.yandex.direct.core.entity.banner.repository.filter.BannerFilterFactory.bannerAdGroupIdFilter;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

@Component
@ParametersAreNonnullByDefault
public class AdsAddRequestValidator {

    private final AdGroupService adGroupService;
    private final BannerTypedRepository bannerTypedRepository;
    private final ShardHelper shardHelper;

    private static final Set<Class> AD_TYPES_FOR_TEXT_AD_GROUP =
            Set.of(TextBanner.class, CpcVideoBanner.class, ImageBanner.class);

    @Autowired
    public AdsAddRequestValidator(AdGroupService adGroupService,
                                  BannerTypedRepository bannerTypedRepository,
                                  ShardHelper shardHelper) {
        this.adGroupService = adGroupService;
        this.bannerTypedRepository = bannerTypedRepository;
        this.shardHelper = shardHelper;
    }

    public ValidationResult<AddRequest, DefectType> validate(AddRequest externalRequest,
                                                             boolean isServicesApplication,
                                                             boolean isDisplayUrlTextAllowed,
                                                             boolean isLeadformAttributesAllowed,
                                                             boolean smartNoCreatives) {
        ItemValidationBuilder<AddRequest, DefectType> vb = ItemValidationBuilder.of(externalRequest);
        vb.item(externalRequest.getAds(), AddRequest.PropInfo.ADS.schemaName.getLocalPart())
                .check(maxListSize(MAX_ELEMENTS_PER_ADD), maxBannersPerAddRequest(MAX_ELEMENTS_PER_ADD))
                .checkBy(items -> {
                    ListValidationBuilder<AdAddItem, DefectType> lvb = ListValidationBuilder.of(items);
                    lvb.checkEach(fromPredicate(
                            item -> item.getContentPromotionServiceAd() == null || isServicesApplication,
                            invalidUseOfField()
                    ));
                    lvb.checkEachBy(item -> {
                        ItemValidationBuilder<AdAddItem, DefectType> ivb = ItemValidationBuilder.of(item);
                        ivb.item(item.getTextAd(), AdAddItem.PropInfo.TEXT_AD.schemaName.getLocalPart())
                                .checkBy(validateDisplayUrlTextAllowed(isDisplayUrlTextAllowed), When.notNull())
                                .checkBy(validateLeadformAttributesAllowed(isLeadformAttributesAllowed), When.notNull());
                        ivb.item(item.getSmartAdBuilderAd(), AdAddItem.PropInfo.SMART_AD_BUILDER_AD.schemaName.getLocalPart())
                                .checkBy(validateSmart(smartNoCreatives), When.notNull());
                        ivb.item(item.getMobileAppAd(), AdAddItem.PropInfo.MOBILE_APP_AD.schemaName.getLocalPart())
                                .checkBy(validateMobileAppShowTitleAndBody(), When.notNull());
                        return ivb.getResult();
                    });
                    return lvb.getResult();
                });
        return vb.getResult();
    }

    Validator<MobileAppAdAdd, DefectType> validateMobileAppShowTitleAndBody() {
        return item -> {
            ItemValidationBuilder<MobileAppAdAdd, DefectType> ivb = ItemValidationBuilder.of(item);
            ivb.item(item.getVideoExtension(), MobileAppAdAdd.PropInfo.VIDEO_EXTENSION.schemaName.getLocalPart())
                    .checkBy(extension -> {
                        ItemValidationBuilder<VideoExtensionAddItem, DefectType> ivbLocal = ItemValidationBuilder.of(extension);
                        ivbLocal
                                .item(extension.getShowTitleAndBody(),
                                        VideoExtensionAddItem.PropInfo.SHOW_TITLE_AND_BODY.schemaName.getLocalPart())
                                .check(fromPredicate(flag -> flag == YesNoEnum.NO, invalidValue()));
                        return ivbLocal.getResult();
                    }, When.notNull());
            return ivb.getResult();
        };
    }

    /**
     * Валидация проверяет, что тексты отображаемого гринурла устанавливаются только если у пользователя есть права
     *
     * @param isDisplayUrlTextAllowed разрешено ли устанавливать кастомные тексты отображаемого гринурла
     * @see <a href="https://st.yandex-team.ru/DIRECT-142656">DIRECT-142656</a>
     */
    private Validator<TextAdAdd, DefectType> validateDisplayUrlTextAllowed(boolean isDisplayUrlTextAllowed) {
        return item -> {
            ItemValidationBuilder<TextAdAdd, DefectType> ivb = ItemValidationBuilder.of(item);
            ivb.item(item.getDutPrefix(), TextAdAdd.PropInfo.DUT_PREFIX.schemaName.getLocalPart())
                    .check(fromPredicate(prefix -> isDisplayUrlTextAllowed, invalidUseOfField()));
            ivb.item(item.getDutPrefix(), TextAdAdd.PropInfo.DUT_SUFFIX.schemaName.getLocalPart())
                    .check(fromPredicate(suffix -> isDisplayUrlTextAllowed, invalidUseOfField()));
            return ivb.getResult();
        };
    }

    /**
     * Валидация проверяет, что атрибуты лидформ баннера устанавливаются, только если у пользователя есть права.
     *
     * @param isLeadformAttributesAllowed разрешено ли устанавливать атрибуты лидформ для баннера
     * @see <a href="https://st.yandex-team.ru/DIRECT-147478">DIRECT-147478</>
     */
    private Validator<TextAdAdd, DefectType> validateLeadformAttributesAllowed(boolean isLeadformAttributesAllowed) {
        return item -> {
            ItemValidationBuilder<TextAdAdd, DefectType> ivb = ItemValidationBuilder.of(item);
            ivb.item(item.getLfHref(), TextAdAdd.PropInfo.LF_HREF.schemaName.getLocalPart())
                    .check(fromPredicate(href -> isLeadformAttributesAllowed, invalidUseOfField()));
            ivb.item(item.getLfButtonText(), TextAdAdd.PropInfo.LF_BUTTON_TEXT.schemaName.getLocalPart())
                    .check(fromPredicate(text -> isLeadformAttributesAllowed, invalidUseOfField()));
            return ivb.getResult();
        };
    }

    private Validator<SmartAdBuilderAdAdd, DefectType> validateSmart(boolean smartNoCreatives) {
        return item -> {
            ItemValidationBuilder<SmartAdBuilderAdAdd, DefectType> ivb = ItemValidationBuilder.of(item);
            ivb.item(item.getCreative(), SmartAdBuilderAdAdd.PropInfo.CREATIVE.schemaName.getLocalPart())
                    .weakCheck(isNull(), bannersWithCreativeDeprecated(), When.isTrue(smartNoCreatives));
            ivb.item(item.getLogoExtensionHash(), SmartAdBuilderAdAdd.PropInfo.LOGO_EXTENSION_HASH.schemaName.getLocalPart())
                    .check(isNull(), logoIsOnlyForBannersWithoutCreative(), When.isTrue(item.getCreative() != null));
            return ivb.getResult();
        };
    }

    /**
     * Валидация опирается на специальные значения из {@link AdsApiValidationSignals},
     * проставляемые конвертером, таким образом он передаёт информацию об ошибках в исходном запросе.
     *
     * @see ru.yandex.direct.api.v5.entity.ads.converter.AdsAddRequestConverter
     */
    public ValidationResult<List<BannerWithAdGroupId>, DefectType> validateInternalRequest(
            ClientId clientId,
            List<BannerWithAdGroupId> internalRequest,
            Set<AdGroupType> allowedAdGroupTypes,
            String allowedAdTypes) {
        Set<Long> adGroupIds = StreamEx.of(internalRequest)
                .map(BannerWithAdGroupId::getAdGroupId)
                .filter(Objects::nonNull)
                .toSet();
        Map<Long, AdGroupType> adGroupTypes = adGroupService.getAdGroupTypes(clientId, adGroupIds);

        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        List<PerformanceBannerMain> banners = bannerTypedRepository.getSafely(shard, bannerAdGroupIdFilter(adGroupIds),
                PerformanceBannerMain.class);
        Set<Long> noCreativeAdGroupIds = listToSet(banners, PerformanceBannerMain::getAdGroupId);

        return ListValidationBuilder.<BannerWithAdGroupId, DefectType>of(internalRequest)
                .checkEach(atLeastOneTypeWasSpecified(allowedAdTypes))
                .checkEach(atMostOneTypeWasSpecified(allowedAdTypes), When.isValid())
                .checkEach(allowedAdTypeForTextAdGroup(),
                        When.valueIs(b -> adGroupTypes.get(b.getAdGroupId()) == AdGroupType.BASE))
                .checkEach(creativeIdIsProperlySetInTextBanner(), When.isValid())
                .checkEach(allowedAdGroupTypeForAdd(adGroupTypes, allowedAdGroupTypes), When.isValid())
                .checkEach(allowedAdGroupIdForAdd(noCreativeAdGroupIds))
                .getResult();
    }

    private static Constraint<BannerWithAdGroupId, DefectType> allowedAdTypeForTextAdGroup() {
        return fromPredicate(banner -> AD_TYPES_FOR_TEXT_AD_GROUP.contains(banner.getClass()),
                inconsistentBannerTypeAndAdgroupType());
    }

    private static Constraint<BannerWithAdGroupId, DefectType> atLeastOneTypeWasSpecified(String allowedAdTypes) {
        return fromPredicate(AdsApiValidationSignals::hasBannerTypeSpecified,
                requiredAtLeastOneOfFields(allowedAdTypes));
    }

    private static Constraint<BannerWithAdGroupId, DefectType> atMostOneTypeWasSpecified(String allowedAdTypes) {
        return fromPredicate(AdsApiValidationSignals::hasUnambiguousType,
                possibleOnlyOneField(allowedAdTypes));
    }

    private static Constraint<BannerWithAdGroupId, DefectType> allowedAdGroupIdForAdd(Set<Long> noCreativeAdGroupIds) {
        return fromPredicate(banner -> !noCreativeAdGroupIds.contains(banner.getAdGroupId()),
                cannotAddBannerToNoCreativeAdGroup());
    }

    private static Constraint<BannerWithAdGroupId, DefectType> creativeIdIsProperlySetInTextBanner() {
        return fromPredicate(AdsApiValidationSignals::hasVideoExtensionIdSpecified,
                requiredAtLeastOneOfFields(
                        capitalize(VideoExtensionAddItem.PropInfo.CREATIVE_ID.propertyName)));
    }
}
